#!/usr/bin/python

import xmlrpclib
import hashlib
import hmac
import time
import random
import httplib
import socket
import ssl

class ProxiedConnection:
    proxy = None

    def get_proxy_hostport(self, proxy_host, proxy_port=None):
        if ':' in proxy_host:
            proxy_host, proxy_port = proxy_host.split(':')
            proxy_port = int(proxy_port)
        elif proxy_port is None:
            proxy_host = proxy_host
            proxy_port = self.default_port
        return (proxy_host, proxy_port)

    def connect(self):
        if not self.proxy:
            parent_class = self.__class__.__bases__[-1]
            parent_class.connect(self)

        try:
            proxy_host, proxy_port = self.proxy.split(':', 2)
        except ValueError:
            proxy_host, proxy_port = self.proxy, 80
        self.sock = socket.create_connection((proxy_host, proxy_port),
                                              self.timeout)
        if self.mode == 'https':
            self.send('CONNECT %s:%d HTTP/1.0\r\n' % (self.host, self.port))
            self.send('\r\n')
            response = self.response_class(self.sock, strict = self.strict,
                                           method = self._method)
            (version, code, message) = response._read_status()
            if code != 200:
                self.close()
                raise socket.error("Tunnel connection failed: %d %s" % (code,
                                                                    message.strip()))
            # eat up headers from proxy
            while True:
                #should not use directly fp probably
                line = response.fp.readline()
                if line == '\r\n': break
            self.sock = ssl.wrap_socket(self.sock, self.key_file, self.cert_file)

class ProxiedHTTPConnection(ProxiedConnection, httplib.HTTPConnection):
    mode = 'http'

class ProxiedHTTPSConnection(ProxiedConnection, httplib.HTTPSConnection):
    mode = 'https'

class ProxiedHTTP(httplib.HTTP):
    _connection_class = ProxiedHTTPConnection

class ProxiedHTTPS(httplib.HTTPS):
    _connection_class = ProxiedHTTPSConnection

class ProxiedTransport(xmlrpclib.Transport):
    proto = 'http'
    proxy = None

    def set_proto(self, proto):
        assert proto in ('http', 'https')
        self.proto = proto

    def set_proxy(self, proxy):
        self.proxy = proxy

    def make_connection(self, host):
        self.realhost = host
        host, extra_headers, x509 = self.get_host_info(host)
        if self.proto == 'http':
            p = ProxiedHTTP(host)
        elif self.proto == 'https':
            p = ProxiedHTTPS(host)
        p._conn.proxy = self.proxy
        return p

    def send_request(self, connection, handler, request_body):
        connection.putrequest("POST", "%s://%s%s" % (self.proto, self.realhost, handler))

    def send_host(self, connection, host):
        connection.putheader('Host', self.realhost)

class drupal_service_object(object):
    def __init__(self, drupal_connection, name):
        self._dc = drupal_connection
        self._name = name

    def __getattr__(self, key):
        self._name += '.%s' % (key)
        return self

    def __call__(self, *args):
        call_params = self._dc._params(self._name, *args)
        return getattr(self._dc._server, self._name)(*call_params)

class drupal_connection(object):
    def __init__(self, hostname, domain, api_key, path='/services/xmlrpc', proto='http', pt=False, p_hostname=None, p_port=None):
        self._domain = domain
        self._api_key = api_key
        self._hostname = hostname
        self._path = path
        self._proto = proto
        self._server_url = '%s://%s%s' % (self._proto, self._hostname, self._path)

        if pt:
            print 1
            pt = ProxiedTransport()
            pt.set_proto(self._proto.lower())
            pt.set_proxy('%s:%s' % (p_hostname, p_port))
        else:
            pt = None

        self._server = xmlrpclib.Server(self._server_url, allow_none = 1,
                            transport=pt
                            )

        self._random = random.Random()
        self._sessid = None

    def __getitem__(self, name):
        return super(drupal_connection, self).__getitem__(name)

    def __getstate__(self):
        pass

    def __getattr__(self, key):
        return drupal_service_object(self, key)

    def _get_sessionid(self):
        if self._sessid is None:
            anon_conn = self._server.system.connect(self._api_key)
            self._sessid = anon_conn['sessid']
        return self._sessid

    def _get_nonce(self):
        return str(self._random.random())

    def _params(self, method, *varargs):
        tstamp = str(time.time())
        nonce = self._get_nonce()
        hash_data = '%s;%s;%s;%s' % (tstamp, self._domain, nonce, method)
        hash = hmac.new(self._api_key, hash_data, hashlib.sha256).hexdigest()
        base_args = [ hash, self._domain, tstamp, nonce, self._get_sessionid() ]
        return base_args + list(varargs)

    def connect(self, username, password):
        user_login_args = self._params('user.login', username, password)
        user = self._server.user.login(*user_login_args)
        if user:
            # replace anonymous session id with the authenticated one
            self._sessid = user['sessid']
            return True
        return False

# usage:

#dpc = drupal_connection(hostname='drupal_website_url',
#                        domain='xmlrpc.domain',
#                        api_key='drupal_service_api_key',
#                        path='/services/xmlrpc')
#dpc.connect('openerp-user', 'password')

#node = dpc.node.get(9)
