# -*- encoding: utf-8 -*-
############################################################################################
#
#    OpenERP, Open Source Management Solution
#    Copyright (C) 2011 Zikzakmedia S.L. (<http://www.zikzakmedia.com>). All Rights Reserved
#    $Id$
#
#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU Affero General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU Affero General Public License for more details.
#
#    You should have received a copy of the GNU Affero General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
############################################################################################

from osv import osv, fields
from tools.translate import _
import netsvc

import decimal_precision as dp
import time

class training_offer_invoice(osv.osv):
    _name = "training.offer.invoice"
    _description = "Training Offer Invoice"

    def _amount_total(self, cr, uid, ids, name, args, context=None):
        if not ids:
            return {}
        res = {}
        invoice_line_obj = self.pool.get('training.offer.invoice.line')
        for invoice in self.browse(cr, uid, ids, context=context):
            amount_total = 0.0
            invoice_line_ids = invoice_line_obj.search(cr, uid, [('invoice_id','=',invoice.id)])
            if len(invoice_line_ids) > 0:
                for invoice_line_id in invoice_line_ids:
                    pay = invoice_line_obj.browse(cr, uid, invoice_line_id)
                    amount_total += pay.price
                res[invoice.id] = amount_total
        return res

    _columns = {
        'name': fields.char('Name', size=32, required=True, translate=True),
        'active': fields.boolean('Active'),
        'invoice_line_ids': fields.one2many('training.offer.invoice.line', 'invoice_id', string='Invoice'),
        'amount_total': fields.function(_amount_total, string="Total Amount", method=True, type='float', digits_compute=dp.get_precision('Account')),
        'offer_id': fields.many2one('training.offer','Offer'),
    }

    _defaults = {
        'active': lambda *a: 1,
    }

training_offer_invoice()

class training_offer_invoice_line(osv.osv):
    _name = "training.offer.invoice.line"
    _description = "Training Offer Invoice Line"
    _rec_name = 'reference_date'

    _columns = {
        'reference_date': fields.selection([('session_date','Session Date'),('subscription_date','Subscription Date')],'Reference Date'),
        'days': fields.integer('Days'),
        'price': fields.float('Price', required=True, digits_compute= dp.get_precision('Account')),
        'sequence': fields.integer('Sequence', help="Gives the sequence order when create Account Invoice."),
        'invoice_id': fields.many2one('training.offer.invoice','Payment'),
        'company_id': fields.many2one('res.company', 'Company', select=1, required=True),
        'payment_type': fields.many2one('payment.type','Payment Type'),
    }

    _order = 'sequence, id desc'

    _defaults = {
        'sequence': 10,
        'reference_date': 'session_date',
    }

training_offer_invoice_line()

class training_offer_invoicing(osv.osv):
    _name = "training.offer.invoicing"
    _description = "Training Offer Invoicing History"
    _rec_name = 'subscription_line_id'

    _columns = {
        'date': fields.date('Date', required=True),
        'price': fields.float('Price', required=True, digits_compute= dp.get_precision('Account')),
        'state': fields.selection([
            ('wait','Waiting'),
            ('inv_done','Invoiced'),
            ('inv_refund','Invoice refunded'),
            ('error','Error'),
            ],'State', readonly=True),
        'company_id': fields.many2one('res.company', 'Company', select=1, required=True),
        'invoice_id': fields.many2one('account.invoice','Invoice', readonly=True),
        'subscription_line_id': fields.many2one('training.subscription.line','Subscription Line', readonly=True),
        'invoice_type': fields.many2one('training.offer.invoice','Offer Invoicing', readonly=True),
        'payment_type': fields.many2one('payment.type','Payment Type'),
        'seq': fields.integer('Sequence', readonly=True),
        'partner': fields.related('subscription_line_id', 'partner_id', type='many2one', relation='res.partner', string='Partner'),
    }

    _defaults = {
        'state': 'wait',
    }

    # Workflow stuff
    #################

    def create_invoice_offer(self, cr, uid, ids):
        """
        Create a offer of invoice from Training Offer Invoicing
        """
        self.write(cr, uid, ids, {'state': 'wait'})
        return True

    def invoice_done(self, cr, uid, ids):
        offer_invoicing_obj = self.pool.get('training.offer.invoicing')
        res_partner_obj = self.pool.get('res.partner')
        account_payment_term_obj = self.pool.get('account.payment.term')
        analytic_account_obj = self.pool.get('account.analytic.account')
        account_payment_term_obj = self.pool.get('account.payment.term')
        invoice_obj = self.pool.get('account.invoice')

        for invoicing in self.browse(cr, uid, ids):

            invoices = []
            if invoicing.invoice_id:
                last_invoice = invoicing.invoice_id.id
                invoices.append(last_invoice)
            else:
                partner_id = invoicing.subscription_line_id.subscription_id.partner_id.id

                if invoicing.company_id.id:
                    company_id = invoicing.company_id.id
                else:
                    raise osv.except_osv(_('Error'),_("Company are not avaible"))

                journal = self.pool.get('account.journal').search(cr, uid, [('type','=','sale'),('company_id','=',company_id)])

                partner = res_partner_obj.browse(cr, uid, partner_id)

                if not partner.property_account_receivable:
                    raise osv.except_osv(_('Error'),_("Account Receivable Partner not avaible. Edit Partner and select an account receivable"))

                #TODO: Search account ID same code property partner. Redesign
                account_ids = self.pool.get('account.account').search(cr, uid, [('code','=',partner.property_account_receivable.code),('company_id','=',company_id)])
                if len(account_ids) == 0:
                    raise osv.except_osv(_('Error'),_("Account Receivable in this company is not avaible"))

                curr_invoice = {
                    'name': invoicing.subscription_line_id.name+' '+invoicing.subscription_line_id.session_id.name+' ('+invoicing.subscription_line_id.session_id.date[:10]+')',
                    'partner_id': partner_id,
                    'company_id': company_id,
                    'journal_id': journal[0],
                    'account_id': account_ids[0],
                    'currency_id': invoicing.company_id.currency_id.id,
                    'date_invoice': invoicing.date,
                    'training_offer_invoicing_id': invoicing.id,
                }

                if invoicing.payment_type.id:
                    curr_invoice['payment_type'] = invoicing.payment_type.id

                values = invoice_obj.onchange_partner_id(cr, uid, [], 'out_invoice', invoicing.subscription_line_id.subscription_id.partner_id.id)
                curr_invoice.update(values['value'])

                if len(account_ids) >0:
                    curr_invoice['account_id'] = account_ids[0]

                if invoicing.subscription_line_id.subscription_id.payment_term_id:
                    curr_invoice['payment_term'] = invoicing.subscription_line_id.subscription_id.payment_term_id.id

                if invoicing.subscription_line_id.subscription_id.partner_id.property_payment_term:
                    pterm_list= account_payment_term_obj.compute(cr, uid,
                        invoicing.subscription_line_id.subscription_id.partner_id.property_payment_term.id, value=1,
                        date_ref=time.strftime('%Y-%m-%d'))
                    if pterm_list:
                        pterm_list = [line[0] for line in pterm_list]
                        pterm_list.sort()
                        curr_invoice['date_due'] = pterm_list[-1]
                last_invoice = invoice_obj.create(cr, uid, curr_invoice)
                invoices.append(last_invoice)

                product = invoicing.subscription_line_id.session_id.offer_id.product_id

                taxes = product.taxes_id
                tax = self.pool.get('account.fiscal.position').map_tax(cr, uid, partner.property_account_receivable, taxes)
                account_id = product.product_tmpl_id.property_account_income.id or product.categ_id.property_account_income_categ.id

                if not account_id:
                    raise osv.except_osv(_('Error'),_("Not account available for product or category product"))

                #TODO: Search account ID same code property product/category. Redesign
                account = self.pool.get('account.account').browse(cr, uid, account_id)
                account_ids = self.pool.get('account.account').search(cr, uid, [('code','=',account.code),('company_id','=',company_id)])
                if len(account_ids) == 0:
                    raise osv.except_osv(_('Error'),_("Income Account in this company is not avaible"))

                curr_line = {
                    'price_unit': invoicing.price,
                    'quantity': 1,
                    'invoice_line_tax_id': [(6,0,tax)],
                    'invoice_id': last_invoice,
                    'company_id': company_id,
                    'name': product.name,
                    'product_id': product.id,
                    'uos_id': product.uom_id.id,
                    'account_id': account_ids[0],
                    #'account_analytic_id': ,
                }

                self.pool.get('account.invoice.line').create(cr, uid, curr_line)

            # change state invoicing
            self.pool.get('training.offer.invoicing').write(cr, uid, invoicing.id, {'invoice_id':last_invoice,'state':'inv_done'})

            # change state subscription line
            ids = self.pool.get('training.offer.invoicing').search(cr, uid, [('state','=','wait'),('subscription_line_id','=',invoicing.subscription_line_id.id)])
            if len(ids) == 0:
                self.pool.get('training.subscription.line').write(cr, uid, [invoicing.subscription_line_id.id], {'state':'done'})

                # change state subscription
                subscription = self.pool.get('training.subscription').browse(cr, uid, invoicing.subscription_line_id.subscription_id.id) #reload state value
                subscription_lines = subscription.subscription_line_ids
                subscription_done = True

                for subscription_line in subscription_lines:
                    if subscription_line.state != 'done' and subscription_done:
                        subscription_done = False

                if subscription_done:
                    self.pool.get('training.subscription').write(cr, uid, [invoicing.subscription_line_id.subscription_id.id], {'state':'done'})

        mod_obj = self.pool.get('ir.model.data')
        act_obj = self.pool.get('ir.actions.act_window')

        mod_id = mod_obj.search(cr, uid, [('name', '=', 'action_invoice_tree1')])[0]
        res_id = mod_obj.read(cr, uid, mod_id, ['res_id'])['res_id']
        act_win = act_obj.read(cr, uid, res_id, [])
        act_win['domain'] = [('id','in',invoices),('type','=','out_invoice')]
        act_win['name'] = _('Invoices')
        return act_win

    def invoice_refund(self, cr, uid, ids):
        self.write(cr, uid, ids, {'state': 'inv_refund'})
        return True


    def unlink(self, cr, uid, vals, context):
        invoicing_ids = self.read(cr, uid, vals, ['state'])
        unlink_ids = []
        for s in invoicing_ids:
            if s['state'] in ['wait']:
                unlink_ids.append(s['id'])
            else:
                raise osv.except_osv(_('Invalid action !'), _('Only wait invoicing could be deleted !'))
        return osv.osv.unlink(self, cr, uid, unlink_ids, context=context)

training_offer_invoicing()

#========Inherits=========

class training_offer(osv.osv):
    _inherit = 'training.offer'

    _columns = {
        'invoice_ids': fields.one2many('training.offer.invoice', 'offer_id', 'Payments'),
    }

training_offer()

class training_subscription_line(osv.osv):
    _inherit = 'training.subscription.line'

    _columns = {
        'invoicing_ids': fields.one2many('training.offer.invoicing', 'subscription_line_id', 'Invoicing'),
    }

    def action_workflow_cancel(self, cr, uid, ids, context=None):
        if context is None:
            context = {}
        for subscription_line in self.browse(cr, uid, ids, context):
            for invoicing in subscription_line.invoicing_ids:
                if invoicing.state not in ['inv_refund']:
                    raise osv.except_osv(_('Invalid action!'), _("You can't delete a subscription line with proposed invoices in states wait or done."))
        return super(training_subscription_line, self).action_workflow_cancel(cr, uid, ids, context)

training_subscription_line()


class account_invoice(osv.osv):
    _inherit = 'account.invoice'

    def unlink(self, cr, uid, ids, context=None):
        if context is None:
            context = {}
        wf_service = netsvc.LocalService("workflow")
        offer_inv_obj = self.pool.get('training.offer.invoicing')
        for inv in self.browse(cr, uid, ids, context):
            # If we delete an invoice, the training.offer changes to wait
            offer_inv_ids = offer_inv_obj.search(cr, uid, [('invoice_id', '=', inv.id)], context=context)
            for offer_inv in offer_inv_obj.browse(cr, uid, offer_inv_ids, context):
                if offer_inv.state == 'inv_refund':
                    wf_service.trg_validate(uid, 'training.offer.invoicing', offer_inv.id, 'return_inv_done', cr)
                wf_service.trg_validate(uid, 'training.offer.invoicing', offer_inv.id, 'return_wait', cr)
            # If we delete a refund invoice (origin_invoices_ids is not empty), the training offers of the original invoices change to inv_done
            for inv_orig in inv.origin_invoices_ids:
                offer_inv_ids = offer_inv_obj.search(cr, uid, [('invoice_id', '=', inv_orig.id)], context=context)
                for offer_inv_id in offer_inv_ids:
                    wf_service.trg_validate(uid, 'training.offer.invoicing', offer_inv_id, 'return_inv_done', cr)
        return super(account_invoice, self).unlink(cr, uid, ids, context=context)

    def write(self, cr, uid, ids, values, context=None):
        if context is None:
            context = {}
        offer_inv_obj = self.pool.get('training.offer.invoicing')
        wf_service = netsvc.LocalService("workflow")
        if 'origin_invoices_ids' in values:
            new_inv_orig = values['origin_invoices_ids'][0][2]
            for inv in self.browse(cr, uid, ids, context):
                old_inv_orig = [x.id for x in inv.origin_invoices_ids]
                add_inv_orig = [x for x in new_inv_orig if x not in old_inv_orig]
                del_inv_orig = [x for x in old_inv_orig if x not in new_inv_orig]
                for offer_inv_id in offer_inv_obj.search(cr, uid, [('invoice_id', 'in', add_inv_orig)], context=context):
                    wf_service.trg_validate(uid, 'training.offer.invoicing', offer_inv_id, 'to_inv_refund', cr)
                for offer_inv_id in offer_inv_obj.search(cr, uid, [('invoice_id', 'in', del_inv_orig)], context=context):
                    wf_service.trg_validate(uid, 'training.offer.invoicing', offer_inv_id, 'return_inv_done', cr)

        return super(account_invoice, self).write(cr, uid, ids, values, context=context)

account_invoice()
