# -* encoding: utf-8 -*-
############################################################################################
#
#    OpenERP, Open Source Management Solution
#    Copyright (C) 2004-2009 Tiny SPRL (<http://tiny.be>). All Rights Reserved
#    Copyright (C) 2008-2009 AJM Technologies S.A. (<http://www.ajm.lu). All Rights Reserved
#    Copyright (C) 2010-2011 Thamini S.à.R.L (<http://www.thamini.com>). All Rights Reserved
#    $Id$
#
#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
############################################################################################

from osv import osv
from osv import fields
import netsvc
import random
import tools
from tools.translate import _
import time
import os
from training.training import training_course_kind_compute

FMT = '%Y-%m-%d %H:%M:%S'

WRITABLE_ONLY_IN_DRAFT = dict(readonly=True, states={'draft': [('readonly', False)]})

class training_course_type(osv.osv):
    _inherit = 'training.course_type'

    _columns = {
        'exam_product_id' : fields.many2one('product.product', 'Exam'),
    }
training_course_type()

class training_config_contact_function(osv.osv):
    _inherit = 'training.config.contact.function'

    def _get_values_of_kind(self):
        vals = super(training_config_contact_function, self)._get_values_of_kind()
        if not vals:
            vals = []
        vals.append(('exam', 'Exam'))
        return vals

training_config_contact_function()

class training_session(osv.osv):
    _inherit = 'training.session'

    def _create_seance(self, cr, uid, session, context=None):
        group_proxy = self.pool.get('training.group')
        if session.kind == 'exam':
            group_ids = group_proxy.search(cr, uid, [('session_id', '=', session.id)])
            if group_ids:
                group_id = group_ids[0]
            else:
                group_id = group_proxy.create(cr, uid, {'name' : _('Class %d') % (1,), 'session_id': session.id}, context=context)

            seance_proxy = self.pool.get('training.seance')
            seance_id = seance_proxy.create(cr, uid, {
                'name' : _("Exam"),
                'original_session_id' : session.id,
                'min_limit' : 1,
                'max_limit' : 100,
                'user_id' : session.user_id.id,
                'date' : session.date,
                'kind' : 'exam',
                'group_id' : group_id,
                'duration' : 3.0,
            })
            return [seance_id]
        else:
            return super(training_session, self)._create_seance(cr, uid, session, context=context)

training_session()

class training_seance(osv.osv):
    _inherit = 'training.seance'

    _columns = {
        'kind' : fields.selection(training_course_kind_compute, 'Kind', required=True),
    }

    def _create_participation(self, cr, uid, seance, subscription_line, context=None):
        participation = super(training_seance, self)._create_participation(cr, uid, seance, subscription_line, context)

        training_participat_pool = self.pool.get('training.participation')

        if subscription_line.session_id.kind == 'exam':
            # for exam session, we must have a course_id on the subscription line
            if not subscription_line.course_id:
                raise osv.except_osv(_('Warning'),
                                     _('You have selected an exam seance but there is no associated course'))
            vals = {'course_questionnaire_id': subscription_line.course_id.id}
            result = training_participat_pool.on_change_course_questionnaire(cr, uid, [], vals['course_questionnaire_id'], False, context=context)
            vals.update(result.get('value',{}))
            training_participat_pool.write(cr, uid, [participation], vals)

        elif seance.kind == 'exam':
            # subscription is not on a exam session, but this seance is an exam
            # so the exam participation with be taken on course from this seance
            vals = {'course_questionnaire_id': seance.course_id.id}
            result = training_participat_pool.on_change_course_questionnaire(cr, uid, [], vals['course_questionnaire_id'], False, context=context)
            vals.update(result.get('value',{}))
            training_participat_pool.write(cr, uid, [participation], vals)

        return participation

    def action_create_procurements(self, cr, uid, ids, context=None):
        if not ids:
            return False
        wf_service = netsvc.LocalService("workflow")
        # For each seance, we will create a procurement for each product
        for seance in self.browse(cr, uid, ids, context=context):
            if seance.kind == 'exam':
                continue
            else:
                super(training_seance, self).action_create_procurements(cr, uid, [seance.id], context=context)
        return True

    def search(self, cr, uid, domain, offset=0, limit=None, order=None, context=None, count=False):
        request_session_id = context.get('request_session_id', False)
        if request_session_id:
            session = self.pool.get('training.session').browse(cr, uid, request_session_id, context=context)
            if session.kind == 'exam':
                return [seance.id for seance in session.seance_ids]
        return super(training_seance, self).search(cr, uid, domain, offset=offset,
                                                   limit=limit, order=order, context=context, count=count)

    def _get_product(self, cr, uid, ids, context=None):
        assert len(ids) == 1
        seance = self.browse(cr, uid, ids[0], context)
        if seance.kind == 'exam':
            # FIXME: Hard coded !!!
            product_proxy = self.pool.get('product.product')
            product_ids = product_proxy.search(cr, uid, [('default_code', '=', 'R50H')], context=context)
            if not product_ids:
                raise osv.except_osv(_('Error'),
                                     _('There is no product with the code equal to R50H !'))
            return product_proxy.browse(cr, uid, product_ids[0], context=context)
        else:
            return super(training_seance, self)._get_product(cr, uid, ids, context=context)

training_seance()

class exam_questionnaire(osv.osv):
    _name = 'training.exam.questionnaire'

exam_questionnaire()

class training_course(osv.osv):
    _inherit = 'training.course'

    def search(self, cr, uid, domain, offset=0, limit=None, order=None, context=None, count=False):
        # If search is done from the "Add Questions" button on
        # "Questionnaires", filter course add to only display courses
        # added to this questionnaire
        if context is None:
            context = {}
        questionnarie_pool = self.pool.get('training.exam.questionnaire.course')
        if context \
            and 'quizz_search' in context \
            and 'active_id' in context:

            question_course_ids = set()
            ques_ids = questionnarie_pool.search(cr, uid, [('questionnaire_id','=',context['active_id'])], context=context)
            for qc in questionnarie_pool.browse(cr, uid, ques_ids, context):
                question_course_ids.add(qc.course_id.id)

            domain.append(('id','in',list(question_course_ids)))
        return super(training_course, self).search(cr, uid, domain, offset, limit, order, context, count)

    def _has_questionnaire_compute(self, cr, uid, ids, fieldnames, args, context=None):
        tran_exam_questionnar_pool = self.pool.get('training.exam.questionnaire')
        res = dict.fromkeys(ids, False)
        for obj in self.browse(cr, uid, ids, context=context):
            res[obj.id] = len(tran_exam_questionnar_pool.search(cr, uid, [('main_course_id', '=', obj.id),('state', 'in', ('deprecated', 'validated'))], context=context)) > 0
        return res

    def _has_valid_questionnaire_compute(self, cr, uid, ids, fieldnames, args, context=None):
        tran_exam_questionnar_pool = self.pool.get('training.exam.questionnaire')
        res = dict.fromkeys(ids, False)

        for obj in self.browse(cr, uid, ids, context=context):
            res[obj.id] = len(tran_exam_questionnar_pool.search(cr, uid, [('main_course_id', '=', obj.id),('state', '=', 'validated'),('printable','=',True)], context=context)) > 0
        return res

    def _get_questionnaires(self, cr, uid, ids, context=None):
        result = set()
        for obj in self.browse(cr, uid, ids, context=context):
            if obj.main_course_id:
                result.add(obj.main_course_id.id)
            for course in obj.course_ids:
                result.add(course.course_id.id)
        return list(result)

    def _questionnaire_min_ready_date(self, cr, uid, ids, fieldnames, args, context=None):
        now_str = time.strftime('%Y-%m-%d 00:00:00')
        result = dict.fromkeys(ids, '2099-01-01 00:00:00')
        cr.execute("SELECT tp.course_questionnaire_id AS id, min(tp.date) AS date"
                   " FROM training_participation tp"
                   " WHERE tp.date >= %s"
                   " GROUP BY tp.course_questionnaire_id", (now_str,))
        for row in cr.dictfetchall():
            result[row['id']] = row['date']
        return result

    def _questionnaire_min_ready_date_search(self, cr, uid, obj, name, args, context=None):
        cmp_op = '='
        cmd_date = time.strftime('%Y-%m-%d %H:%M:%S')
        for arg in args:
            if isinstance(arg, (tuple,list,)) and arg[0] == name:
                cmd_op = arg[1]
                cmd_date = arg[2]
                break
        cr_query = """
            SELECT
                tc.id,
                CASE WHEN (z.id IS NOT NULL) THEN z.date ELSE 'infinity' END AS min_ready_date
            FROM
                training_course tc
            LEFT JOIN
                (SELECT tp.course_questionnaire_id AS id,min(tp.date) AS date
                 FROM training_participation tp WHERE tp.date >= now()
                 GROUP BY tp.course_questionnaire_id) AS z
              ON (tc.id = z.id)
            WHERE CASE WHEN (z.id IS NOT NULL) THEN z.date ELSE 'infinity' END %s %%s
            """ % (cmd_op[:2],)
        cr.execute(cr_query, (cmd_date,))
        result = set()
        for row in cr.fetchall():
            result.add(row[0])
        return [('id', 'in', list(result))]

    _columns = {
        'questionnaire_ids' : fields.one2many('training.exam.questionnaire', 'main_course_id', string='Questionnaire', readonly=True),
        'has_questionnaire' : fields.function(_has_questionnaire_compute, method=True, string='Has Questionnaires', type='boolean',
                                              store= { 'training.exam.questionnaire' : (_get_questionnaires, None, 10) }
                                             ),
        'has_valid_questionnaire': fields.function(_has_valid_questionnaire_compute, method=True, string='Has Valid Questionnaires', type='boolean',
                                              store = { 'training.exam.questionnaire': (_get_questionnaires, None, 10) }
                                             ),
        'questionnaire_min_ready_date': fields.function(_questionnaire_min_ready_date, type='datetime', string='Questionnaire Min Ready Date', method=True,
                                          fnct_search=_questionnaire_min_ready_date_search, help='The earliest date on which the course must have a valid questionnaire'),
        'forced_noquestionnaire': fields.boolean('Forced No Questionnaire', help='This course has no questionnaire, and that is normal', select=2),
        'forced_passing_score': fields.float('Forced Passing Score'),
        'forced_total_points': fields.float('Forced Total Points'),
        'no_exam': fields.boolean('No Exam', help='If checked, this mean we cannot make an exam subscription on this course'),
    }

    _defaults = {
        'forced_passing_score': lambda *a: 50.0,
        'forced_total_points': lambda *a: 100.0,
    }

training_course()

class training_offer(osv.osv):
    _inherit = 'training.offer'

    def _no_exam_compute(self, cr, uid, ids, fields, args, context=None):
        """ compute no_exam value on offer, this find out if we can do an
            'exam' for each offer

            rules are:
            - if offer is standalone (i.e contain only one course), and
              is not an 'exam' (make no sense to have an exam on an exam),
              then take the value defined for that unique course
            - in all other cases we can't take an exam on it (no_exam = True)
        """
        res = dict.fromkeys(ids, True)
        for offer in self.browse(cr, uid, ids):
            if offer.is_standalone and offer.kind != 'exam':
                res[offer.id] = offer.course_ids[0].course_id.no_exam
        return res

    _columns = {
        'questionnaire_ids' : fields.many2many('training.course', 'training_questionnaire_offer_rel', 'offer_id', 'questionnaire_id', 'Exams',
                                               domain=[('has_questionnaire', '=', True)]),
        'no_exam': fields.function(_no_exam_compute, method=True, string='No Exam', type='boolean', help='If checked, this mean we cannot make an exam subscription on this offer'),
    }

training_offer()

class training_exam_question(osv.osv):
    _name = 'training.exam.question'

training_exam_question()

class training_exam_questionnaire_question(osv.osv):
    _name = 'training.exam.questionnaire.question'
    _order = 'sequence asc'
    _columns = {
        'sequence' : fields.integer('Sequence', required=True),
        'question_id' : fields.many2one('training.exam.question', 'Question', required=True),
        'question_type' : fields.related('question_id', 'type', type='selection', string='Type', readonly=True,
                                         selection=[('qcm', 'QCM'),
                                                    ('qcu', 'QCU'),
                                                    ('plain', 'Plain'),
                                                    ('yesno', 'Yes/No')],
                                        ),
        'question_reference': fields.related('question_id', 'reference', type='char', string='Reference', size=32, readonly=True),
        'question_version': fields.related('question_id', 'version', type='integer', string='Version', readonly=True),
        'question_exposition' : fields.related('question_id', 'question', type='text', string='Exposition', readonly=True),
        'questionnaire_id' : fields.many2one('training.exam.questionnaire', 'Questionnaire', required=True, ondelete='cascade'),
    }

training_exam_questionnaire_question()

class training_question(osv.osv):
    _name = 'training.exam.question'
    _description = 'Training Exam Question'

    _count_points_from_selection = [
        ('question', 'Question'),
        ('answers', 'Answers'),
    ]

    def _number_of_good_answer(self, cr, uid, ids, name, args, context=None):
        values = dict.fromkeys(ids, 0)
        for obj in self.browse(cr, uid, ids, context=context):
            if obj.type in ('qcm', 'qcu'):
                if obj.question_answer_ids:
                    values[obj.id] = reduce(lambda prev, curr: prev + int(curr.is_solution == 'yes'), obj.question_answer_ids, 0)
            else:
                values[obj.id] = 1
        return values

    def _calc_question_answers_min_max(self, cr, uid, answers, context=None):
        """Compute question min / max points for question with 'count_points_from' == 'answers'
        @return {'min': 0.0, 'max': 0.0}
        """
        result = {'min': 0.0, 'max': 0.0}
        for answer in answers:
            if answer.get('is_solution','no') == 'yes':
                result['max'] += answer.get('point_checked',0.0) or 0.0
                result['min'] += answer.get('point_not_checked',0.0) or 0.0
            else:
                result['max'] += answer.get('point_not_checked',0.0) or 0.0
                result['min'] += answer.get('point_checked',0.0) or 0.0
        return result

    def onchange_question_answers(self, cr, uid, ids, count_points_from, answers, context=None):
        if count_points_from == 'answers':
            answers = [ a[2] for a in answers ]
            calc = self._calc_question_answers_min_max(cr, uid, answers, context=context)
            return {
                'value': {
                    'question_answer_min_points': calc['min'],
                    'question_answer_max_points': calc['max'],
                }
            }
        return {}

    def _get_answer_min_max_points(self, cr, uid, ids, fieldname, args, context=None):
        result = {}
        answer_obj = self.pool.get('training.exam.question.answer')
        for question_id in ids:
            answer_ids = answer_obj.search(cr, uid, [('question_id','=',question_id)], context=context)
            answer_data = answer_obj.read(cr, uid, answer_ids, ['is_solution','point_checked','point_not_checked'], context=context)
            calc = self._calc_question_answers_min_max(cr, uid, answer_data, context=context)
            values = {
                'question_answer_min_points': calc['min'],
                'question_answer_max_points': calc['max'],
            }
            result[question_id] = values
        return result

    def _get_question_ids_from_answers(self, cr, uid, ids, context=None):
        """return question ids which have answer ids listed in 'ids' param
        (store helper)
        """
        question_obj = self.pool.get('training.exam.question')
        return question_obj.search(cr, uid, [('question_answer_ids','in',ids)], context=context)

    def _get_question_point(self, cr, uid, ids, fieldname, args, context=None):
        if not ids:
            return {}
        result = dict.fromkeys(ids, 0.0)
        # get raw results
        cr.execute("SELECT id, point FROM training_exam_question WHERE id in %s", (tuple(ids),))
        raw_result = dict([(id, point) for id, point in cr.fetchall()])
        for question in self.browse(cr, uid, ids, context=context):
            if question.count_points_from == 'answers':
                if question.type == 'yesno':
                    result[question.id] = question.point_yesno_good_answer
                elif question.type in ('qcu','qcm'):
                    calc = self._get_answer_min_max_points(cr, uid, [question.id], None, None, context=context)
                    result[question.id] = calc[question.id]['question_answer_max_points']
            else:
                result[question.id] = raw_result[question.id]
        return result

    def _set_question_point(self, cr, uid, id, name, value, fnct_inv_arg, context=None):
        question = self.browse(cr, uid, id, context=context)
        if name =='point' and question.count_points_from != 'answers':
            cr.execute("UPDATE %s SET point = %%s WHERE id = %%s" % (self._table), (value, id,))

    _columns = {
        'name' : fields.char('Name', size=128, required=True, select=1, help='Name of Question', **WRITABLE_ONLY_IN_DRAFT),
        'reference': fields.char('Reference', size=32, required=True, readonly=True, help='The reference of the question'),
        'version': fields.integer('Version', required=True, readonly=True),
        'question' : fields.text('Question', required=True, select=1),
        'is_mandatory' : fields.boolean('Mandatory', help='Question is mandatory or not', **WRITABLE_ONLY_IN_DRAFT),
        'is_eliminatory' : fields.boolean('Eliminatory', **WRITABLE_ONLY_IN_DRAFT),
        'note' : fields.text('Note'),
        'type' : fields.selection(
                    [('qcm', 'QCM'),
                     ('qcu', 'QCU'),
                     ('plain', 'Plain'),
                     ('yesno', 'Yes/No') ],
                    'Type', required=True, select=1 ,help='Question type', **WRITABLE_ONLY_IN_DRAFT),
        'number_of_good_answers' : fields.function(_number_of_good_answer, method=True, string='Number of Good Answers', type='integer'),
        'response_plain' : fields.text('Solution', **WRITABLE_ONLY_IN_DRAFT),
        'response_yesno' : fields.selection([('yes', 'Yes'),('no', 'No')], 'Solution', **WRITABLE_ONLY_IN_DRAFT),
        'point_yesno_good_answer': fields.float('Yes/No Good Answer', digits=(12,2), **WRITABLE_ONLY_IN_DRAFT),
        'point_yesno_wrong_answer': fields.float('Yes/No Wrong Answer', digits=(12,2), **WRITABLE_ONLY_IN_DRAFT),
        'point_yesno_no_answer': fields.float('Yes/No No Answer', digits=(12,2), **WRITABLE_ONLY_IN_DRAFT),
        'question_answer_min_points': fields.function(_get_answer_min_max_points, type='float', string='Min Points', digits=(12,2), multi='min_max_points', method=True),
        'question_answer_max_points': fields.function(_get_answer_min_max_points, type='float', string='Max Points', digits=(12,2), multi='min_max_points', method=True),
        'question_answer_ids' : fields.one2many('training.exam.question.answer', 'question_id', 'Solution'),
        'questionnaire_ids': fields.many2many('training.exam.questionnaire',
                                              'training_exam_questionnaire_question',
                                              'question_id',
                                              'questionnaire_id',
                                              'Questionnaire', readonly=True),
        'course_ids' : fields.many2many('training.course',
                                        'training_question_course_rel',
                                        'question_id',
                                        'course_id',
                                        'Courses',
                                        select=1, **WRITABLE_ONLY_IN_DRAFT),
        'point': fields.function(_get_question_point, fnct_inv=_set_question_point, type='float', string='Point', digits=(12,2), required=True, method=True,
                                store={
                                    'training.exam.question': (lambda self, cr, uid, ids, context: ids, ['count_points_from','type'], 10),
                                    'training.exam.question.answer': (_get_question_ids_from_answers, ['is_solution', 'point_checked', 'point_not_checked'], 10),
                                },
                                states={'validated': [('readonly', True)], 'deprecated': [('readonly', True)]}, select=True,
                                help='Point related to question'),
        'count_points_from': fields.selection(_count_points_from_selection, 'Count Points From', required=True, **WRITABLE_ONLY_IN_DRAFT),
        'duration': fields.float('Duration', required=True, help='Time related to question', **WRITABLE_ONLY_IN_DRAFT),
        'image': fields.binary('Image', help='Image to be display next to question text', **WRITABLE_ONLY_IN_DRAFT),
        'image_position': fields.selection(
                        [('before_text', 'Before text'),
                         ('after_text', 'After text')],
                        'Image Position',
                        **WRITABLE_ONLY_IN_DRAFT),
        # plain question
        'free_lines_count': fields.integer('Free Lines',help='How many free line available to formulate a response to a plain question'),
        # states
        'state': fields.selection([
                                ('draft', 'Draft'),
                                ('validated', 'Validated'),
                                ('deprecated', 'Deprecated'),
                            ], 'State', required=True, readonly=True, select=1),

        'review_line_ids': fields.one2many('training.content.review.line', 'question_id', 'Review Lines', readonly=True),
        'deprecated_at': fields.datetime('Deprecated At', readonly=True),
        'deprecated_by': fields.many2one('res.users', 'Deprecated By', readonly=True),
        'deprecated_note': fields.text('Note', readonly=True),
    }

    def copy_data(self, cr, uid, id, default=None, context=None):
        if context is None:
            context = {}
        data = super(training_question, self).copy_data(cr, uid, id, default=default, context=context)
        if data:
            fields_to_remove = []
            if context.get('__internal_exam_resource_copy'):
                fields_to_remove += ['questionnaire_ids', 'version']
            if not context.get('new_revision'):
                # keep reference only in 'new revision' mode
                fields_to_remove += ['reference']
            for k in fields_to_remove:
                data.pop(k, None)
        # 'point' is a writable field (fcnt_inv), but the ORM won't copy it
        # because it's a function field, so we force it
        data['point'] = self.browse(cr, uid, id, context=context).point
        return data

    def copy(self, cr, uid, id, default=None, context=None):
        if context is None:
            context = {}
        # mark as internal copy - to not trigger course_ids, domain_ids updates!
        context['__internal_exam_resource_copy'] = True
        return super(training_question, self).copy(cr, uid, id, default=default, context=context)

    def action_draft(self, cr, uid, ids, context=None):
        self.write(cr, uid, ids, {'state': 'draft'}, context)
        return True

    def action_validate(self, cr, uid, ids, context=None):
        self._check_answer(cr, uid, ids, context=context)
        self.write(cr, uid, ids, {'state': 'validated'}, context=context)
        return True

    def action_deprecate(self, cr, uid, ids, context=None):
        if context is None:
            context = {}
        quizz_pool = self.pool.get('training.exam.questionnaire')
        linked_quizz_ids = quizz_pool.search(cr, uid, [('state','=','validated'),('question_ids.question_id','in',ids)])
        vals = {
            'state': 'deprecated',
            'deprecated_at': time.strftime('%Y-%m-%d %H:%M:%S'),
            'deprecated_by': uid,
            'deprecated_note': context.get('_deprecated_note', ''),
        }
        self.write(cr, uid, ids, vals, context)
        for quizz_id in linked_quizz_ids:
            quizz_pool._create_new_revision(cr, uid, quizz_id, context=context, note=context.get('_deprecated_note',''))
        return True

    def _ref_default(self, cr, uid, context=None):
        return self.pool.get('ir.sequence').get(cr, uid, 'training.exam.question')

    _defaults = {
        'name' : lambda *a: '',
        'reference': _ref_default,
        'version': lambda *a: 0,
        'is_mandatory' : lambda *a: 0,
        'is_eliminatory' : lambda *a: 0,
        'type' : lambda *a: 'plain',
        'response_yesno' : lambda *a: 0,
        'point' : lambda *a: 0,
        'duration':lambda *a: 1.0,
        'free_lines_count': lambda *a: 10,
        'state': lambda *a: 'draft',
        'count_points_from': 'question',
    }

    def _check_point(self, cr, uid, ids, context=None):
        if not ids:
            return False
        obj = self.browse(cr, uid, ids[0], context=context)
        return obj.point > 0

    def _check_answer(self, cr, uid, ids, context=None, raise_on_error=True):
        valid = True
        if not ids:
            valid = False
        for obj in self.browse(cr, uid, ids, context=context):
            if obj.type == 'qcm' and obj.number_of_good_answers:
                valid = int(obj.number_of_good_answers) >= 2
            elif obj.type == 'qcu':
                valid =  int(obj.number_of_good_answers) == 1
            if not valid:
                break
        if not valid and raise_on_error:
            raise osv.except_osv(_('Error'), _("Please check the number of good answer, it doesn't match your current question type"))
        return valid

    _constraints = [
        (_check_point, "Can you check the point of your question ?", ['point']),
    ]

    def search(self, cr, uid, domain, offset=0, limit=None, order=None, context=None, count=False):
        if context is None:
            context = {}
        course_id = context.get('course_id', False)
        kind = context.get('kind', False)
        without_course = context.get('course', False)

        if without_course:
            cr.execute("SELECT id "
                       "FROM training_exam_question "
                       "WHERE id NOT IN ( SELECT question_id FROM training_question_course_rel)")

            return [x[0] for x in cr.fetchall()]

        # Used by the Generator of Questionnaire
        if course_id and kind:
            values = ['qcm', 'qcu', 'yesno']

            if kind == 'manual':
                values.append('plain')

            cr.execute(
                "SELECT tqcr.question_id "
                "FROM training_question_course_rel tqcr, training_exam_question tq "
                "WHERE tq.id = tqcr.question_id "
                "  AND tqcr.course_id = %s "
                "  AND tq.type in (" + ",".join(['%s'] * len(values)) + ")",
                [course_id] + values,
            )

            return [x[0] for x in cr.fetchall()]

        questionnaire_id = context.get('questionnaire_id', False)
        if questionnaire_id:
            course_id = self.pool.get('training.exam.questionnaire').browse(cr, uid, questionnaire_id, context=context).main_course_id.id

        if course_id:
            cr.execute("SELECT question_id FROM training_question_course_rel WHERE course_id = %s",
                       (course_id,))

            domain.append(('id','in',[ x[0] for x in cr.fetchall() ]))

        return super(training_question, self).search(cr, uid, domain, offset=offset, limit=limit, order=order, context=context, count=count)

    def _get_extended_question_infos_elements(self, cr, uid, question, context=None, report_context=None):
        """
        @param question: browse_record(training.exam.question, ?)
        """
        if report_context is None:
            report_context = {}
        result = [('id', _('ID: %d') % (question.id,))]
        qn_fields = self.fields_get(cr, uid, fields=['type'], context=context)
        qn_type = question.type
        qn_type_list = [ x[1] for x in qn_fields.get('type', {}).get('selection', []) if x[0] == qn_type ]
        if qn_type_list:
            result.append(('type', _('TYPE: %s') % (qn_type_list[0])))
        if question.course_ids:
            if not report_context.get('force_course_id',False):
                course_list = [ c.name for c in question.course_ids ]
                result.append(('courses', _('COURSES: %s') % (', '.join(course_list))))
        if question.count_points_from == 'answers':
            if question.type == 'yesno':
                result.append(('points_info', _('Points info: Good: %s, Wrong: %s, No: %s') % (
                    question.point_yesno_good_answer,
                    question.point_yesno_wrong_answer,
                    question.point_yesno_no_answer,
                )))
            elif question.type in ('qcu','qcm'):
                result.append(('points_info', _('Points info: Min: %s, Max: %s') % (
                    question.question_answer_min_points,
                    question.question_answer_max_points,
                )))
        return result

    def _create_new_revision(self, cr, uid, id, context=None, note=None):
        def _get_last_revision(reference):
            cr.execute("SELECT MAX(version) FROM training_exam_question WHERE reference = %s", (reference,))
            r = cr.fetchone()
            if r:
                return r[0]
            return 0

        quest_pool = self
        quizz_pool = self.pool.get('training.exam.questionnaire')
        obj_record =  quest_pool.browse(cr, uid, id, context=context)
        last_rev = _get_last_revision(obj_record.reference)
        question_newrev_context = context.copy()
        question_newrev_context['new_revision'] = True
        new_id = quest_pool.copy(cr, uid, id, context=question_newrev_context)

        vals = {
            'version': last_rev + 1,
            'state': 'deprecated',
            'deprecated_at': time.strftime('%Y-%m-%d %H:%M:%S'),
            'deprecated_by': uid,
            'deprecated_note': note and note or False,
        }
        quest_pool.write(cr, uid, [id], vals, context=context)

        qz_note_info = _('ORIGIN: %s (id: %s)\n%s') % (obj_record.name, obj_record.id, note and tools.ustr(note) or '')
        #¸update related questionnaires
        qq_pool = self.pool.get('training.exam.questionnaire.question')
        qq_ids = qq_pool.search(cr, uid, [('question_id','=',id)], context=context)
        for qq in qq_pool.browse(cr, uid, qq_ids, context=context):
            if qq.questionnaire_id.state == 'deprecated':
                # questionnaire is not used anymore, nothing to do.
                continue
            elif qq.questionnaire_id.state in ['draft','pending','inprogress']:
                # questionnaire in draft mode, update the question reference
                qq_pool.write(cr, uid, [qq.id], {'question_id': new_id}, context=question_newrev_context)
            elif qq.questionnaire_id.state == 'validated':
                # questionnaire valid, must find or create a draft questionnaire
                # TODO: Need to implement question duplication of questionnaire!
                draft_quizzes = quizz_pool.search(cr, uid, [
                                    ('state','=','draft'),
                                    ('reference','=',qq.questionnaire_id.reference)],
                                    context=context)
                if not draft_quizzes:
                    new_quizz_id = quizz_pool._create_new_revision(cr, uid, qq.questionnaire_id.id, context=question_newrev_context, note=qz_note_info)
                    draft_quizzes = [ new_quizz_id ]

                qq_to_update = qq_pool.search(cr, uid, [('questionnaire_id', 'in', draft_quizzes),('question_id', '=', id)], context=context)
                if qq_to_update:
                    qq_pool.write(cr, uid, qq_to_update, {'question_id': new_id}, context=question_newrev_context)
            else:
                raise osv.except_osv(_('Error'), _('Unhandled case'))

        return new_id


training_question()

class training_question_answer(osv.osv):
    _name = 'training.exam.question.answer'
    _description = 'Training Question Answer'

    _states_selection = [
        ('draft', 'Draft'),
        ('validated', 'Validated'),
        ('deprecated', 'Deprecated')
    ]

    _columns = {
        'name' : fields.text('Solution', required=True, select=1),
        'is_solution' : fields.selection([('yes', 'Yes'),('no', 'No')], 'Acceptable Solution', required=True, **WRITABLE_ONLY_IN_DRAFT),
        'question_id' : fields.many2one('training.exam.question', 'Question', select=True, required=True, ondelete="cascade"),
        'state': fields.related('question_id', 'state', type='selection', selection=_states_selection, string='State', readonly=True),
        'point_checked': fields.float('Point (case checked)', digits=(12,2), **WRITABLE_ONLY_IN_DRAFT),
        'point_not_checked': fields.float('Point (case not checked)', digits=(12,2), **WRITABLE_ONLY_IN_DRAFT),
    }

    def create(self, cr, uid, vals, context=None):
        question_id = vals.get('question_id')
        if question_id:
            question = self.pool.get('training.exam.question').browse(cr, uid, question_id, context=context)
            if question.state != 'draft':
                raise osv.except_osv(_('Error'), _('You can only add answer to a draft question'))
        return super(training_question_answer, self).create(cr, uid, vals, context=context)

    def unlink(self, cr, uid, ids, context=None):
        if ids:
            for answer in self.browse(cr, uid, ids, context=context):
                if answer.question_id.state != 'draft':
                    raise osv.except_osv(_('Error'), _('You can only remove answer to a draft question'))
        return super(training_question_answer, self).unlink(cr, uid, ids, context=context)

    _defaults = {
        'point_checked': 0.0,
        'point_not_checked': 0.0,
    }

training_question_answer()

class training_exam_questionnaire_course(osv.osv):
    _name = 'training.exam.questionnaire.course'
    _description = 'Training Exam Questionnaire Course'
    _rec_name = 'course_id'

    def f_total_point(self, cr, uid, ids, fn, args, context=None):
        quizz_proxy = self.pool.get('training.exam.questionnaire')
        res = dict.fromkeys(ids, 0.0)

        course_ids = {} # course_ids -> ids
        quizz_ids = set() # list of quizz to question from
        for obj in self.browse(cr, uid, ids, context=context):
            quizz_ids.add(obj.questionnaire_id.id)
            course_ids[obj.course_id.id] = obj.id
        quizz_ids = list(quizz_ids)

        for quizz in quizz_proxy.browse(cr, uid, quizz_ids, context=context):
            if not len(quizz.question_ids):
                # quizz has no question on it
                continue
            for questions in quizz.question_ids:
                courses = questions.question_id.course_ids
                # double conversion to avoid duplicate id, in case
                # many2many contains duplicate keyse_ids
                courses = list(set(courses))
                for course in courses:
                    try:
                        obj_id = course_ids[course.id]
                    except KeyError:
                        # course is not related to this quizz
                        continue
                    res[obj_id] += questions.question_id.point
        return res or 0.0

    def _get_states_selection(self, cr, uid, context=None):
        return self.pool.get('training.exam.questionnaire')._states_selection

    _columns = {
        'course_id': fields.many2one('training.course', 'Course', required=True, ondelete="cascade"),
        'questionnaire_id': fields.many2one('training.exam.questionnaire', 'Questionnaire', required=True, ondelete='cascade'),
        'category_id': fields.related('course_id', 'category_id', type='many2one', relation='training.course_category',
                                    string='Category of Course', readonly=True),
        'kind': fields.related('course_id', 'kind', type='selection', selection=training_course_kind_compute, string='Kind', readonly=True),
        'passing_score': fields.float('Passing Score', required=True, help='Passing score of  related to Exam'),
        'total_point': fields.function(f_total_point, method=True, type='float', string='Total Point'),
        'state': fields.related('questionnaire_id', 'state', type='selection', selection=_get_states_selection,
                                string='State', readonly=True),
    }

    _defaults = {
        'passing_score': lambda *a: 0.0,
    }

    def on_change_course(self, cr, uid, ids, course_id, context=None):
        if not course_id:
            return {'value' : {'category_id':''}}
        course = self.pool.get('training.course').browse(cr, uid, course_id, context=context)
        result = self.pool.get('training.course_category').search(cr, uid, [('analytic_account_id','=',course.parent_id.id)], context=context)
        return {'value' : {'category_id' :  result[0]}}

training_exam_questionnaire_course()

class training_exam_questionnaire_course_passing_score_wizard(osv.osv_memory):
    _name = 'training.exam.questionnaire.course.passingscore.wizard'
    _columns = {
        'passing_score': fields.float('Passing Score'),
    }

    def _default_get_passing_score(self, cr, uid, context=None):
        print("default_get_passing_score: %s" % (context,))
        if context is None:
            context = {}
        active_id = context.get('active_id')
        if active_id:
            questionnaire_course_obj = self.pool.get('training.exam.questionnaire.course')
            return questionnaire_course_obj.browse(cr, uid, active_id, context=context).passing_score
        else:
            return 0.0

    _defaults = {
        'passing_score': _default_get_passing_score,
    }

    def button_update_passing_score(self, cr, uid, ids, context=None):
        if context is None:
            context = {}
        active_id = context.get('active_id')
        if not active_id or not ids:
            return {}
        wizard = self.browse(cr, uid, ids[0], context=context)
        value = {'passing_score': wizard.passing_score}
        questionnaire_course_obj = self.pool.get('training.exam.questionnaire.course')
        questionnaire_course_obj.write(cr, uid, [active_id], value, context=context)
        return {}
training_exam_questionnaire_course_passing_score_wizard()

class exam_questionnaire(osv.osv):
    _name = 'training.exam.questionnaire'
    _description = 'Training Exam Questionnaire'

    def copy_data(self, cr, uid, id, values=None, context=None):
        if context is None:
            context = {}
        data = super(exam_questionnaire, self).copy_data(cr, uid, id, values, context=context)
        if data:
            fields_to_remove = []
            if context.get('__internal_exam_resource_copy'):
                fields_to_remove += ['version']
            if not context.get('new_revision'):
                # only keep reference in 'new revision' mode
                fields_to_remove += ['reference']
            for k in fields_to_remove:
                data.pop(k, None)
        return data

    def copy(self, cr, uid, id, default=None, context=None):
        if context is None:
            context = {}
        # mark as internal copy - to not trigger course_ids, domain_ids updates!
        context['__internal_exam_resource_copy'] = True
        return super(exam_questionnaire, self).copy(cr, uid, id, default=default, context=context)

    def _func_get_type(self, cr, uid, ids, name, args, context=None):
        result = {}
        for quizz in self.browse(cr, uid, ids, context=context):
            type = 'automatic'
            for question in quizz.question_ids:
                if question.question_type == 'plain':
                    type = 'manual'
                    break
            result[quizz.id] = type
        return result

    def _get_quizz_from_quizz_lines(self, cr, uid, ids, context=None):
        if not ids:
            return []
        quizzes = set()
        for quizz_line in self.read(cr, uid, ids, ['questionnaire_id'], context=context):
            quizzes.add(quizz_line['questionnaire_id'] and quizz_line['questionnaire_id'][0])
        return quizzes

    def _get_quizz_from_questions(self, cr, uid, ids, context=None):
        if not ids:
            return []
        pool_quizz_q = self.pool.get('training.exam.questionnaire.question')
        quizzes = set()
        quizz_lines = pool_quizz_q.search(cr, uid, [('question_id','in',ids)])
        for quizz_line in pool_quizz_q.read(cr, uid, quizz_lines, ['questionnaire_id'], context=context):
            quizzes.add(quizz_line['questionnaire_id'] and quizz_line['questionnaire_id'][0])
        return quizzes

    def _len_question_ids(self, cr, uid, ids, field_name, value, context=None):
        return dict([(obj.id, len(obj.question_ids)) for obj in self.browse(cr, uid, ids, context=context)])

    #compute the time of questionnaire
    def point_compute(self, cr, uid, ids, name, args, context=None):
        res = dict.fromkeys(ids, '')
        for questionnaire in self.browse(cr, uid, ids, context=context):
            res[questionnaire.id] = reduce(lambda acc,question: acc + question.question_id.point, questionnaire.question_ids, 0.0)
        return res or 0.0

    def duration_compute(self, cr, uid, ids, name, args, context=None):
        res = dict.fromkeys(ids, '')
        for questionnaire in self.browse(cr, uid, ids ,context=context):
            res[questionnaire.id] = reduce(lambda acc,question: acc + question.question_id.duration, questionnaire.question_ids, 0.0)
        return res or 0.0

    def course_category(self, cr, uid, ids, name, args, context=None):
        coursecat_proxy = self.pool.get('training.course_category')
        res = dict.fromkeys(ids, '')
        for obj in self.browse(cr,uid,ids,context=context):
            results = []
            for course in obj.course_ids:
                category_id = course.course_id.category_id.id
                if category_id:
                    results.append(category_id)
            # only keep the first reference (exam quizz should only have
            # one course attached, for other don't specify product line
            # to avoid errors
            if len(results) == 1:
                res[obj.id] = results[0]
            else:
                res[obj.id] = []
        return res

    def f_passing_score(self, cr, uid, ids, fn, args, context=None):
        """Quizz Global Passing Score
        The global passing score is the sum of all passing score by
        courses
        """
        res = dict.fromkeys(ids, '')
        for obj in self.browse(cr, uid, ids, context=context):
            if not obj.course_ids:
                res[obj.id] = obj.total_point / 2.0
            else:
                res[obj.id] = reduce(lambda acc, c: acc + c.passing_score,
                                      obj.course_ids, 0.0)
        return res or 0.0

    def _f_update_from_questionnaire_questions(self, cr, uid, ids, context=None):
        """return questionnaire ids for which question were added/removed/modified
           (note: we only take into account questionnaire in draft state)"""
        r = [ x.questionnaire_id.id for x in self.browse(cr, uid, ids, context=context) if x.questionnaire_id.state == 'draft' ]
        return r

    def _f_update_from_questions(self, cr, uid, ids, context=None):
        """return questionnaire ids which contain question which were modified
           (note: we only take into account questionnaire and question in draft state)"""
        r = []
        for q in self.browse(cr, uid, ids, context=context):
            if q.state != 'draft':
                continue
            r.extend([ x.id for x in q.questionnaire_ids if x.state == 'draft' ])
        return r

    def _f_update_from_question_answers(self, cr, uid, ids, context=None):
        r = []
        for a in self.browse(cr, uid, ids, context=context):
            q = a.question_id
            if q.state != 'draft':
                continue
            r.extend([ x.id for x in q.questionnaire_ids if x.state == 'draft' ])
        return r

    def _f_update(self, cr, uid, ids, fieldname, args, context=None):
        self._trigger_categorization_update(cr, uid, ids, context=context, passing_score_update=True)
        return dict.fromkeys(ids, False)

    def _is_printable(self, cr, uid, ids, fn, args, context=None):
        res = dict.fromkeys(ids, True)
        disabled_ids = self._search_is_printable(cr, uid, ids, '', [('printable', '=', False)])[0][2]
        for id in disabled_ids:
            res[id] = False
        return res

    def _search_is_printable(self, cr, uid, ids, fn, args, context=None):
        op = 'not in'
        for a in args:
            if a[0] == 'printable':
                if int(a[2]) > 0:
                    op = 'not in'
                else:
                    op = 'in'
                break
        rids = self.search(cr, uid, [('forced_noprint','=',True)])
        cr.execute("""
            SELECT distinct(questionnaire_id)
            FROM training_exam_questionnaire_question teqq
            WHERE question_id IN (
                    SELECT tcrl.question_id
                    FROM training_content_review tcr
                    LEFT JOIN training_content_review_reason tcrr ON (tcr.reason_id = tcrr.id)
                    LEFT JOIN training_content_review_line tcrl ON (tcrl.review_id = tcr.id)
                    WHERE tcrr.type = 'exam_material'
                      AND tcr.state in ('validated','inprogress')
                    GROUP BY tcrl.question_id, tcr.id
                    ORDER BY tcrl.question_id
                )""")
        rids.extend([ x[0] for x in  cr.fetchall() ])
        return [('id', op, rids)]

    _states_selection = [
        ('draft', 'Draft'),
        ('validated', 'Validated'),
        ('deprecated', 'Deprecated'),
    ]

    _columns = {
        'name' : fields.char( 'Name', size=128, required=True, select=1 ,help='Name of questionnaire'),
        'reference': fields.char('Reference', size=32, select=1, required=True, readonly=True, help='Reference number of the questionnaire'),
        'version': fields.integer('Version', required=True, readonly=True),
        'state' : fields.selection(_states_selection,
                                   'State', required=True, readonly=True, select=1, help='The state of the Questionnaire'),
        'kind': fields.function(_func_get_type, method=True, type='selection', string='Correction Mode',
                                 selection=[('automatic','Automatic'),
                                            ('manual','Manual')],
                                 store={
                                     'training.exam.questionnaire.question': (_get_quizz_from_quizz_lines, [], 0),
                                     'training.exam.question': (_get_quizz_from_questions, ['type'], 0),
                                 },
                                 select=1, help='How this questionnaire could be corrected'),
        'type': fields.selection([('examen', 'Examen')], 'Type', required=True, select=2),
        'objective' : fields.text('Objective'),
        'description' : fields.text('Description'),
        'remark_firstpage': fields.text('Remark First Page'),
        'main_course_id' : fields.many2one('training.course', 'Main Course', required=True, select=1, ondelete='restrict',
                                      domain="[('state_course', 'in', ('draft', 'validated', 'pending'))]"),
        'questionnaire_min_ready_date': fields.related('main_course_id', 'questionnaire_min_ready_date',
                                            type='datetime', string='Questionnaire Min Ready Date', readonly=True),
        'course_ids': fields.one2many('training.exam.questionnaire.course', 'questionnaire_id', string='Courses'),
        'total_point' : fields.function(point_compute,method=True, type='float', string='Total Point', help='Total point for the questionnaire'),
        'duration' : fields.function(duration_compute,method=True, type='float', string='Duration', help='Duration for the exam'),
        'question_ids' : fields.one2many('training.exam.questionnaire.question', 'questionnaire_id', 'Questions'),
        'len_question_ids' : fields.function(_len_question_ids, method=True, type='integer', string='Number of Questions',
                                             help='Total number of questions'),
        'passing_score': fields.function(f_passing_score, method=True, type='float', string='Global Passing Score',
                                        help='The global passing score of the questionnaire'),
        'category' : fields.function(course_category, method=True, type='many2one', relation='training.course_category',
                                     string='Category of  Course', select=1),
        'attachment_ids': fields.one2many('training.exam.questionnaire.attachment', 'questionnaire_id', 'Attachments',),
        'forced_noprint': fields.boolean('Forced No Print', help='Check this to avoid choosing this questionnaire when assigning it to the participation', select=2),
        'printable': fields.function(_is_printable, fnct_search=_search_is_printable, type='boolean', method=True,
                                string='Printable', help='If check this questionnaire is printable', select=2),
        'deprecated_at': fields.datetime('Deprecated At', readonly=True),
        'deprecated_by': fields.many2one('res.users', 'Deprecated By', readonly=True),
        'deprecated_note': fields.text('Note', readonly=True),
        'update': fields.function(_f_update, type='boolean', string='Internal Store Update', help='This stored field help updating categorization (courses) on questionnaire',
                                        method=True,
                                        store={
                                            'training.exam.questionnaire.question': (_f_update_from_questionnaire_questions, [], 30),
                                            'training.exam.question': (_f_update_from_questions, ['point', 'count_points_from'], 20),
                                            'training.exam.question.answer': (_f_update_from_question_answers, ['point_checked', 'point_not_checked'], 10),
                                        }),
    }

    def _ref_default(self, cr, uid, context=None):
        return self.pool.get('ir.sequence').get(cr, uid, 'training.exam.questionnaire')

    _defaults = {
        'name' : lambda *a: '',
        'reference': _ref_default,
        'version': lambda *a: 0,
        'state' : lambda *a: 'draft',
        'kind' : lambda *a: 'manual',
        'duration' : lambda *a: 2.0,
        'passing_score' : lambda *a: 1.0,
        'type': lambda *a: 'examen',
    }

    def _check_course_limit(self, cr, uid, ids, context=None):
        if not ids:
            return False
        obj = self.browse(cr, uid, ids[0], context=None)
        if obj.type == 'examen' and len(obj.course_ids) > 1:
            return False
        return True

    def _check_score(self, cr, uid, ids, context=None):
        if not ids:
            return False
        obj = self.browse(cr, uid, ids[0], context=context)
        return 0.0 <= obj.passing_score <= obj.total_point

    _constraints = [
        #(_check_course_limit, "Only one course allowed for questionnaire with type 'Examen'", ['course_ids']),
        (_check_score, "Can you check the passing score  ?", ['passing_score']),
    ]

    def set_pending_status(self, cr, uid, ids, context=None):
        workflow = netsvc.LocalService('workflow')
        for oid in ids:
            cr.execute("SELECT id FROM wkf_instance WHERE res_type = %s AND res_id = %s AND state = 'active'",
                        (self._name, oid))
            r = cr.fetchall()
            if not len(r):
                # we are in 'deprecated' state or 'validated' state (only for preceding workflow version)
                # where those node are final state (flow_stop = True), the workflow is frozen, we need to
                # create a new one.
                workflow.trg_create(uid, self._name, oid, cr)
            workflow.trg_validate(uid, self._name, oid, 'signal_teq_pending', cr)

        return self.write(cr, uid, ids, {'state' : 'pending'}, context=context)

    def reset_to_draft_cb(self, cr, uid, ids, context=None):
        workflow = netsvc.LocalService('workflow')
        for oid in ids:
            workflow.trg_create(uid, self._name, oid, cr)
        return self.write(cr, uid, ids, {'state' : 'draft'}, context=context)

    def wkf_validate(self, cr, uid, ids, context=None):
        for quizz in self.browse(cr, uid, ids, context=context):
            if not all([ x.question_id.state == 'validated' for x in quizz.question_ids]):
                raise osv.except_osv(_('Error'), _('You cannot validate a questionnaire which don\'t have all their questions validated'))
        return self.write(cr, uid, ids, {'state': 'validated'})

    def wkf_deprecate(self, cr, uid, ids, context=None):
        self.write(cr, uid, ids, {
            'state': 'deprecated',
            'deprecated_at': time.strftime('%Y-%m-%d %H:%M:%S'),
            'deprecated_by': uid
            }, context=context)

    def _create_new_revision(self, cr, uid, id, context=None, note=None):
        def _get_last_revision(reference):
            cr.execute("SELECT MAX(version) FROM training_exam_questionnaire WHERE reference = %s", (reference,))
            r = cr.fetchone()
            if r:
                return r[0]
            return 0

        ira_pool = self.pool.get('ir.attachment')
        quizz_pool = self
        obj_record =  quizz_pool.browse(cr, uid, id, context=context)
        last_rev = _get_last_revision(obj_record.reference)
        quizz_newrev_context = context.copy()
        quizz_newrev_context['new_revision'] = True
        new_id = quizz_pool.copy(cr, uid, id, context=quizz_newrev_context)
        new_obj = quizz_pool.browse(cr, uid, new_id, context=context)
        new_obj_vals = {}
        for line in new_obj.attachment_ids:
            new_aid = ira_pool.copy(cr, uid, line.attachment_id.id, context=context)
            ira_pool.write(cr, uid, [new_aid], {
                'name': line.attachment_id.name,
                'res_model': 'training.exam.questionnaire',
                'res_id': new_id,
                }, context=context)
            new_obj_vals.setdefault('attachment_ids', []).append((1, line.id, {'attachment_id': new_aid}))
        if new_obj_vals:
            quizz_pool.write(cr, uid, [new_id], new_obj_vals, context=context)

        workflow = netsvc.LocalService('workflow')
        workflow.trg_validate(uid, quizz_pool._name, id, 'signal_teq_deprecate', cr)
        vals = {
            'version': last_rev + 1,
            'deprecated_at': time.strftime('%Y-%m-%d %H:%M:%S'),
            'deprecated_by': uid,
            'deprecated_note': note and note or False,
        }
        quizz_pool.write(cr, uid, [id], vals, context=quizz_newrev_context)
        return new_id

    def _trigger_categorization_update(self, cr, uid, ids, context=None, passing_score_update=True):
        """add/remove categorizations field update questionnaire
           (this imply course_ids [training_exam] and domain_ids [training_diagnostic])"""
        quizz_course = self.pool.get('training.exam.questionnaire.course')
        print("TRIGGER << COURSE UPDATE >>")
        if context is None:
            context = {}
        if context.get('__internal_exam_resource_copy') or context.get('new_revision'):
            print("SKIPPP INTENAL COPY or NEW REVISION")
            return True
        for quizz in self.browse(cr, uid, ids, context=context):
            if quizz.state != 'draft':
                continue # only update draft questionnaires!

            quizz_type = quizz.type
            # collect existing courses
            existing_course_dict = dict(( (c.course_id.id, c) for c in quizz.course_ids ))
            existing_course_ids = set(existing_course_dict.iterkeys())
            # search current domains based on questions
            current_course_ids = set()
            for q in quizz.question_ids:
                for c in q.question_id.course_ids:
                    current_course_ids.add(c.id)
            if quizz_type == 'examen' and quizz.main_course_id:
                current_course_ids = set([quizz.main_course_id.id])

            # course compute differences
            course_ids_to_remove = existing_course_ids - current_course_ids
            course_ids_to_add = current_course_ids - existing_course_ids

            # remove un-necessary courses
            cids = [ existing_course_dict[x].id for x in course_ids_to_remove ]
            quizz_course.unlink(cr, uid, cids, context=context)

            quizz_id = quizz.id

            # add new courses
            for cid in course_ids_to_add:
                q = {
                    'course_id': cid,
                    'questionnaire_id': quizz_id,
                    'passing_score': 0.0, # make sure constraint wont block us!
                }
                #XXX: other value of training.exam.questionnaire.course should
                #     have default values otherwise next call will fail!
                quizz_course.create(cr, uid, q)

            # update passing score on course (default to 50%)
            if passing_score_update:
                new_all_quizz_courses = quizz_course.search(cr, uid, [('questionnaire_id','=',quizz_id)], context=context)
                for qc in quizz_course.browse(cr, uid, new_all_quizz_courses, context=context):
                    quizz_course.write(cr, uid, [qc.id], {'passing_score': qc.total_point / 2.0}, context=context)
        return True

exam_questionnaire()

class training_subscription(osv.osv):
    _inherit = 'training.subscription'

    def action_subscription_line_compute(self, cr, uid, ids, context=None):
        if not len(ids):
            return False

        sl_proxy = self.pool.get('training.subscription.line')
        session_proxy = self.pool.get('training.session')
        questionnaire_proxy = self.pool.get('training.exam.questionnaire')

        for subscription in self.browse(cr, uid, ids, context=context):
            for sl in subscription.subscription_line_ids:
                if (not sl.computed) and sl.exam_session_id:
                    # Rechercher tous les cours provenant de l'offre associée à la session
                    # Rechercher les questionnaires appartenant à ces cours
                    course_ids = set()

                    for course in sl.session_id.offer_id.course_ids:
                        course_ids.add(course.course_id.id)

                    for course_id in course_ids:
                        course = self.pool.get('training.course').browse(cr, uid, course_id, context=context)
                        if not course.course_type_id.product_id:
                            raise osv.except_osv(_('Warning'),
                                                 _("Can you check the product on the course type of the course ?"))

                        price = self.pool.get('product.pricelist').price_get(cr, uid,
                                                                                 [sl.price_list_id.id],
                                                                                 course.course_type_id.exam_product_id.id,
                                                                                 subscription.partner_id.id,
                                                                                 1.0)[sl.price_list_id.id]
                        values = {
                                'job_id' : sl.job_id.id,
                                'job_email' : sl.job_id.email or '',
                                'subscription_id' : sl.subscription_id.id,
                                'session_id' : sl.exam_session_id.id,
                                'parent_id' : sl.id,
                                'course_id' : course.id,
                                'price_list_id' : sl.price_list_id.id,
                                'price' : price,
                        }
                        sl_proxy.create(cr, uid, values, context=context)
                        sl.write({'computed' : 1})

        return True

training_subscription()

class training_subscription_line(osv.osv):
    _inherit = 'training.subscription.line'
    _columns = {
        'exam_session_id' : fields.many2one('training.session', 'Exam Session', domain=[('kind', '=', 'exam')], **WRITABLE_ONLY_IN_DRAFT),
        'course_id' : fields.many2one('training.course', 'Exam',
                                      domain=[('state_course', 'in', ['validated', 'pending'])], ondelete='restrict'),
        'parent_id' : fields.many2one('training.subscription.line', 'Parent', ondelete='set null'),
        'computed' : fields.boolean('Computed'),
        'no_exam': fields.related('session_id', 'offer_id', 'no_exam', type='boolean', string="No Exam", readonly=True),
    }

    _defaults = {
        'computed' : lambda *a: 0,
    }

    def _check_subscription(self, cr, uid, ids, context=None):
        standard_line_ids = []
        for line in self.browse(cr, 1, ids, context=context):
            if line.session_id.kind != 'exam':
                standard_line_ids.append(line.id)
                continue

            session = line.session_id
            contact = line.job_id.contact_id
            course = line.course_id
            name = "%s %s" % (contact.first_name, contact.name)

            same_contact_same_session = ['&', '&', '&', '&',
                                            ('contact_id', '=', contact.id),
                                            ('session_id', '=', session.id),
                                            ('course_id', '=', course.id),
                                            ('id', '!=', line.id),
                                            ('state', '!=', 'cancelled')]

            line_ids = self.search(cr, 1, same_contact_same_session, context=context)
            if line_ids:
                raise osv.except_osv(_('Error'),
                                     _('%(participant)s is already following the session "%(session)s" with exam "%(exam)s"') % {
                                        'participant': name, 'session': session.name, 'exam': course.name})
        if not standard_line_ids:
            return True # all lines was of kind 'exam'
        return super(training_subscription_line, self)._check_subscription(cr, uid, standard_line_ids, context=context)

    def on_change_session(self, cr, uid, ids, session_id, price_list_id, partner_id, context=None):
        if not session_id:
            return False
        session = self.pool.get('training.session').browse(cr, uid, session_id, context=context)
        if session.kind == 'exam':
            price_list = self._get_price_list_from_product_line(session, partner_id, price_list_id)
            return {'value': {'kind': 'exam', 'price_list_id': price_list, 'no_exam': True}} #, 'course_id': False}}

        res = super(training_subscription_line, self).on_change_session(cr, uid, ids, session_id,
                                                                        price_list_id, partner_id,
                                                                        context=None)
        if res and 'kind' in res['value']:
            res['value']['course_id'] = 0

        res['value']['no_exam'] = session.offer_id.no_exam

        return res

    def on_change_exam(self, cr, uid, ids, session_id, price_list_id, course_id, partner_id, context=None):
        # If exam course is from a product line with special members,
        # we must apply this pricelist if current partner is one of those
        ocv = {'value':{}}
        if course_id:
            course = self.pool.get('training.course').browse(cr, uid, course_id, context=context)
            if course.category_id and course.category_id.price_list_id:
                if any(partner.id == partner_id for partner in course.category_id.partner_ids):
                    price_list_id = course.category_id.price_list_id.id
                    ocv['value']['price_list_id'] = price_list_id
            else:
                # get partner default pricelist
                p = self.pool.get('res.partner').browse(cr, uid, partner_id, context=context)
                pl = p.property_product_pricelist and p.property_product_pricelist.id or False
                ocv['value']['price_list_id'] = pl

        ocv_p = self.on_change_price_list(cr, uid, ids, session_id, ocv['value'].get('price_list_id',price_list_id), course_id, context)
        if ocv_p:
            ocv['value'].update(ocv_p['value'])
        return ocv

    def on_change_price_list(self, cr, uid, ids, session_id, price_list_id, course_id, context=None):
        if not session_id or not price_list_id:
            return False
        session = self.pool.get('training.session').browse(cr, uid, session_id, context=context)
        if session.kind != 'exam':
            return super(training_subscription_line, self).on_change_price_list(cr, uid, ids, session_id, price_list_id, context=context)

        if not course_id:
            return False
        course = self.pool.get('training.course').browse(cr, uid, course_id, context=context)

        pricelist_proxy = self.pool.get('product.pricelist')

        if not course.course_type_id.exam_product_id:
            raise osv.except_osv(_('Warning'),
                                 _("Can you check the product on the course type of the course %s") % course.name)

        return {
            'value' : {
                'price' : pricelist_proxy.price_get(cr, uid, [price_list_id], course.course_type_id.exam_product_id.id, 1.0)[price_list_id]
            }
        }

    def _get_values_from_wizard(self, cr, uid, subscription_id, job, subscription_mass_line, context=None):
        subscription = self.pool.get('training.subscription').browse(cr, uid, subscription_id, context=context)
        session = subscription_mass_line.session_id
        def_pricelist_id = job.name.property_product_pricelist.id


        values = {
            'subscription_id' : subscription_id,
            'job_id' : job.id,
            'job_email': job.email,
            'session_id' : session.id,
            'price' : 0.0,
        }
        ocv = self.on_change_session(cr, uid, [], session.id, def_pricelist_id, job.name.id, context=context)
        if ocv and 'value' in ocv:
            values.update(ocv['value'])

        course_id = subscription_mass_line.course_id and subscription_mass_line.course_id.id or False
        ocv = self.on_change_exam(cr, uid, [], values['session_id'], values.get('price_list_id', False), course_id, job.name.id, context=context)
        if ocv and 'value' in ocv:
            values.update(ocv['value'])

        values.update({
            'exam_session_id' : getattr(subscription_mass_line.exam_session_id, 'id'),
            'course_id' : course_id,
        })
        return values

    def _get_courses(self, cr, uid, ids, context=None):
        res = dict.fromkeys(ids, [])

        standard_ids = []
        for sl in self.browse(cr, uid, ids, context=context):
            if sl.kind == 'exam':
                res[sl.id] = [sl.course_id]
            else:
                standard_ids.append(sl.id)

        values = super(training_subscription_line, self)._get_courses(cr, uid, standard_ids, context=context)
        res.update(values)
        return res

    # training.subscription.line
    def _get_invoice_line_data(self, cr, uid, name, invoice_id, partner, session, subline, context=None):
        values = super(training_subscription_line, self)._get_invoice_line_data(cr, uid, name, invoice_id, partner, session, subline, context=context)
        if not values:
            values = {}
        if subline.kind == 'exam' and subline.course_id:
            name = "%s %s" % (subline.contact_id.first_name, subline.contact_id.name,)
            values['name'] = "%s: %s (%s)" % (_('Exam'), subline.course_id.name, name)
        return values

    # training.subscription.line
    def _get_invoice_line_taxes(self, cr, uid, subline, fiscal_position, partner, session, context=None):
        if subline.kind == 'exam':
            _slc = subline.course_id
            if _slc:
                _slcp = subline.course_id.course_type_id.exam_product_id
                if _slcp.taxes_id:
                    taxes = _slcp.taxes_id
                    tax_id = self.pool.get('account.fiscal.position').map_tax(cr, uid, fiscal_position, taxes)
                    return [(6, 0, [t for t in tax_id])]
            return []
        # else
        return super(training_subscription_line, self)._get_invoice_line_taxes(cr, uid, subline, fiscal_position, partner, session, context=context)

    # training.subscription.line
    def _invoicing_get_grouping_key(self, cr, uid, ids, subline, context=None):
        if context is None:
            context = {}
        key = super(training_subscription_line, self)._invoicing_get_grouping_key(cr, uid, ids, subline, context)
        if context.get('group_by_subscription') is True:
            # do not override grouping key if user want to invoice whole subscription
            return key
        if subline.kind == 'exam':
            key = key + (subline.course_id.id,)
        return key

    # training.subscription.line
    def _invoicing_get_grouping_data(self, cr, uid, ids, key, subline, journal=None, context=None):
        if context is None:
            context = {}
        data = super(training_subscription_line, self)._invoicing_get_grouping_data(cr, uid, ids, key, subline, journal, context)
        if subline.kind == 'exam':

            # setup context
            session = subline.session_id
            partner = subline.partner_id
            course = subline.course_id
            # overwrite value specific to exam grouping
            if not context.get('group_by_subscription',False):
                data.update({
                    'name' : "%s - %s" % (session.name, course.name),
                    'origin' : "%s - %s" % (session.name, course.name),
                    'reference' : "%s - %s - %s" % (partner.name, session.name, course.name),
                })
                if context.get('cancellation', False) is True:
                    data['name'] += ' - ' + _('Cancellation')

        return data

training_subscription_line()

class training_participation_course_line(osv.osv):
    _name = 'training.participation.course.line'
    _rec_name = 'course_id'

    def f_get_score(self, cr, uid, ids, fieldnames, args, context=None):
        res = dict.fromkeys(ids, 0.0)
        result_per_part = {}
        courses_per_part = {}

        for tpcl in self.browse(cr, uid, ids, context=context):
            # get participation
            part = tpcl.participation_id
            if part.id not in result_per_part:
                result_per_part[part.id] = {}
            else:
                ddict = result_per_part[part.id]
                if tpcl.course_id.id in ddict:
                    res[tpcl.id] = ddict[tpcl.course_id.id]
                continue

            # get courses
            courses = set()
            for cl in part.course_line_ids:
                courses.add(cl.course_id.id)
            courses_per_part[part.id] = courses

            ddict = result_per_part[part.id]
            for respline in part.participation_line_ids:
                for course in respline.question_id.course_ids:
                    if course.id in courses:
                        if course.id not in ddict:
                            ddict[course.id] = 0.0
                        ddict[course.id] += respline.point
            if tpcl.course_id.id in ddict:
                res[tpcl.id] = ddict[tpcl.course_id.id]
        return res

    def f_is_succeeded(self, cr, uid, ids, fieldnames, args, context=None):
        res = dict.fromkeys(ids, 'n/a')
        for tpcl in self.browse(cr, uid, ids, context=context):
            if tpcl.passing_score >= 0.0001:
                res[tpcl.id] = ['no', 'yes'][(tpcl.score >= tpcl.passing_score)]
        return res

    _columns= {
        'participation_id': fields.many2one('training.participation', 'Participation', required=True, ondelete='cascade', select=True),
        'course_id': fields.many2one('training.course', 'Course', required=True, readonly=True, select=True),
        'passing_score': fields.float('Passing Score', required=True),
        'score': fields.function(f_get_score, method=True, type='float', required=True, string='Score', readonly=True),
        'max_score': fields.float('Max Score', required=True),
        'succeeded': fields.function(f_is_succeeded, method=True, type='selection',
                                    selection=[('n/a','N/A'),('no','No'),('yes','Yes')], string='Succeeded'),
        'confirmed_by_assessor': fields.boolean('Confirmed by Assessor'),
    }

training_participation_course_line()

class training_participation_exam(osv.osv):
    _inherit = 'training.participation'

    def copy_data(self, cr, uid, object_id, values=None, context=None):
        if not values:
            values = {}
        if 'course_line_ids' not in values:
            values.update({'course_line_ids' : []})
        if 'participation_line_ids' not in values:
            values.update({'participation_line_ids' : []})
        return super(training_participation_exam, self).copy_data(cr, uid, object_id, values, context=context)

    def after_exam_sheet_generation(self, cr, uid, participation_id, context=None):
        if context is None:
            context = {}
        # Need to register questions printed on PDF into corresponding training.participation record
        qn = 1 # question number
        qset = set() # question id set
        qdict = {}

        tpline = self.pool.get('training.participation.line')
        tpline_current_ids = tpline.search(cr, uid, [('participation_id','=',participation_id)], context=context)
        if len(tpline_current_ids) != 0:
            raise osv.except_osv("Error", "Can't generate an exam sheet on participation which already have participation lines")

        participation = self.browse(cr, uid, participation_id, context=context)
        if participation and participation.questionnaire_id:
            for qnum, question_line in enumerate(participation.questionnaire_id.question_ids):
                new_tpline = {
                    'sequence': qnum+1,
                    'participation_id': participation_id,
                    'question_id': question_line.question_id.id,
                    'graded': False,
                }
                new_id = tpline.create(cr, uid, new_tpline, context=context)
                qdict[str(question_line.question_id.id)] = new_id

        for page_num, page in enumerate(context.get('checkboxes_context',False)):
            if not len(page):
                continue
            for qanswer in page:
                qid, aid = qanswer.split('-')
                new_tpline = {
                    'page_num': page_num,
                    'participation_id': participation_id,
                    'question_id': qid,
                    'graded': False
                }
                if qid not in qdict:
                    new_qn = qn
                    qn += 1
                    new_tpline.update({
                        'sequence': new_qn,
                    })
                    new_id = tpline.create(cr, uid, new_tpline, context=context)
                    qdict[qid] = new_id
                else:
                    tpline.write(cr, uid, qdict[qid], new_tpline, context=context)
        return True

    def _result_compute(self, cr, uid, ids, fieldnames, args, context=None):
        res = dict.fromkeys(ids, 0.0)
        for p in self.browse(cr, uid, ids, context=context):
            if p.forced_result >= -0.001 and p.forced_result <= 0.001:
                # Result not forced
                if p.participation_line_ids:
                    res[p.id] = reduce(lambda acc,x: acc + x.point, p.participation_line_ids, 0.0)
            else:
                # Result is forced, return that value
                res[p.id] = p.forced_result
        return res

    def _all_pourcentage_compute(self, cr, uid, ids, fieldnames, args, context=None):
        res = dict.fromkeys(ids, 0.0)
        for p in self.read(cr, uid, ids, ['result','passing_score', 'total_points'], context=context):
            if p['total_points'] <= 0.0:
                res[p['id']] = {
                    'result_pourcentage': 100.0,
                    'passing_score_pourcentage': p['passing_score'],
                }
            else:
                res[p['id']] = {
                    'result_pourcentage': p['result'] * 100 / p['total_points'],
                    'passing_score_pourcentage': p['passing_score'] * 100 / p['total_points'],
                }
        return res

    def _total_points_compute(self, cr, uid, ids, fieldnames, args, context=None):
        res = dict.fromkeys(ids, 0.0)
        quizz_cache = {}
        quizz_proxy = self.pool.get('training.exam.questionnaire')
        course_pool = self.pool.get('training.course')
        for p in self.read(cr, uid, ids, ['questionnaire_id', 'course_questionnaire_id'], context=context):
            quizz_id = p['questionnaire_id'] and p['questionnaire_id'][0] or None
            if quizz_id:
                if quizz_id in quizz_cache:
                    res[p['id']] = quizz_cache[quizz_id]
                else:
                    q = quizz_proxy.read(cr, uid, [quizz_id], ['total_point'])
                    totpoints = q[0]['total_point']
                    if totpoints <= 0.0:
                        totpoints = 100.0 # if no question defined on questionnaire, use 100% as default
                    res[p['id']] = totpoints
            else:
                # no quizz defined, force it to 100.0 so it's easy for percentage calculation
                forced_totpts = 100.0
                if p['course_questionnaire_id']:
                    course = course_pool.read(cr, uid, p['course_questionnaire_id'][0], ['forced_noquestionnaire', 'forced_total_points'], context=context)
                    if course.get('forced_noquestionnaire',False):
                        forced_totpts = course['forced_total_points']
                res[p['id']] = forced_totpts
        return res

    def _succeeded(self, cr, uid, ids, fn, args, context=None):
        res = dict.fromkeys(ids, 'n/a')
        for p in self.browse(cr, uid, ids, context=context):
            # if passing_score == 0, that means it's not set!
            if p.passing_score >= 0.0001:
                res[p.id] = ['no', 'yes'][(p.result >= p.passing_score)]
        return res

    def _succeeded_search(self, cr, uid, obj, name, args, context=None):
        if not args:
            return []
        # default cond = False
        search_succeeded = 'no'
        # search for condition
        for arg in args:
            if isinstance(arg, (tuple,list)) and arg[0] == 'succeeded':
                search_succeeded = arg[2]
        # get info from training.exam.contact.results.stat obj
        if search_succeeded in ['yes','no']:
            search_cond = search_succeeded == 'yes' and True or False
            query = ("SELECT DISTINCT(participation_id) "
                     "FROM training_exam_contact_results_stat "
                     "WHERE succeeded = %s")
            cr.execute(query, (search_cond,))
        else:
            # searching for N/A participations
            query = ("SELECT tp.id "
                     "FROM training_participation tp "
                     "LEFT JOIN training_exam_contact_results_stat st "
                     "  ON (st.participation_id = tp.id) "
                     "GROUP BY tp.id "
                     "HAVING count(st.course_id) < 1")
            cr.execute(query)
        search_ids = [ x[0] for x in cr.fetchall() ]
        return [('id', 'in', search_ids)]

    def _certif_printed(self, cr, uid, ids, name, args, context=None):
        res = dict.fromkeys(ids, 0)
        q = """
            SELECT  res_id, count(1)
            FROM    ir_attachment
            WHERE   res_id in (%s)
              AND   res_model = 'training.participation'
              AND   name like 'CERTIF%%'
            GROUP
                BY  res_id
            """ % (','.join(map(str, map(int, ids))))
        cr.execute(q)
        res.update(dict([(x[0], bool(x[1])) for x in cr.fetchall()]))
        return res

    def _checkstore_update_certif(self, cr, uid, ids, context=None):
        proxy = self.pool.get('ir.attachment')
        attach_ids = proxy.search(cr, uid, [('id', 'in', ids),('res_model', '=', 'training.participation'),('name', 'like', 'CERTIF%')], context=context)
        res = set()
        for attach in proxy.browse(cr, uid, attach_ids, context=context):
            res.add(attach.res_id)
        return list(res)

    def _check_self_particip(self, cr, uid, ids, context=None):
        return ids

    def on_change_forced_result(self, cr, uid, ids, result, context=None):
        if not len(ids):
            return {'value':{}}
        ret_val = result > 0.0 and True or False
        return {'value': {'result_received': ret_val }}

    def on_change_course_questionnaire(self, cr, uid, ids, course_id, questionnaire_id, context=None):
        ocv = {'value': {}}
        course_pool = self.pool.get('training.course')
        quizz_pool = self.pool.get('training.exam.questionnaire')
        if not questionnaire_id:
            ocv['value'].update({
                'passing_score': 0.0,
                'total_points': 100.0
                })
            if course_id:
                # check if course is in mode forced no questionnaire
                course = course_pool.browse(cr, uid, course_id, context=context)
                if course.forced_noquestionnaire:
                    if course.forced_passing_score is not None:
                        ocv['value']['passing_score'] = course.forced_passing_score
                    if course.forced_total_points is not None:
                        ocv['value']['total_points'] = course.forced_total_points
        else:
            quizz = quizz_pool.browse(cr, uid, questionnaire_id, context=context)
            ocv['value'].update({
                'passing_score': quizz.passing_score,
                'total_points': quizz.total_point,
            })
        return ocv

    def on_change_questionnaire(self, cr, uid, ids, course_id, questionnaire_id, context=None):
        if not questionnaire_id:
            return {'value': {
                        'passing_score': 0.0,
                        'total_points': 100.0
                    }
            }

        quizz_proxy = self.pool.get('training.exam.questionnaire')
        quizz = quizz_proxy.browse(cr, uid, questionnaire_id, context=context)
        return {
            'value': {
                'passing_score': quizz.passing_score,
                'total_points': quizz.total_point,
            }
        }

    def random_questionnaire_get_domain(self, cr, uid, participation):
        return [('type','=','examen'),('state','in',['validated']),('main_course_id','=',participation.course_questionnaire_id.id),('printable','=',True)]

    def random_questionnaire_assign(self, cr, uid, ids,  context=None):
        t_quizz = self.pool.get('training.exam.questionnaire')

        for p in self.browse(cr, uid, ids,context=context):
            if p.questionnaire_id:
                # questionnaire already assigned, nothing more to do
                continue
            course = p.course_questionnaire_id
            if not course:
                raise osv.except_osv(_('Error'),
                                     _('Can\'t assign questionnaire to participation %s because there is not exam defined on this participation') % (p.id))

            # combinate quizz relate to "course", and standard filter
            domain = self.random_questionnaire_get_domain(cr, uid, p)

            # search for quizz matching the computated domain
            quizz_ids = t_quizz.search(cr, uid, domain)
            if not quizz_ids:
                raise osv.except_osv(_('Error'),
                                    _('Can\'t find validated questionnaire for course "%s"' % (course.name)))

            # make random choice of quizz
            final_quizz_id = random.choice(quizz_ids)
            vals = {'questionnaire_id': final_quizz_id}
            ocv = self.on_change_course_questionnaire(cr, uid, [], course.id, final_quizz_id, context=context)
            vals.update(ocv.get('value',{}))
            self.write(cr, uid, [p.id], vals)
        return True

    def _show_graded_answers(self, cr, uid, ids, fname, args, context=None):
        cr.execute("""
            SELECT
                tp.id,
                count(tpl.id) AS all_count,
                CASE WHEN count(tpl.id) = 0 THEN 0 ELSE SUM(CASE WHEN (tpl.graded = False OR tpl.graded IS NULL) THEN 1 ELSE 0 END) END AS not_graded_count,
                CASE WHEN count(tpl.id) = 0 THEN 0 ELSE SUM(CASE WHEN tpl.graded = True THEN 1 ELSE 0 END) END AS graded_count
            FROM training_participation tp
            LEFT JOIN training_participation_line tpl ON (tpl.participation_id = tp.id)
            WHERE tp.id IN %s
            GROUP BY tp.id
        """, (tuple(ids),))
        result = dict.fromkeys(ids, '')
        for sqlrow in cr.dictfetchall():
            sqlid = sqlrow['id']
            result[sqlid] = '%s / %s' % (sqlrow['graded_count'], sqlrow['all_count'])
        return result

    _columns = {
        'graded_count': fields.function(_show_graded_answers, type='char', size=64, string='Graded', method=True, readonly=True),
        'exam_id': fields.related('subscription_line_id', 'course_id', type='many2one', relation='training.course', readonly=True, string="Exam"),
        'questionnaire_id' : fields.many2one('training.exam.questionnaire', 'Questionnaire', help='Select the Questionnaire for participant'),
        'course_questionnaire_id': fields.many2one('training.course', string='Exam', select=True),
        'duration_questionnaire_id' : fields.related('questionnaire_id', 'duration', type='float', string='Duration', store=True,
                                                     readonly=True,help='Duration of selected  Questionnaire'),
        'kind_questionnaire_id': fields.related('questionnaire_id', 'kind', type='selection', selection=[('automatic','Automatic'),('manual','Manual')], string='Kind of Questionnaires', select=2, readonly=True),
        'course_line_ids': fields.one2many('training.participation.course.line', 'participation_id', 'Course Lines'),
        'participation_line_ids' : fields.one2many('training.participation.line', 'participation_id', 'Participation Lines'),
        'result_received': fields.boolean('Result received', select=True),
        'forced_result': fields.float('Forced Result', digits=(12,2), help='If not zero, this is the score that will be forced'),
        'forced_noresult': fields.boolean('Forced No Result', help='Check if this participation won\'t have any result, and that is normal. This particaption will not be taken anymore in account for correction request, exam certificate'),
        'result' : fields.function(_result_compute, method=True, string='Result', type='float',
                                   store={
                                        'training.participation': (
                                            _check_self_particip,
                                            ['forced_result','result_received','questionnaire_id','course_questionnaire_id','participation_line_ids','passing_score'],
                                            1,
                                        )
                                    },
                                   help='Exam Result of participate', select=True),
        'total_points': fields.function(_total_points_compute, method=True, string='Total Points', type='float',
                                        store = {
                                            'training.participation': (
                                                _check_self_particip,
                                                ['id', 'course_questionnaire_id', 'participation_line_ids', 'questionnaire_id'],
                                                1,
                                            ),
                                        },
                                        help='Total Point on Questionnaire', select=True,
                                        ),
        'result_pourcentage': fields.function(_all_pourcentage_compute, method=True, string='Result (%)', type='float', multi='all-percentage',
                                            store=True, group_operator='avg',
                                            help="Exam result in percentage"),
        'passing_score_pourcentage': fields.function(_all_pourcentage_compute, method=True, string='Passing Score (%)', type='float', multi='all-percentage',
                                            help="Passing Score of participation in pourcentage"),
        'passing_score': fields.float('Passing Score', help='The minimum score needed to succueed to this exam, assigned when questionnaire is affected to the participation (on "Exam Sheet" generation) and not updated after', select=True),
        'kind' : fields.related('seance_id', 'kind', type='selection', selection=[('standard', 'Course'),('exam', 'Exam')], string='Kind', select=1),
        'succeeded': fields.function(_succeeded, fnct_search=_succeeded_search, method=True, type='selection',
                                     selection=[('n/a', 'N/A'),
                                                ('no', 'No'),
                                                ('yes', 'Yes')],
                                     select=2, string='Succeeded'),
        'certif_printed' : fields.function(_certif_printed, method=True, type="boolean",
                                        store={
                                            'ir.attachment' : (_checkstore_update_certif, None, 10),
                                        }, select=2, string="Certificate Printed"),
    }

    _defaults = {
        'result_received': lambda *a: False,
        'forced_result': lambda *a: 0.0,
        'passing_score': lambda *a: 0.0,
    }

training_participation_exam()

class training_participation_line(osv.osv):
    _name = 'training.participation.line'
    _rec_name = 'sequence'

    _columns = {
        'sequence': fields.integer('Question N°'),
        'page_num': fields.integer('Page N°'),
        'participation_id' : fields.many2one('training.participation', 'Participation', required=True, ondelete="cascade", select=True),
        'question_id' : fields.many2one('training.exam.question', 'Question', ondelete='restrict', select=True),
        'question_question_id': fields.related('question_id', 'question', type='text', string='Question', readonly=True, help='Exposition of the question'),
        'point_question_id' : fields.related('question_id', 'point', type='float', string='Max Point', readonly=True,help='Point of question'),
        'type_question_id' : fields.related('question_id', 'type', type='selection', string='Type', readonly=True,
                                            selection = [('qcm', 'QCM'),
                                                         ('qcu', 'QCU'),
                                                         ('plain', 'Plain'),
                                                         ('yesno', 'Yes/No') ]),
        'yesno_question_id' : fields.related('question_id', 'response_yesno', type='char', string='Solution YesNo', readonly=True,help='Question type'),
        'plain_question_id' : fields.related('question_id', 'response_plain', type='text', string='Solution Plain', readonly=True),
        'qcm_question_id' : fields.related('question_id', 'question_answer_ids', type='one2many', relation='training.exam.question.answer',
                                           readonly=True, string='Solution QCM'),
        'graded': fields.boolean('Is Graded'),
        # Response from the user
        'response_qcm_ids' : fields.many2many('training.exam.question.answer', 'training_result_line_answer_rel',
                                        'exam_line_id', 'question_answer_id', 'Solutions', domain="[('question_id', '=', question_id)]"),
        'response_plain' : fields.text('Solution'),
        'response_yesno' : fields.selection([('yes', 'Yes'),('no', 'No')], 'Solution'),
        'point' : fields.float('Point', digits=(12,2), help='Number of point get from question', select=True),
        'note' : fields.text('Note'),
    }

    _defaults = {
        'sequence': lambda *a: 0,
        'page_num': lambda *a: 0,
    }

    def _check_score(self, cr, uid, ids, context=None):
        # check if given score do not exceed maximum points
        # TODO: currently there is no minimum limit, so we don't check that
        obj = self.browse(cr, uid, ids[0], context=context)
        question = self.pool.get('training.exam.question').browse(cr, uid, obj.question_id.id, context=context)
        if question.count_points_from == 'answers':
            question_max = 0.0
            if question.type == 'yesno':
                question_max += question.point_yesno_good_answer
            elif question.type in ('qcu','qcm'):
                for answer in question.question_answer_ids:
                    if answer.is_solution == 'yes':
                        question_max += answer.point_checked
                    else:
                        question_max += answer.point_not_checked
            else: # free text
                question_max = question.point
        else:
            question_max = question.point
        return obj.point <= question_max

    def on_change_question(self, cr, uid, ids, question_id, context=None):
        if not question_id:
            return False
        question = self.pool.get('training.exam.question').browse(cr, uid, question_id, context=context)
        return {
            'value': {
                'point_question_id': question.point,
                'type_question_id' : question.type,
            }
        }

    _constraints = [
        (_check_score, "Can you check the give point  ?", ['point']),
    ]

training_participation_line()

class training_participation_stakeholder(osv.osv):
    _inherit = 'training.participation.stakeholder'
    _columns = {
        'kind' : fields.related('seance_id', 'kind',
                                type='selection',
                                selection=[ ('standard', 'Course'), ('exam', 'Exam')],
                                string='Kind',
                                readonly=True,
                                select=1)
    }

    def _check_disponibility(self, cr, uid, ids, context=None):
        # FIXME check also the overlaps for exam sessions at different places...
        ids = [sh.id for sh in self.browse(cr, uid, ids, context=context) if sh.kind != 'exam']
        return super(training_participation_stakeholder, self)._check_disponibility(cr, uid, ids, context)

    def _default_price_compute(self, cr, uid, job, seance, product_id=None, context=None):
        if not job or not seance:
            return False
        if isinstance(seance, (int, long)):
            seance = self.pool.get('training.seance').browse(cr, uid, seance, context=context)

        if seance.kind != 'exam':
            return super(training_participation_stakeholder, self)._default_price_compute(cr, uid, job, seance, product_id=product_id, context=None)
        if product_id and isinstance(product_id, (int,long)):
            product = self.pool.get('product.product').browse(cr, uid, product_id, context=context)
        else:
            product = seance._get_product()
        if not product:
            raise osv.except_osv(_('Error'),
                                 _("The type of the course (%s) of this seance has no product defined") % course.name)
        return product.standard_price * seance.duration

training_participation_stakeholder()

class training_subscription_line_second(osv.osv):
    _inherit = 'training.subscription.line.second'

    _columns = {
        'exam_session_id' : fields.many2one('training.session', 'Exam Session'),
        'course_id' : fields.many2one('training.course', 'Exam',
                                      domain="[('state_course', '=', 'validated')]"),
    }

    def _create_from_wizard(self, cr, uid, the_wizard, job, subscription_line_mass, context=None):
        proxy = self.pool.get('training.subscription.line.second')
        return proxy.create(cr, uid,
                            {
                                'job_id' : job.id,
                                'session_id' : subscription_line_mass.session_id.id,
                                'exam_session_id' : subscription_line_mass.exam_session_id.id,
                                'course_id' : subscription_line_mass.course_id and subscription_line_mass.course_id.id,
                            },
                            context=context)

training_subscription_line_second()

class training_participation(osv.osv):
    _inherit = 'training.participation'

training_participation()


class training_email(osv.osv):
    _inherit = 'training.email'

    def _get_lang(self, session, seance, **objects):
        DEFAULT_LANG = 'en_US'
        lng = None
        subline = objects.get('subline', None)
        if (session and session.kind == 'exam') or (seance and seance.kind == 'exam'):
            if not subline:
                return DEFAULT_LANG
            return subline.course_id.lang_id.code
        return super(training_email, self)._get_lang(session, seance, **objects)

training_email()

class training_exam_questionnaire_attachment(osv.osv):
    _name = 'training.exam.questionnaire.attachment'
    _description = 'training.exam.questionnaire.attachment'
    _order = 'position DESC, sequence ASC'

    _columns = {
        'questionnaire_id': fields.many2one('training.exam.questionnaire', 'Questionnaire', required=True, ondelete='cascade'),
        'attachment_id': fields.many2one('ir.attachment', 'Attachment', required=True,),
        'position': fields.selection([('before','Before'),('after','After')], 'Position', required=True,),
        'sequence': fields.integer('Sequence', ),
        'store_in_dms': fields.boolean('Store In DMS', help='Indicate if generated document will be store is the document management system or discared on scan'),
    }

    _defaults = {
        'position': lambda *a: 'after',
        'sequence': lambda *a: 0,
        'store_in_dms': lambda *a: False,
    }

training_exam_questionnaire_attachment()

class training_exam_contact_results_stat(osv.osv):
    _name = 'training.exam.contact.results.stat'
    _rec_name = 'participation_id'
    _auto = False

    _columns = {
        'participation_id': fields.many2one('training.participation', 'Participation', readonly=True,),
        'contact_id': fields.many2one('res.partner.contact', 'Contact', readonly=True,),
        'course_id': fields.many2one('training.course', 'Course', readonly=True,),
        'points': fields.float('Points', readonly=True,),
        'passing_score': fields.float('Passing Score', readonly=True,),
        'max_points': fields.float('Max Points', readonly=True,),
        'succeeded': fields.boolean('Succeeded', readonly=True,),
    }

    def init(self, cr):
        tools.drop_view_if_exists(cr, 'training_exam_contact_results_stat')
        cr.execute("""CREATE OR REPLACE VIEW training_exam_contact_results_stat AS (
            SELECT
                tp.id AS participation_id,
                rpj.contact_id AS contact_id,
                COALESCE(tpcl.course_id, tp.course_questionnaire_id) AS course_id,
                COALESCE(z.points, tp.result) AS points,
                COALESCE(tpcl.passing_score, tp.passing_score) AS passing_score,
                COALESCE(z.max_points, tp.total_points) AS max_points,
                (COALESCE(z.points, tp.result) >= COALESCE(tpcl.passing_score, tp.passing_score)) AS succeeded
            FROM
                training_participation tp
            LEFT JOIN
                res_partner_job rpj ON tp.job_id = rpj.id
            LEFT JOIN
                training_participation_course_line tpcl ON tpcl.participation_id = tp.id
            LEFT JOIN (
                    SELECT
                        tpl.participation_id,
                        tc.id,
                        sum(tpl.point) AS points,
                        sum(tq.point) AS max_points
                    FROM
                        training_participation_line tpl
                    LEFT JOIN
                        training_exam_question tq ON tpl.question_id = tq.id
                    LEFT JOIN
                        training_question_course_rel tqcr ON tqcr.question_id = tq.id
                    LEFT JOIN
                        training_course tc ON tqcr.course_id = tc.id
                    GROUP BY
                        tc.id,
                        tpl.participation_id
                ) AS z ON tpcl.course_id = z.id AND tp.id = z.participation_id
        )""")

training_exam_contact_results_stat()

# vim:expandtab:smartindent:tabstop=4:softtabstop=4:shiftwidth=4:
