# -*- encoding: utf-8 -*-
############################################################################################
#
#    OpenERP, Open Source Management Solution
#    Copyright (C) 2004-2009 Tiny SPRL (<http://tiny.be>). All Rights Reserved
#    Copyright (C) 2008-2009 AJM Technologies S.A. (<http://www.ajm.lu). All Rights Reserved
#    Copyright (C) 2010-2011 Thamini S.à.R.L (<http://www.thamini.com>). All Rights Reserved
#    $Id$
#
#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
############################################################################################

import time
from datetime import datetime
from dateutil.relativedelta import relativedelta
from dateutil import rrule

from osv import osv
from osv import fields
from tools.translate import _
from tools.misc import DEFAULT_SERVER_DATE_FORMAT as DT_FORMAT

class holiday_year(osv.osv):
    _name = 'training.holiday.year'
    _rec_name = 'year'
    _columns = {
        'year' : fields.integer('Year', select=1, required=True),
        'period_ids' : fields.one2many('training.holiday.period', 'year_id', 'Holiday Periods'),
    }
    _defaults = {
        'year' : lambda *a: datetime.today().year,
    }
    _sql_constraints = [
        ('uniq_year', 'unique(year)', 'The year must be unique !'),
    ]

holiday_year()

class holiday_period(osv.osv):
    _name = 'training.holiday.period'
    _columns = {
        'year_id' : fields.many2one('training.holiday.year', 'Year', required=True, ondelete='cascade'),
        'name' : fields.char('Name', size=64, required=True),
        'date_start' : fields.date('Date Start', required=True),
        'date_stop' : fields.date('Date Stop', required=True),
        'active' : fields.boolean('Active'),
        'contact_id' : fields.many2one('res.partner.contact', 'Contact'),
    }
    _defaults = {
        'active' : lambda *a: 1,
        'date_start' : lambda *a: time.strftime(DT_FORMAT),
        'date_stop' : lambda *a: time.strftime(DT_FORMAT),
    }

    def _check_date_start_stop(self, cr, uid, ids, context=None):
        if not ids:
            return False
        obj = self.browse(cr, uid, ids[0], context=context)
        return obj.date_start <= obj.date_stop

    def is_in_period(self, cr, date):
        if not date:
            raise osv.except_osv(_('Error'),
                                 _('No date specified for \'Is in period\' holiday period check'))
        cr.execute("SELECT count(id) "
                   "FROM training_holiday_period "
                   "WHERE %s BETWEEN date_start AND date_stop AND active='1'",
                   (date,))
        return cr.fetchone()[0] > 0

    _constraints = [
        (_check_date_start_stop, "Please, check the start date !", ['date_start', 'date_stop']),
    ]

holiday_period()

class holiday_year_wizard(osv.osv):
    _name = 'training.holiday.year.wizard'
    _columns = {
        'year' : fields.integer('Year', required=True),
    }
    _defaults = {
        'year' : lambda *a: datetime.today().year,
    }

    def action_cancel(self, cr, uid, ids, context=None):
        return {'type': 'ir.actions.act_window_close'}

    def action_apply(self, cr, uid, ids, context=None):
        if not ids:
            return False
        holiday_year_obj = self.pool.get('training.holiday.year')
        holiday_period_obj = self.pool.get('training.holiday.period')

        wizard = self.browse(cr, uid, ids[0], context=context)
        year_start = datetime.strptime('%04s-01-01' % (wizard.year,), DT_FORMAT)
        year_end = datetime.strptime('%04s-12-31' % (wizard.year,), DT_FORMAT)
        year_id = holiday_year_obj.create(cr, uid, {'year' : wizard.year}, context=context)

        # Generate holiday periods for each week-end of requested year
        # NOTE: we use ISO week number, but if the 1st saturday of the
        #       year is before the 1st thursday we force week-num to 0
        year_rule = rrule.rrule(rrule.DAILY, dtstart=year_start, until=year_end, byweekday=(rrule.SA))
        for saturday in year_rule:
            iso_year, iso_weeknum, iso_weekday = saturday.isocalendar()
            weeknum = iso_year == wizard.year and iso_weeknum or 0
            holiday_period_obj.create(cr, uid, {
                'year_id' : year_id,
                'date_start' : saturday.strftime(DT_FORMAT),
                'date_stop' : (saturday+relativedelta(days=1)).strftime(DT_FORMAT),
                'name' : _('Week-End %02d') % (weeknum,)
            }, context=context),

        return {
            'view_type': 'form',
            'view_mode': 'form',
            'res_model': 'training.holiday.year',
            'type': 'ir.actions.act_window',
            'target': 'current',
            'res_id' : year_id,
        }

holiday_year_wizard()

# vim:expandtab:smartindent:tabstop=4:softtabstop=4:shiftwidth=4:
