# -*- encoding: utf-8 -*-
##############################################################################
#
#    training module for OpenERP, Training Management
#    Copyright (C) 2010 Thamini S.à.R.L (<http://www.thamini.com>) Xavier ALT
#
#    This file is a part of training
#
#    training is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    training is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
##############################################################################

from osv import osv
from osv import fields

import time
from tools.translate import _
import decimal_precision as dp

content_review_type = [
    ('other_material', 'Other Material'),
    ('course_material', 'Course Material'),
    ('', ''),
]

class training_content_review_line_advance(osv.osv):
    _name = 'training.content.review.line'
training_content_review_line_advance()

class training_content_review_writer_advance(osv.osv):
    _name = 'training.content.review.writer'
training_content_review_writer_advance()

class training_content_review_purchase_line(osv.osv):
    _name = 'training.content.review.purchase.line'
training_content_review_purchase_line()

class training_content_review_reason(osv.osv):
    _name = 'training.content.review.reason'

    def _get_type_selection(self, cr, uid, context=None):
        return content_review_type

    _columns = {
        'type': fields.selection(_get_type_selection, 'Type', required=True, select=True),
        'code' : fields.char('Code', size=32, required=True, select=True),
        'name' : fields.char('Name', size=32, translate=True, required=True, select=True),
        'to_invoice': fields.boolean('To Invoice', help='If checked, this means that this type should normally be invoiced', select=True),
    }

    _defaults = {
        'type': lambda *a: 'course_material',
    }

    _sql_constraints = [
        ('uniq_code', 'unique(code)', "Must be unique"),
    ]

training_content_review_reason()

class training_content_review(osv.osv):
    _STATES = [
        ('draft', 'Draft'),
        ('validated', 'Validated'),
        ('inprogress', 'In Progress'),
        ('done', 'Done'),
        ('cancelled', 'Cancelled'),
    ]

    def _get_type_selection(self, cr, uid, context=None):
        return content_review_type

    def _seance_next_date_query_get(self, cr, uid, review_record, context=None):
        """
        Return the SQL query to get the next seance
        date for the current 'review_record' object
        @return string (the SQL query with)
        """
        if review_record.reason_id.type == 'course_material':
            return """
                SELECT tcr.id,min(tsea.date) AS min_ready_date
                FROM training_content_review tcr
                LEFT JOIN training_content_review_reason tcrr ON (tcr.reason_id = tcrr.id)
                LEFT JOIN training_seance tsea ON (tcr.course_id = tsea.course_id)
                LEFT JOIN training_session_seance_rel tssr ON (tssr.seance_id = tsea.id)
                LEFT JOIN training_session ts ON (tssr.session_id = ts.id)
                WHERE tcrr.type = 'course_material'
                  AND tsea.date >= %s
                  AND ts.state in ('opened', 'opened_confirmed', 'inprogress', 'closed_confirmed')
                  AND tcr.id = %s
                GROUP BY tcr.id
                ORDER BY tcr.id
                """
        if review_record.reason_id.type == 'other_material':
            return ''
        # if type is unknown
        raise osv.except_osv(_("Can't compute seance_next_date field with unknown review type"))

    def _seance_next_date_compute(self, cr, uid, ids, fieldnames, args, context=None):
        res = dict.fromkeys(ids, '') #time.strftime('%Y-%m-%d %H:%M:%S'))
        time_cmp = time.strftime('%Y-%m-%d 00:00:00')

        for obj in self.browse(cr, uid, ids, context=context):
            value = False
            q = self._seance_next_date_query_get(cr, uid, obj, context=context)
            if q:
                cr.execute(q, (time_cmp, obj.id,))
                values = cr.dictfetchone()
                if values:
                    value = values['min_ready_date']

            if value:
                res[obj.id] = value

        return res

    def _amount_total(self, cr, uid, ids, fieldname, args, context=None):
        res = dict.fromkeys(ids, 0.0)
        for review in self.browse(cr, uid, ids, context=context):
            total = 0.0
            for line in review.purchase_line_ids:
                total += line.price
            res[review.id] = total
        return res

    def create_invoice(self, cr, uid, ids, context=None):
        inv_pool = self.pool.get('account.invoice')
        jrnl_pool = self.pool.get('account.journal')
        adist_pool = self.pool.get('account.analytic.plan.instance')

        for review in self.browse(cr, uid, ids, context=context):
            if review.invoice_id:
                # refuse to create another invoice is one is currently set
                continue
            if not review.partner_id or not review.address_id:
                raise osv.except_osv(_('Error !'),_("Can you please check Partner and Address assign before create invoice"))

            inv_vals = inv_pool.default_get(cr, uid,
                    ['type', 'currency_id', 'journal_id', 'company_id'],
                    {'type': 'in_invoice'})
            inv_journal = jrnl_pool.browse(cr, uid, inv_vals['journal_id'], context=context)
            inv_vals.update({
                'name': _('[REVIEW] %s') % (review.ref),
                'origin': _('[REVIEW] %s') % (review.ref),
                'reference': _('[REVIEW] %s') % (review.ref),
                'partner_id': review.partner_id.id,
                'address_contact_id': review.address_id.id,
                'address_invoice_id': review.address_id.id,
                'date_invoice': time.strftime('%Y-%m-%d'),
                'invoice_line': [],
            })
            inv_ocv = inv_pool.onchange_partner_id(cr, uid, None, 'in_invoice', review.partner_id.id)
            if inv_ocv and inv_ocv.get('value', None):
                for f in ['account_id', 'payment_term', 'fiscal_position']:
                    inv_vals[f] = inv_ocv['value'][f]

            course_analytic_id = review.course_id.analytic_account_id.id
            for line in review.purchase_line_ids:
                aid = adist_pool.create(cr, uid, {
                        'journal_id': inv_journal.analytic_journal_id.id,
                        'account_ids': [
                            (0, 0, {'analytic_account_id': course_analytic_id, 'rate': 100.0}),
                        ]
                        }, context=context)
                invline = {
                    'name': line.description,
                    'uos_id': line.product_uom.id,
                    'product_id': line.product_id.id,
                    'invoice_line_tax_id': [(6, 0, [ x.id for x in line.tax_ids ])],
                    'account_id': line.account_id.id,
                    'price_unit': line.unit_price,
                    'quantity': line.product_qty,
                    'analytics_id': aid,
                }
                inv_vals['invoice_line'].append((0, 0, invline))
            inv_id = inv_pool.create(cr, uid, inv_vals, context=context)
            inv_pool.button_compute(cr, uid, [inv_id], context=context, set_total=True)

            self.write(cr, uid, [review.id], {'invoice_id': inv_id}, context=context)
        return True

    _name = 'training.content.review'
    _rec_name = 'ref'
    _order = 'delivery_date ASC'
    _columns = {
        'ref': fields.char('Reference', size=32, required=True, readonly=True, select=True),
        'followup_by': fields.many2one('res.users', 'Followup By', required=True, select=True),
        'reason_id': fields.many2one('training.content.review.reason', 'Reason', required=True, select=True),
        'reason_to_invoice': fields.related('reason_id', 'to_invoice', type='boolean', string='To Invoice', readonly=True),
        'reason_text': fields.text('Note'),
        'create_date': fields.datetime('Create Date', readonly=True),
        'validation_date': fields.datetime('Validation Date', readonly=True, select=True),
        'contract_date': fields.date('Contract Date'),
        'delivery_date': fields.date('Delivery Date', select=True),
        'estimated_delivery_date': fields.date('Estimated Delivery Date', select=2),
        'course_id': fields.many2one('training.course', 'Course', required=True, select=True),
        'line_ids': fields.one2many('training.content.review.line', 'review_id', 'Documents'),
        'purchase_line_ids': fields.one2many('training.content.review.purchase.line', 'review_id', 'Purchase Lines'),
        'writer_ids': fields.one2many('training.content.review.writer', 'review_id', 'Writers'),
        #'amount_total': fields.float('Total Amount', required=True),
        'amount_total': fields.function(_amount_total, type='float', string='Total', method=True),
        'invoice_id': fields.many2one('account.invoice', 'Invoice', readonly=True, select=2),
        'type': fields.related('reason_id', 'type', type='selection', selection=_get_type_selection, string='Type', readonly=True, select=2),
        'state': fields.selection(_STATES, 'State', required=True, readonly=True, select=1),
        'seance_next_date' : fields.function(_seance_next_date_compute, method=True, string='Seance Next Date', type='datetime'),
        'partner_id': fields.many2one('res.partner', 'Partner', select=True),
        'address_id': fields.many2one('res.partner.address', 'Address'),
        'team_id': fields.many2one('res.partner.team', 'Team', select=2),
    }

    _defaults = {
        'type': lambda *a: '',
        'state': lambda *a: 'draft',
        'amount_total': lambda *a: 0.0,
        'ref': lambda self, cr, uid, context: self.pool.get('ir.sequence').get(cr, uid, 'training.content.review'),

    }

    def onchange_reason_id(self, cr, uid, ids, reason_id, context=None):
        result = {'value': {'type': '', 'reason_to_invoice': True}}
        reason_pool = self.pool.get('training.content.review.reason')
        if reason_id:
            reason = reason_pool.browse(cr, uid, reason_id, context=context)
            result['value'].update({
                        'type' : reason.type,
                        'reason_to_invoice' : reason.to_invoice
                        })
        return result

    def onchange_delivery_date(self, cr, uid, ids, delivery_date, context=None):
        result = {'value': {}}
        if not delivery_date:
            return result
        result['value'].update({'estimated_delivery_date' : delivery_date})
        return result

    def onchange_partner_id(self, cr, uid, ids, partner_id, context=None):
        result = {'value': {'address_id' : False}}
        if partner_id:
            partner = self.pool.get('res.partner').browse(cr, uid, partner_id, context=context)
            result['value'].update({'address_id' : partner.address and partner.address[0].id or False})
        return result

    def action_draft(self, cr, uid, ids, context=None):
        return self.write(cr, uid, ids, {'state': 'draft'}, context=context)

    def action_cancel(self, cr, uid, ids, context=None):
        return self.write(cr, uid, ids, {'state': 'cancelled'}, context=context)

    def action_validate(self, cr, uid, ids, context=None):
        if not ids:
            return False
        vals = {
            'state': 'validated',
            'validation_date': time.strftime('%Y-%m-%d %H:%M:%S'),
        }
        for review in self.browse(cr, uid, ids, context=context):
            if review.delivery_date:
                vals.update({'estimated_delivery_date' : review.delivery_date})
            result = self.write(cr, uid, [review.id], vals, context=context)
        return result

    def action_inprogress(self, cr, uid, ids, context=None):
        if not ids:
            return False
        vals = {
            'state': 'inprogress',
        }
        for review in self.browse(cr, uid, ids, context=context):
            if review.delivery_date:
                vals.update({'estimated_delivery_date' : review.delivery_date})
            result = self.write(cr, uid, [review.id], vals, context=context)
        return result

    def action_done(self, cr, uid, ids, context=None):
        return self.write(cr, uid, ids, {'state': 'done'}, context=context)

training_content_review()

class training_content_review_line(osv.osv):
    _name = 'training.content.review.line'

    def _get_type_selection(self, cr, uid, context=None):
        return content_review_type

    def _get_document_name_by_type(self, cr, uid, review_line, context=None):
        if review_line.review_id.type == 'course_material':
            material = review_line.course_material_id
            return material and material.name_get()[0][1] or ''
        return ''

    def _get_document_name(self, cr, uid, ids, fname, args, context=None):
        res = dict.fromkeys(ids, '')
        for revline in self.browse(cr, uid, ids, context=context):
            res[revline.id] = self._get_document_name_by_type(cr, uid, revline, context=context)
        return res

    _columns = {
        'review_id': fields.many2one('training.content.review', 'Review', required=True),
        'type': fields.related('review_id', 'reason_id', 'type', type='selection', selection=_get_type_selection, string='Type', readonly=True, select=2),
        'content_name': fields.function(_get_document_name, string='Content Name', type='char', size=255, method=True),
        'course_material_id': fields.many2one('ir.attachment', 'Course Material'),
    }

training_content_review_line()

class training_content_review_line_wizard(osv.osv_memory):
    _name = 'training.content.review.line.wizard'

    def _get_type_selection(self, cr, uid, context=None):
        return content_review_type

    def _get_default_course_id(self, cr, uid, context=None):
        return False

    def _get_default_type(self, cr, uid, context=None):
        return ''

    def action_add_content(self, cr, uid, ids, context=None):
        if context is None:
            context = {}
        if not ids:
            return False
        wizard = self.browse(cr, uid, ids[0], context=context)
        active_id = context.get('active_id')

        if wizard.type == 'course_material':
            values = [ (0, 0, {'course_material_id': x.id}) for x in wizard.course_material_ids ]
        elif wizard.type == 'exam_material':
            values = [ (0, 0, {'question_id': x.id}) for x in wizard.question_ids ]
        else:
            raise osv.except_osv('Unhandled type')

        if values:
            self.pool.get('training.content.review').write(cr, uid, [ active_id ], { 'line_ids': values})
        return {'type': 'ir.actions.act_window_close'}

    _columns = {
        'course_id': fields.many2one('training.course', 'Course',),
        'type': fields.selection(_get_type_selection, 'Type', readonly=True, select=2),
        'course_material_ids': fields.many2many('ir.attachment', 'tcrl_wiz_courses', 'wiz_id', 'aid', 'Course Material'),
        'question_ids': fields.many2many('training.exam.question', 'tcrl_wiz_questions', 'wiz_id', 'qid', 'Questions'),
    }

training_content_review_line_wizard()

class training_content_review_writer(osv.osv):
    _name = 'training.content.review.writer'
    _columns = {
        'review_id': fields.many2one('training.content.review', 'Review', required=True),
        'job_id': fields.many2one('res.partner.job', 'Writer', required=True),
        'job_email': fields.char('Email', size=255),
    }

    def onchange_job_id(self, cr, uid, ids, job_id, context=None):
        result = {'value': {'job_email' : ''}}
        if job_id:
            job_pool = self.pool.get('res.partner.job')
            result['value'].update({'job_email' : job_pool.browse(cr, uid, job_id, context=context).email or False})
        return result

training_content_review_writer()

class training_content_review_purchase_line(osv.osv):
    _name = 'training.content.review.purchase.line'

    def onchange_product(self, cr, uid, ids, product_id, context=None):
        if not product_id:
            return {'value' : {'unit_price' : 0.0,
                               'product_uom' : 0}
                   }

        product = self.pool.get('product.product').browse(cr, uid, product_id)
        return {
            'value' : {
                'unit_price' : product.standard_price,
                'account_id': product.product_tmpl_id.property_account_expense.id,
                'tax_ids': [ x.id for x in product.supplier_taxes_id ],
                'product_uom' : product.uom_id.id,
                'description' : product.name,
            }
        }

    def _price_compute(self, cr, uid, ids, fieldnames, args, context=None):
        res = dict.fromkeys(ids, 0.0)
        for obj in self.browse(cr, uid, ids, context=context):
            res[obj.id] = obj.product_qty * obj.unit_price
        return res

    _columns = {
        'review_id': fields.many2one('training.content.review', 'Review', required=True),
        'product_id': fields.many2one('product.product', 'Product', select=1),
        'account_id': fields.many2one('account.account', 'Account', select=1, required=True),
        'tax_ids': fields.many2many('account.tax', 'training_content_purchase_line_taxes', 'purchase_line_id', 'tax_id', 'Taxes'),
        'description': fields.char('Description', size=128, required=True),
        'product_qty': fields.float('Quantity', required=True),
        'product_uom': fields.many2one('product.uom', 'Product UoM'),
        'unit_price': fields.float('Unit Price', required=True, digits_compute=dp.get_precision('Account')),
        'price': fields.function(_price_compute, method=True, string='Price', store=True, digits_compute=dp.get_precision('Account'), type='float'),
    }

    _defaults = {
        'product_qty': lambda *a: 1.0,
    }

training_content_review_purchase_line()

class training_course_review_inherit(osv.osv):
    _inherit = 'training.course'

    def _get_active_review(self, cr, uid, ids, type, context=None):
        review_pool = self.pool.get('training.content.review')
        domain = [('type', '=', type),('course_id', 'in', ids),('state', 'in', ['validated', 'inprogress'])]
        rev_ids = review_pool.search(cr, uid, domain, context=context)

        result = set()
        for r in review_pool.browse(cr, uid, rev_ids, context=context):
            if r.course_id.id in ids:
                result.add(r.course_id.id)
        result = dict([ (id, id in result) for id in ids ])

        return result

    def _search_active_review(self, cr, uid, type, fname, args, context=None):
        op = 'in'
        for a in args:
            if a[0] == fname:
                if int(a[2]) > 0:
                    op = 'in'
                else:
                    op = 'not in'
        review_pool = self.pool.get('training.content.review')
        domain = [('type', '=', type),('state', 'in', ['validated', 'inprogress'])]
        rev_ids = review_pool.search(cr, uid, domain, context=context)

        result = set()
        for r in review_pool.browse(cr, uid, rev_ids, context=context):
            result.add(r.course_id.id)

        return [('id', op, list(result))]

    def _get_active_course_review(self, cr, uid, ids, fn, args, context=None):
        return self._get_active_review(cr, uid, ids, 'course_material', context=context)

    def _search_active_course_review(self, cr, uid, obj, name, args, context=None):
        return self._search_active_review(cr, uid, 'course_material', 'active_course_review', args, context=context)

    def _get_active_exam_review(self, cr, uid, ids, fn, args, context=None):
        return self._get_active_review(cr, uid, ids, 'exam_material', context=context)

    def _search_active_exam_review(self, cr, uid, obj, name, args, context=None):
        return self._search_active_review(cr, uid, 'exam_material', 'active_exam_review', args, context=context)

    def _get_active_review_name(self, cr, uid, ids, type, context=None):
        review_pool = self.pool.get('training.content.review')
        domain = [('type', '=', type),('course_id', 'in', ids),('state', 'in', ['validated', 'inprogress'])]
        rev_ids = review_pool.search(cr, uid, domain, context=context)

        rev_data = {}
        for r in review_pool.browse(cr, uid, rev_ids, context=context):
            rev_data.setdefault(r.course_id.id, []).extend(r.name_get())

        names = review_pool.name_get(cr, uid, rev_ids, context=context)
        result = dict.fromkeys(ids, '')
        for k, v in rev_data.iteritems():
            result[k] = ','.join([ x[1] for x in v ])

        return result

    def _get_active_course_review_name(self, cr, uid, ids, fn, args, context=None):
        return self._get_active_review_name(cr, uid, ids, 'course_material', context=context)

    def _get_active_exam_review_name(self, cr, uid, ids, fn, args, context=None):
        return self._get_active_review_name(cr, uid, ids, 'exam_material', context=context)

    _columns = {
        'active_course_review': fields.function(_get_active_course_review, fnct_search=_search_active_course_review, type='boolean', string='Active Course Review', method=True),
        'active_course_review_name': fields.function(_get_active_course_review_name, type='text', string='Course Reviews', method=True),
        'active_exam_review': fields.function(_get_active_exam_review, fnct_search=_search_active_exam_review, type='boolean', string='Active Exam Review', method=True),
        'active_exam_review_name': fields.function(_get_active_exam_review_name, type='text', string='Exam Reviews', method=True),
    }
training_course_review_inherit()

class training_seance_review_inherit(osv.osv):
    _inherit = 'training.seance'
    _columns = {
        'active_course_review': fields.related('course_id', 'active_course_review', string='Active Course Review', type='boolean', select=1),
    }
training_seance_review_inherit()

# vim:expandtab:smartindent:tabstop=4:softtabstop=4:shiftwidth=4:
