# -*- encoding: utf-8 -*-
############################################################################################
#
#    OpenERP, Open Source Management Solution
#    Copyright (C) 2004-2009 Tiny SPRL (<http://tiny.be>). All Rights Reserved
#    Copyright (C) 2008-2009 AJM Technologies S.A. (<http://www.ajm.lu). All Rights Reserved
#    Copyright (C) 2010-2011 Thamini S.à.R.L (<http://www.thamini.com>). All Rights Reserved
#    $Id$
#
#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
############################################################################################

from osv import osv
from osv import fields

from cStringIO import StringIO
from tools.translate import _
from zipfile import PyZipFile, ZIP_DEFLATED
from functools import partial
import base64
import netsvc
import os
import re
import time
import datetime
from dateutil.relativedelta import relativedelta
import tools
import netsvc
from product import _get_product_procurement_trigger
from product import _get_product_procurement_methods
import decimal_precision as dp

WRITABLE_ONLY_IN_DRAFT = dict(readonly=True, states={'draft': [('readonly', False)]})

def _partner_from_uid(obj, cr, uid, context=None):
    user = obj.pool.get('res.users').browse(cr, uid, uid, context)
    if user and user.partner_id:
        return user.partner_id.id
    return False

class training_course(osv.osv):
    _name = 'training.course'
training_course()

class training_session(osv.osv):
    _name = 'training.session'
training_session()

class training_seance(osv.osv):
    _name = 'training.seance'
training_seance()

class training_offer(osv.osv):
    _name = 'training.offer'
training_offer()


class account_analytic_account(osv.osv):
    _inherit = 'account.analytic.account'

    _columns = {
        'name' : fields.char('Account Name', size=64, required=True),
        'course_ids': fields.one2many('training.course', 'analytic_account_id', 'Courses', readonly=True),
    }

    def copy_data(self, cr, uid, id, default=None, context=None):
        if default is None:
            default = {}
        default['course_ids'] = []
        return super(account_analytic_account, self).copy_data(cr, uid, id, default=default, context=context)

account_analytic_account()

class training_config_contact_function(osv.osv):
    _name = 'training.config.contact.function'

    def _get_values_of_kind(self):
        return [('standard', 'Course'),('writer','Writer')]

    def _get_functions_by_kind(self, cr, uid, kind, context=None):
        function_ids = self.search(cr, uid, [('kind', '=', kind)], context=context)
        result_ids = [ obj.function_id.id for obj in self.browse(cr, uid, function_ids, context=context) if obj.function_id ]
        if not result_ids:
            raise osv.except_osv(_('Error'),
                                 _("No 'Stakeholder Function' defined for kind '%s', you have to define one before continuing!"))
        return result_ids

    _rec_name = 'kind'
    _columns = {
        'kind' : fields.selection(lambda obj, cr, uid, context=None: obj._get_values_of_kind(), 'Kind', required=True),
        'function_id': fields.many2one('res.partner.function', 'Function', required=True),
    }

    _sql_constraints = [
        ('uniq_kind_function', 'unique(kind, function_id)', 'You can not define twice the same relation !'),
    ]

training_config_contact_function()

class training_course_category(osv.osv):
    _name = 'training.course_category'
    _description = 'The category of a course'
    _inherits = {
        'account.analytic.account' : 'analytic_account_id',
    }

    def _complete_name_calc(self, cr, uid, ids, fieldname, args, context=None):
        result = self.name_get(cr, uid, ids, context=context)
        return dict(result)

    _columns = {
        'complete_name': fields.function(_complete_name_calc, method=True, type='char', string='Full Name', size=255),
        'analytic_account_id' : fields.many2one('account.analytic.account', 'Analytic Account', ondelete='cascade', required=True),
        'partner_ids' : fields.many2many('res.partner',
                                         'training_purchase_line_partner_rel',
                                         'purchase_line_id',
                                         'partner_id',
                                         'Partners'),
        'price_list_id' : fields.many2one('product.pricelist', 'Pricelist', domain="[('type', '=', 'sale')]"),
        'course_ids': fields.one2many('training.course', 'parent_id', 'Child Accounts'),
        # category_code is a hack so that the field is present in 'ir.model'
        # (this is required for sync modules)
        'category_code': fields.related('analytic_account_id', 'code', type='char', size=24, string='Category Code', readonly=True),
        'category_name': fields.related('analytic_account_id', 'name', type='char', size=24, string='Category Name', readonly=True),
    }

    def name_get(self, cr, uid, ids, context=None):
        """ return name of course categories following analytic structure
            (stopping on first analytic account which is not an course cateogory)
        :param cr: database cursor
        :param user: current user id
        :type user: integer
        :param ids: list of ids
        :param context: context arguments, like lang, time zone
        :type context: dictionary
        :return: tuples with the text representation of requested objects for to-many relationships

        """
        res = []
        # precompute all analytic accounts related to a course category
        category_ids = self.search(cr, uid, [], context=context)
        category_analytic_ids = set([ c.analytic_account_id.id \
                                        for c in self.browse(cr, uid, category_ids, context=context) ])

        for record in self.browse(cr, uid, ids, context):
            acc = []
            analytic_account = record.analytic_account_id
            while analytic_account:
                if analytic_account.id not in category_analytic_ids:
                    break
                acc.insert(0, analytic_account.name)
                analytic_account = analytic_account.parent_id
            res.append((record.id, ' / '.join(acc)))
        return res

training_course_category()

class training_course_type(osv.osv):
    _name = 'training.course_type'
    _description = "The Course's Type"

    _columns = {
        'name' : fields.char('Course Type', size=32, required=True, select=1, help="The course type's name"),
        'objective' : fields.text('Objective', help="Allows to the user to write the objectives of the course type", translate=True),
        'description' : fields.text('Description', translate=True, help="Allows to the user to write the description of the course type"),
        'min_limit' : fields.integer('Minimum Threshold', required=True, help="The minimum threshold is the minimum for this type of course"),
        'max_limit' : fields.integer('Maximum Threshold', required=True, help="The maximum threshold is the maximum for this type of course"),
        'product_id' : fields.many2one('product.product', 'Lecturer'),
    }

    def _check_limits(self, cr, uid, ids, context=None):
        obj = self.browse(cr, uid, ids)[0]
        return obj.min_limit <= obj.max_limit

    _constraints = [
        (_check_limits,
         'The minimum limit is greater than the maximum limit',
         ['min_limit', 'max_limit']),
    ]

training_course_type()

class training_base_purchase_line(osv.osv):
    _name = 'training.base.purchase_line'

    def _fcalc_product_price(self, cr, uid, ids, fieldname, args, context=None):
        res = dict.fromkeys(ids, 0.0)
        for prsh_line in self.browse(cr, uid, ids, context=context):
            if prsh_line.product_id.procurement_price == 'from_attachment' \
                    and prsh_line.attachment_id:
                res[prsh_line.id] = prsh_line.attachment_id.price
            else:
                res[prsh_line.id] = prsh_line.product_id.standard_price
        return res

    _columns = {
        'product_id' : fields.many2one('product.product', 'Product', required=True, help="The product for this purchase line"),
        'product_qty' : fields.integer('Quantity', required=True, help="The quantity of this product"),
        'product_uom' : fields.many2one('product.uom', 'Product UoM', required=True, help="The unit of mesure for this product"),
        'product_price' : fields.function(_fcalc_product_price, type='float', digits_compute=dp.get_precision('Account'), string='Cost Price', method=True),
        'attachment_id' : fields.many2one('ir.attachment', 'Attachment'),
        'attachment_price' : fields.related('attachment_id', 'price', type='float', digits_compute=dp.get_precision('Account'), string='Attachment Price', readonly=True),
    }

    _defaults = {
        'product_qty' : lambda *a: 1,
    }

    def on_change_product_or_attachment(self, cr, uid, ids, product_id, attachment_id, context=None):
        ret_value = {'product_uom': 0, 'product_price': 0.0, 'attachment_price': 0.0}
        if not product_id:
            return {'value' : ret_value }

        if attachment_id:
            attachment = self.pool.get('ir.attachment').browse(cr, uid, attachment_id, context=context)
            ret_value['attachment_price'] = attachment.price

        product = self.pool.get('product.product').browse(cr, uid, product_id, context=context)
        if product.procurement_price == 'from_attachment' and attachment_id:
            ret_value['product_price'] = attachment.price
        else:
            ret_value['product_price'] = product.standard_price
        ret_value['product_uom'] = product.uom_id.id
        return { 'value' : ret_value }

training_base_purchase_line()

class training_course_purchase_line(osv.osv):
    _inherit = 'training.base.purchase_line'
    _name = 'training.course.purchase_line'
    _rec_name = 'course_id'

    _columns = {
        'course_id' : fields.many2one('training.course', 'Course', required=True,
                                      help="The course attached to this purchase line",
                                      domain="[('state_course', '=', 'validated')]",
                                      ondelete='cascade'),
    }
training_course_purchase_line()

class training_course_theme(osv.osv):
    _name ='training.course.theme'
    _description = "Course Theme"
    _order = 'nr ASC'

    def _complete_name_calc(self, cr, uid, ids, fieldname, args, context=None):
        result = self.name_get(cr, uid, ids, context=context)
        return dict(result)

    _columns = {
        'name' : fields.char('Name', size=128, required=True, select=1),
        'complete_name': fields.function(_complete_name_calc, method=True, type='char', string='Full Name', size=255),
        'active' : fields.boolean('Active'),
        'parent_id': fields.many2one('training.course.theme', 'Parent', select=1),
        'priority': fields.integer('Priority'),
        'nr': fields.integer('Nr'),
        'partner_ids': fields.many2many('res.partner',
                                      'res_partner_themes_rel',
                                      'theme_id', 'partner_id',
                                      string='Related Partners'),
    }

    _defaults = {
        'active' : lambda *a: 1,
        'priority': lambda *a: 0,
        'nr': lambda *a: 1,
    }

    def name_get(self, cr, uid, ids, context=None):
        res = []
        for theme in self.browse(cr, uid, ids, context):
            acc = []
            theme_record = theme
            while theme_record:
                acc.insert(0, theme_record.name)
                theme_record = theme_record.parent_id
            res.append((theme.id, ' / '.join(acc)))
        return res

training_course_theme()

class training_course_kind(osv.osv):
    _name = 'training.course.kind'

    _columns = {
        'code' : fields.char('Code', size=16, required=True),
        'name' : fields.char('Name', size=32, translate=True, required=True),
    }

    _sql_constraints = [
        ('uniq_code', 'unique(code)', "The code of this kind must be unique"),
    ]
training_course_kind()

def training_course_kind_compute(obj, cr, uid, context=None):
    proxy = obj.pool.get('training.course.kind')
    return [(kind.code, kind.name) for kind in proxy.browse(cr, uid, proxy.search(cr, uid, []))] or []

class training_course_offer_rel(osv.osv):
    _name = 'training.course.offer.rel'
    _rec_name = 'course_id'
    _order = 'sequence asc'
    _columns = {
        'sequence' : fields.integer('Sequence'),
        'course_id' : fields.many2one('training.course', 'Course', required=True, select=1, ondelete='cascade'),
        'offer_id' : fields.many2one('training.offer', 'Offer', required=True, select=1, ondelete='cascade'),
        'category_id' : fields.related('course_id', 'category_id', type='many2one', relation='training.course_category',  string='Product Line', readonly=True),
        'course_type_id' : fields.related('course_id', 'course_type_id', type='many2one', relation='training.course_type', string='Course Type', readonly=True),
        'lang_id' : fields.related('course_id', 'lang_id', type='many2one', relation='res.lang', string='Language', readonly=True),
        'kind': fields.related('course_id','kind',type='selection', selection=training_course_kind_compute, string='Course Kind', readonly=True),
        'duration' : fields.related('course_id', 'duration', type='float', string='Duration', readonly=True),
    }
    _sql_constraints = [
        ('uniq_course_offer', 'unique(course_id, offer_id)', "You cannot have twice courses in this offer"),
    ]

    def on_change_course(self, cr, uid, ids, course_id, context=None):
        if not course_id:
            return False
        course = self.pool.get('training.course').browse(cr, uid, course_id, context=context)
        return {
            'value' : {
                'category_id' : course.category_id.id,
                'course_type_id' : course.course_type_id.id,
                'lang_id' : course.lang_id.id,
                'duration' : course.duration,
            }
        }
training_course_offer_rel()

class training_course(osv.osv):
    _name = 'training.course'
    _description = 'Course'
    _inherits = {
        'account.analytic.account' : 'analytic_account_id'
    }

    def _total_duration_compute(self, cr, uid, ids, name, args, context=None):
        res = dict.fromkeys(ids, 0.0)
        for course in self.browse(cr, uid, ids, context=context):
            res[course.id] = reduce(lambda acc, child: acc + child.duration, course.course_ids, 0.0)
        return res

    def _has_support(self, cr, uid, ids, name, args, context=None):
        res = dict.fromkeys(ids, 0)
        cr.execute("SELECT res_id, count(1) "
                   "FROM ir_attachment "
                   "WHERE res_id in ("+ ",".join(['%s'] * len(ids)) + ")"
                   "AND res_model = 'training.course' "
                   "AND type = %s "
                   "GROUP BY res_id",
                   ids + ['course_material'],
        )
        res.update(dict([(x[0], bool(x[1])) for x in cr.fetchall()]))
        return res

    def _duration_compute(self, cr, uid, ids, name, args, context=None):
        res = dict.fromkeys(ids, 0.0)
        for course in self.browse(cr, uid, ids, context=context):
            res[course.id] = len(course.course_ids) and course.duration_with_children or course.duration_without_children
        return res

    def _with_children_compute(self, cr, uid, ids, name, args, context=None):
        res = dict.fromkeys(ids, 0.0)
        for course in self.browse(cr, uid, ids, context=context):
            res[course.id] = bool(len(course.course_ids))
        return res

    def _price_compute(self, cr, uid, ids, fieldnames, args, context=None):
        res = dict.fromkeys(ids, 0.0)
        proxy = self.pool.get('ir.attachment')
        for course in self.browse(cr, uid, ids, context=context):
            support_ids = proxy.search(cr, uid,
                                       [('res_model', '=', self._name),
                                        ('res_id', '=', course.id),
                                        ('type', '=', 'course_material')],
                                       context=context)
            price = reduce(lambda acc, support: acc + support.price,
                          proxy.browse(cr, uid, support_ids, context=context),
                           0.0)
            res[course.id] = price

        return res

    def _attachment_compute(self, cr, uid, ids, fieldnames, args, context=None):
        res = dict.fromkeys(ids, [])
        proxy = self.pool.get('ir.attachment')
        for course in self.browse(cr, uid, ids, context=context):
            res[course.id] = proxy.search(cr, uid,
                                          [('res_model', '=', self._name),
                                           ('res_id', '=', course.id),
                                           ('type', '=', 'course_material')],
                                          context=context
                                         )

        return res

    def _get_support(self, cr, uid, ids, context=None):
        proxy = self.pool.get('ir.attachment')
        attach_ids = proxy.search(cr, uid, [('id', 'in', ids),('res_model', '=', 'training.course'),('type', '=', 'course_material')], context=context)
        res = set()
        for attach in proxy.browse(cr, uid, attach_ids, context=context):
            res.add(attach.res_id)

        return list(res)

    _columns = {
        'active': fields.boolean('Active', select=True),
        'splitted_by' : fields.selection([('2', '2 Hours'),
                                          ('4', '4 Hours'),
                                          ('8', '8 Hours')
                                         ], 'Split By', required=True),
        'price' : fields.function(_price_compute, method=True, string='Price', type='float', digits_compute=dp.get_precision('Account'),
                                  #store = {
                                  #    'ir.attachment' : (_get_support, None, 10),
                                  #},
                                  help='The price of the support of the courses'),
        # seance_ids: neede for ir.rule usability!
        'seance_ids': fields.one2many('training.seance', 'course_id', 'Seances', readonly=True),
        'theme_ids' : fields.many2many('training.course.theme', 'training_course_theme_rel', 'course_id', 'theme_id', 'Theme'),
        'duration' : fields.function(_duration_compute, method=True, string='Duration', type='float', store=True, help='The duration of the course'),
        'duration_with_children':fields.function(_total_duration_compute, method=True, string='Duration', type='float',
                                        store=True, help='The duration of the course'),
        'duration_without_children':fields.float('Duration', help="The duration of the course"),
        'with_children': fields.function(_with_children_compute, method=True, string='With Children', store=True,
                                        type='boolean', help='Allows to know if the course contains some subcourses or not'),
        'p_id' : fields.many2one('training.course', 'Parent Course', help="The parent course", readonly=True, domain="[('state_course', '=', 'validated')]"),
        'course_ids' : fields.one2many('training.course', 'p_id', "Sub Courses", help="A course can be completed with some subcourses"),
        'sequence' : fields.integer('Sequence', help="The sequence can help the user to reorganize the order of the courses"),
        'reference_id' : fields.many2one('training.course', 'Master Course',
                                         help="The master course is necessary if the user wants to link certain courses together to easy the managment"),
        'child_reference_id' : fields.one2many('training.course', 'reference_id', 'Children'),
        'reference_lang_id' : fields.related('reference_id', 'lang_id', type='many2one', relation='res.lang', string="Master Course's Language", readonly=True),
        'reference_type' : fields.related('reference_id', 'course_type_id', type='many2one', relation='training.course_type', string="Master Course's Type", readonly=True),
        'analytic_account_id' : fields.many2one('account.analytic.account', 'Account', ondelete='cascade', required=True),
        'course_type_id' : fields.many2one('training.course_type', 'Type', select=1),
        'category_id': fields.many2one('training.course_category', 'Product Line', select=1),
        'lecturer_ids' : fields.many2many('res.partner.job', 'training_course_job_rel', 'course_id', 'job_id', 'Lecturers',
                                          help="The lecturers who give the course"),
        'internal_note' : fields.text('Note', help="The user can write some internal note for this course"),
        'lang_id' : fields.many2one('res.lang', 'Language', required=True, domain=[('active', '=', True), ('translatable', '=', True)],
                                    help="The language of the course"),
        'kind': fields.selection(training_course_kind_compute, 'Kind', required=True, help="The kind of course"),
        'state_course' : fields.selection([('draft', 'Draft'),
                                           ('deprecated', 'Deprecated'),
                                           ('validated', 'Validated'),
                                          ],
                                          'State',
                                          required=True,
                                          readonly=True,
                                          select=1,
                                          help="The state of the course"
                                         ),
        'offer_ids': fields.one2many('training.course.offer.rel', 'course_id', 'Offers'),
        'purchase_line_ids' : fields.one2many('training.course.purchase_line', 'course_id', 'Supplier Commands',
                                              help="The purchase line helps to create a purchase order for the seance",
                                              read=['training.group_course_manager'], write=['training.group_course_manager']),
        'has_support' : fields.function(_has_support, method=True, type="boolean",
                                        store={
                                            'ir.attachment' : (_get_support, None, 10),
                                        }, string="Has Support"),
        'forced_nosupport': fields.boolean('Forced No Support', help="This course has no attachment, and that is normal", select=2),
        'long_name' : fields.char('Long Name', size=256, help='Allows to show the long name of the course for the external view'),
        'attachment_ids' : fields.function(_attachment_compute, method=True, string='Supports of Course', type='one2many', relation='ir.attachment'),
        'attachment_note': fields.text('Support of Course Note'),
        'pending_ids': fields.one2many('training.content.review', 'course_id', 'Reviews', readonly=True),
    }

    _defaults = {
        'state_course' : lambda *a: 'draft',
        'duration_without_children' : lambda *a: 1.0,
        'splitted_by' : lambda *a: '8',
        'kind': lambda *a: 'standard',
        'active': lambda *a: True,
    }

    def _check_duration(self, cr, uid, ids, context=None):
        this = self.browse(cr, uid, ids[0], context=context)
        #return (this.duration > 0.0 or this.duration_without_children > 0.0 or this.duration_with_children > 0.0)
        return this.duration_without_children > 0.0

    _constraints = [
        (_check_duration, "Please, Can you check the duration ?", ['duration']),
    ]

    def create(self, cr, uid, values, context=None):
        if 'category_id' in values:
            proxy = self.pool.get('training.course_category')
            values['parent_id'] = proxy.browse(cr, uid, values['category_id'], context).analytic_account_id.id
        return super(training_course, self).create(cr, uid, values, context)

    def write(self, cr, uid, ids, values, context=None):
        if 'category_id' in values:
            proxy = self.pool.get('training.course_category')
            values['parent_id'] = proxy.browse(cr, uid, values['category_id'], context).analytic_account_id.id
        return super(training_course, self).write(cr, uid, ids, values, context)


    def copy_data(self, cr, uid, id, default=None, context=None):
        training_name = self.browse(cr, uid, id, context=context)
        if not default:
            default = {}
        default = default.copy()
        default.update({'name' : training_name.name + _(' (copy)'),
                        'theme_ids':[],
                        'questionnaire_ids':[],
                        'pending_ids':[],
                        'line_ids':[],
                        'purchase_line_ids':[],
                        'lecturer_ids':[],
                        'budget_intra_line_ids':[],
                        'attachment_ids':[]})
        return super(training_course, self).copy_data(cr, uid, id, default=default, context=context)

    def on_change_reference(self, cr, uid, ids, reference_id, context=None):
        if not reference_id:
            return False
        course = self.browse(cr, uid, reference_id, context=context)
        values = {
            'duration': 0.0,
            'theme_ids': [ x.id for x in course.theme_ids],
            'course_type_id': course.course_type_id.id,
            'category_id': course.category_id.id,
        }
        return {'value' : values}

    def action_workflow_validate(self, cr, uid, ids, context=None):
        return self.write(cr, uid, ids, {'state_course' : 'validated'}, context=context)

    def reset_to_draft(self, cr, uid, ids, context=None):
        workflow = netsvc.LocalService('workflow')

        for oid in ids:
            workflow.trg_create(uid, self._name, oid, cr)

        return self.write(cr, uid, ids, {'state_course': 'draft'}, context=context)

training_course()

class training_course_job_rel(osv.osv):
    """training.course.job.rel
    This object overwrite training_course_job_rel table used by
    training.course (lecturer_ids many2many field) to add
    auto-incremental 'id' field

    This is required for later correct reporting on training.course.contact
    SQL view

    NOTE: Keep that after training.course and res.partner.job objects
    """
    _name = 'training.course.job.rel'
    _auto = False
    _log_access = False
    _columns = {
        'id': fields.integer('ID', readonly=True),
        'course_id': fields.many2one('training.course', 'Course', required=True),
        'job_id': fields.many2one('res.partner.job', 'Contact function', required=True),
    }

    def _auto_init(self, cr, context=None):
        # XXX: we do not call self._field_create() as this object only exist
        #      to add 'id' field over a many2many() relationship, we do not want
        #      to have it referenced under 'ir.model'

        # check if table has id
        cr.execute("SELECT c.relname FROM pg_class c, pg_attribute a "
                   "WHERE c.relname = %s AND a.attname = %s AND c.oid = a.attrelid",
                   (self._table, 'id'))
        if not cr.rowcount:
            cr.execute('ALTER TABLE "%s" ADD COLUMN "id" SERIAL NOT NULL' % (self._table,))

training_course_job_rel()

class training_offer_public_target(osv.osv):
    _name = 'training.offer.public.target'
    _columns = {
        'name':fields.char('Name', translate=True, size=256, help="Allows to the participants to select a course whose can participate"),
        'note' : fields.text('Target Audience', translate=True),
    }

    _sql_constraints = [
        ('target_name', 'unique (name)', 'The name must be unique !')
    ]

training_offer_public_target()

class training_offer_kind(osv.osv):
    _name = 'training.offer.kind'

    _columns = {
        'code' : fields.char('Code', size=16, required=True),
        'name' : fields.char('Name', size=32, translate=True, required=True),
    }

    _sql_constraints = [
        ('uniq_code', 'unique(code)', "The code of this kind must be unique"),
    ]

training_offer_kind()

class training_offer_purchase_line_update_wizard(osv.osv_memory):
    _name = 'training.offer.purchase.line.update.wizard'
    _columns = {
        'name': fields.char('Summary', size=256),
        'log': fields.text('Log Text'),
        'date': fields.datetime('Date'),
        'state': fields.selection([('confirm','Confirm'),('update','Update')]),
    }

    _defaults = {
        'state': lambda *a: 'confirm',
    }

    def action_cancel(self, cr, uid, ids, context=None):
        return {'type': 'ir.actions.act_window_close' }

    def action_close(self, cr, uid, ids, context=None):
        return {'type': 'ir.actions.act_window_close' }

    def action_confirm(self, cr, uid, ids, context=None):
        if context is None:
            context = {}
        offer_proxy = self.pool.get('training.offer')
        offer_id = context.get('active_id', False)
        val = offer_proxy.action_update_seance_procurements(cr, uid, offer_id, context=context)
        if val:
            val['state'] = 'update'
        else:
            val = { 'name': _('FAILED')}
        return self.write(cr, uid, ids, val, context=context)

training_offer_purchase_line_update_wizard()

class training_offer_purchase_line_log(osv.osv):
    _name = 'training.offer.purchase.line.log'
    _columns = {
        'offer_id': fields.many2one('training.offer', 'Offer', required=True, ondelete='cascade'),
        'name': fields.char('Summary', size=256, required=True),
        'log': fields.text('Log Text', required=True),
        'date': fields.datetime('Date', required=True),
    }
training_offer_purchase_line_log()

class training_offer_purchase_line(osv.osv):
    _inherit = 'training.base.purchase_line'
    _name = 'training.offer.purchase.line'
    _rec_name = 'offer_id'
    _columns = {
        'offer_id' : fields.many2one('training.offer', 'Offer', required=True, ondelete='cascade'),
        'course_id' : fields.many2one('training.course', 'Course'),
        'description' : fields.char('Description', size=128),
        'fix' : fields.selection(_get_product_procurement_methods, 'Invoiced Quantity', required=True),
        'procurement_quantity' : fields.selection(_get_product_procurement_trigger, 'Trigger', required=True),
        'auto_update': fields.boolean('Auto. Update'),
    }

    _defaults = {
        'product_qty' : lambda *a: 1,
        'fix' : lambda *a: 'fix',
        'procurement_quantity' : lambda *a: 'on_first_seance',
        'auto_update': lambda *a: True,
    }

    def on_change_product_or_attachment(self, cr, uid, ids, product_id, attachment_id, context=None):
        super_val = super(training_offer_purchase_line, self).on_change_product_or_attachment(cr, uid, ids, product_id, attachment_id, context=context)
        if super_val and super_val.get('value', False):
            ret_val = super_val['value']
        else:
            ret_val = {'product_uom': 0, 'product_price': 0.0, 'attachment_price': 0.0}

        if not product_id:
            return {'value' : ret_val}

        # most part are done in training.base.purchase_line
        # we only need to care about training.offer.purchase.line specific fields
        product = self.pool.get('product.product').browse(cr, uid, product_id)
        ret_val.update({
            'fix': product.procurement_method,
            'procurement_quantity': product.procurement_trigger,
            'description': product.name,
        })

        return {'value': ret_val}

training_offer_purchase_line()


def training_offer_kind_compute(obj, cr, uid, context=None):
    proxy = obj.pool.get('training.offer.kind')
    return [(kind.code, kind.name) for kind in proxy.browse(cr, uid, proxy.search(cr, uid, []))] or []

class training_offer_category(osv.osv):
    _name = 'training.offer.category'
    _description = 'Categories'
    _columns = {
        'code': fields.char('Code', size=16, required=True, select=1),
        'name': fields.char('Category', size=128, required=False, select=1 ),
        'description': fields.text('Description'),
    }

training_offer_category()

def training_offer_category_compute(obj, cr, uid, context=None):
    proxy = obj.pool.get('training.offer.category')
    cr.execute("SELECT id FROM %s" % (proxy._table))
    ids = [ x[0] for x in cr.fetchall() ]
    r = [ (kind['code'], kind['name']) \
            for kind in proxy._read_flat(cr, uid, ids, ['code', 'name']) ]
    r.append(('',''))
    return r

class training_offer(osv.osv):
    _name = 'training.offer'
    _description = 'Offer'

    def on_change_course_ids(self, cr, uid, ids, course_ids, context=None):
        values = {
            'type_id' : 0,
            'product_line_id' : 0,
        }

        if len(course_ids) == 1:
            course = self.pool.get('training.course').browse(cr, uid, course_ids[0][2]['course_id'])
            values.update({
                'name' : course.name,
                'type_id' : course.course_type_id.id,
                'product_line_id' : course.category_id.id,
            })
        return {'value' : values}

    def _is_standalone_compute(self, cr, uid, ids, fieldnames, args, context=None):
        res = dict.fromkeys(ids, 0)
        for offer in self.browse(cr, uid, ids, context=context):
            res[offer.id] = len(offer.course_ids) == 1 and 1 or 0

        return res

    def draft_cb(self, cr, uid, ids, context=None):
        workflow = netsvc.LocalService('workflow')
        for offer_id in ids:
            workflow.trg_create(uid, 'training.offer', offer_id, cr)
        return self.write(cr, uid, ids, {'state' : 'draft'}, context=context)

    def action_load_procurements(self, cr, uid, ids, context=None):
        proxy = self.pool.get('training.offer.purchase.line')
        for offer in self.browse(cr, uid, ids, context=context):
            for course_offer_rel in offer.course_ids:
                for purchase_line in course_offer_rel.course_id.purchase_line_ids:
                    values = {
                        'offer_id' : offer.id,
                        'product_id' : purchase_line.product_id.id,
                        'product_qty' : purchase_line.product_qty,
                        'product_uom' : purchase_line.product_uom.id,
                        'attachment_id' : purchase_line.attachment_id and purchase_line.attachment_id.id,
                        'course_id' : course_offer_rel.course_id.id,
                        'procurement_quantity' : 'on_first_seance',
                        'fix' : purchase_line.product_id.procurement_method,
                    }
                    proxy.create(cr, uid, values, context=context)

        return True

    def action_update_seance_procurements(self, cr, uid, ids, context=None):
        session_proxy = self.pool.get('training.session')
        seance_proxy = self.pool.get('training.seance')
        group_proxy = self.pool.get('training.group')
        offer_pol_proxy = self.pool.get('training.offer.purchase.line')
        seance_pol_proxy = self.pool.get('training.seance.purchase_line')
        offer_id = isinstance(ids, (int,long)) and ids or ids[0]
        today = datetime.datetime.now().strftime('%Y-%m-%d 00:00:00')

        log_proxy = self.pool.get('training.offer.purchase.line.log')
        log = []
        log_seance_cnt = 0

        def create_purchase_lines(purchase_line_ids, seance):
            result = []
            proxy = self.pool.get('training.seance.purchase_line')
            for pl in purchase_line_ids:
                # only ambigious case: purchase line per course
                if pl.procurement_quantity == 'on_seance_course':
                    if not (pl.course_id and pl.course_id.id == seance['course_id'][0]):
                        # not course on purchase line, or differ from seance's one
                        continue

                if pl.attachment_id:
                    product_price = pl.attachment_price
                    description = "%s (%s)" % (pl.product_id.name, pl.attachment_id.datas_fname)
                else:
                    product_price = pl.product_price
                    description = pl.description or pl.product_id.name

                if pl.description:
                    description = "%s - %s" % (description, pl.description,)

                values = {
                    'seance_id' : seance['id'],
                    'course_id': pl.course_id.id,
                    'product_id' : pl.product_id.id,
                    'description' : description,
                    'product_qty' : pl.product_qty,
                    'product_uom' : pl.product_uom.id,
                    'product_price' : product_price,
                    'fix' : pl.fix,
                    'attachment_id' : pl.attachment_id and pl.attachment_id.id,
                }
                result.append((0, 0, values))
            return result

        # get offer purchase lines (auto_update = True),
        # and group them by 'procurement_quantity' (aka trigger)
        offer_pol_ids = offer_pol_proxy.search(cr, uid, [('offer_id','=',offer_id)])
        offer_pol = {} # PO lines by trigger
        for pol in offer_pol_proxy.browse(cr, uid, offer_pol_ids):
            if pol.auto_update:
                offer_pol.setdefault(pol.procurement_quantity, []).append(pol)

        futur_sessions_ids = session_proxy.search(cr, uid, [('offer_id','=',offer_id),('state', 'not in', ['closed', 'cancelled'])])
        for session in session_proxy.read(cr, uid, futur_sessions_ids, ['name', 'seance_ids'], context=context):
            log.append("SESSION: %s" % (session['name']))

            # group seance by 'group_id'
            seance_by_group = {}
            seance_ids = seance_proxy.search(cr, uid, [('id','in',session['seance_ids']),('state','!=','cancelled')], order='date')
            for seance in seance_proxy.read(cr, uid, seance_ids, ['is_first_seance','state','course_id','purchase_line_ids','group_id','date', 'name']):
                seance_by_group.setdefault(seance['group_id'], []).append(seance)

            for group, seances in seance_by_group.iteritems():
                # get remarquable idx
                first_seance_idx = 0
                last_seance_idx = len(seances) and (len(seances) - 1) or 0

                for idx, seance in enumerate(seances):
                    s_pol_updates = []

                    # get all purchase line in automatic (will be deleted)
                    pol_ids_to_del = seance_pol_proxy.search(cr, uid, [('seance_id','=',seance['id']),('auto_update','=',True)])
                    s_pol_updates.extend([(2, id) for id in pol_ids_to_del])

                    extra_info = ''
                    # compute purchase line depending on trigger
                    if idx == first_seance_idx:
                        extra_info += 'first'
                        s_pol_updates.extend(create_purchase_lines(offer_pol.get('on_first_seance', []), seance))
                    if idx == last_seance_idx:
                        extra_info += 'last'
                        s_pol_updates.extend(create_purchase_lines(offer_pol.get('on_last_seance', []), seance))
                    s_pol_updates.extend(create_purchase_lines(offer_pol.get('on_all_seances', []), seance))
                    s_pol_updates.extend(create_purchase_lines(offer_pol.get('on_seance_course', []), seance))

                    if seance['state'] != 'opened':
                        # opened = draft, after that state, seance purchase order are already existing
                        continue
                    log_seance_cnt += 1
                    log.append("    %s (%s) / %s" % (seance['name'], seance['date'], (seance['group_id'] and seance['group_id'][1] or '')))

                    seance_proxy.write(cr, uid, [seance['id']], {'purchase_line_ids': s_pol_updates})

        log_values = {
            'name': _('%d seance(s) updated') % (log_seance_cnt),
            'date': today,
            'log': u'\n'.join(log),
            'offer_id': offer_id,
        }
        new_log_id = log_proxy.create(cr, uid, log_values)
        if new_log_id:
            retval = log_values.copy()
            del retval['offer_id']
            return retval
        return None

    def _duration_compute(self, cr, uid, ids, fieldnames, args, context=None):
        res = dict.fromkeys(ids, 0.0)

        for offer in self.browse(cr, uid, ids, context=context):
            res[offer.id] = reduce(lambda acc,obj: acc + obj.course_id.duration, offer.course_ids, 0.0)

        return res

    def can_be_planned_search(self, cr, uid, context=None):
        return self.search(cr, uid, [('state', '=', 'validated')], context=context)

    def can_be_planned_compute(self, cr, uid, ids, context=None):
        return dict((obj.id, obj.state == 'validated') for obj in self.browse(cr, uid, ids, context=context))

    def _can_be_planned_search(self, cr, uid, obj, name, args, context=None):
        can_be_planned_criteria = False
        for arg in args:
            if len(arg) == 3 and arg[0] == name:
                can_be_planned_criteria = arg[2] and True or False
                break
        operator = can_be_planned_criteria and 'in' or 'not in'
        return [('id', operator, self.can_be_planned_search(cr, uid, context=context))]

    def _can_be_planned_compute(self, cr, uid, ids, fields, args, context=None):
        return self.can_be_planned_compute(cr, uid, ids, context=context)

    def is_planned_compute(self, cr, uid, ids, fname, args, context=None):
        result = dict.fromkeys(ids, False)
        cr.execute("""SELECT toff.id
                      FROM training_offer toff
                      LEFT JOIN training_session ts ON (ts.offer_id = toff.id)
                      WHERE toff.id IN %s
                      GROUP BY toff.id
                      HAVING count(ts.id) > 0""", (tuple(ids),))
        for offer_id, in cr.fetchall():
            result[offer_id] = True
        return result

    def is_planned_search(self, cr, uid, obj, name, args, context=None):
        is_planned_criteria = False
        for arg in args:
            if len(arg) == 3 and arg[0] == name:
                is_planned_criteria = arg[2] and True or False
        cond = is_planned_criteria and " > 0" or " = 0"
        cr.execute("""
            SELECT toff.id,count(ts.id)
            FROM training_offer toff
            LEFT JOIN training_session ts ON (ts.offer_id = toff.id)
            GROUP BY toff.id
            HAVING count(ts.id) %s;""" % (cond))
        res = [ x['id'] for x in cr.dictfetchall() ]
        return [('id', 'in', res)]

    def copy_data(self, cr, uid, id, default=None, context=None):
        data = super(training_offer, self).copy_data(cr, uid, id, default=default, context=context)
        if isinstance(data, dict):
            for f in ['session_ids', 'purchase_line_log_ids']:
                data.pop(f, None)
        return data

    _columns = {
        'active': fields.boolean('Active', select=2),
        'product_line_id' : fields.many2one('training.course_category', 'Product Line', select=1, required=True),

        'type_id' : fields.many2one('training.course_type', 'Type'),
        'name' : fields.char('Name', size=64, required=True, help="The name's offer"),
        'product_id' : fields.many2one('product.product', 'Product', help="An offer can be a product for invoicing"),
        'course_ids' : fields.one2many('training.course.offer.rel', 'offer_id', 'Courses', help='An offer can contain some courses'),
        'duration' : fields.function(_duration_compute, method=True, string='Duration', type='float'),
        'objective' : fields.text('Objective', help='The objective of the course will be used by the internet web site'),
        'long_name': fields.char('Long Name', size=255),
        'description' : fields.text('Description', help="Allows to write the description of the course"),
        'state' : fields.selection([('draft', 'Draft'),
                                    ('validated', 'Validated'),
                                    ('deprecated', 'Deprecated')
                                   ], 'State', required=True, readonly=True, help="The status of the course"),
        'kind' : fields.selection(training_offer_kind_compute, 'Kind', required=True),
        'target_public_ids': fields.many2many('training.offer.public.target', 'training_offer_public_target_ids', 'offer_id', 'public_target_id',
                                            string='Target Audience', select=1),
        'lang_id' : fields.many2one('res.lang', 'Language'),
        'create_date' : fields.datetime('Create Date', readonly=True),
        'preliminary_offer_ids' : fields.many2many('training.offer',
                                                   'training_offer_pre_offer_rel',
                                                   'offer_id',
                                                   'prel_offer_id',
                                                   'Preliminary Offers',
                                                   domain="[('state', '=', 'validated')]"),

        'complementary_offer_ids' : fields.many2many('training.offer',
                                                     'training_offer_cpl_offer_rel',
                                                     'offer_id',
                                                     'cpl_offer_id',
                                                     'Complementary Offers',
                                                     domain="[('state', '=', 'validated')]"),
        'is_standalone' : fields.function(_is_standalone_compute, method=True, string='Is Standalone',
                                          type="boolean", help="Allows to know if an offer is standalone or a block of courses"),
        'session_ids': fields.one2many('training.session', 'offer_id', 'Sessions', readonly=True),
        'purchase_line_ids' : fields.one2many('training.offer.purchase.line', 'offer_id', 'Procurements'),
        'purchase_line_log_ids': fields.one2many('training.offer.purchase.line.log', 'offer_id', 'Procurements Logs'),
        'theme_ids' : fields.many2many('training.course.theme', 'training_offer_them_rel', 'offer_id', 'theme_id', 'Theme'),
        'notification_note': fields.text('Notification Note', help='This note will be show on notification emails'),
        'internal_session_notif_note': fields.text('Session Notification Note (Internal)', help='This note will be shown on session confirm internal email'),
        'internal_session_notif_active': fields.boolean('Session Notification Note (Internal) Active', help='If active, a note will send on session confirmation'),
        'is_certification': fields.boolean('Is a certification?', help='Indicate is this Offer is a Certification Offer'),
        'can_be_planned' : fields.function(_can_be_planned_compute, method=True, fnct_search=_can_be_planned_search, type='boolean', string='Can Be Planned'),
        'is_planned': fields.function(is_planned_compute, method=True, fnct_search=is_planned_search, type='boolean', string='Is Planned'),
        'categorie_id': fields.selection(training_offer_category_compute, 'Category', required=False, select=1,),
    }

    _defaults = {
        'state' : lambda *a: 'draft',
        'kind' : lambda *a: 'standard',
        'is_certification': lambda *a: False,
        'active': lambda *a: True,
    }

    def action_workflow_deprecate(self, cr, uid, ids, context=None):
        return self.write(cr, uid, ids, {'state' : 'deprecated'}, context=context)

training_offer()

class training_catalog(osv.osv):
    _name = 'training.catalog'
    _description = 'Catalog'
    _columns = {
        'name' : fields.char('Title', size=64, required=True, select=1),
        'year' : fields.integer('Year', size=4, required=True, help="The year when the catalog has been published"),
        'session_ids' : fields.one2many('training.session', 'catalog_id', 'Sessions', help="The sessions in the catalog"),
        'note' : fields.text('Note', translate=True, help="Allows to write a note for the catalog"),
        'state' : fields.selection([('draft','Draft'),
                                    ('validated', 'Validated'),
                                    ('inprogress', 'In Progress'),
                                    ('deprecated', 'Deprecated'),
                                    ('cancelled','Cancelled'),
                                   ], 'State', required=True, readonly=True, help="The status of the catalog"),
    }

    _defaults = {
        'year' : lambda *a: int(time.strftime('%Y'))+1,
        'state' : lambda *a: 'draft',
    }

training_catalog()

def get_zip_from_directory(directory, b64enc=True):
    RE_exclude = re.compile('(?:^\..+\.swp$)|(?:\.py[oc]$)|(?:\.bak$)|(?:\.~.~$)', re.I)

    def _zippy(archive, path):
        path = os.path.abspath(path)
        base = os.path.basename(path)
        for f in tools.osutil.listdir(path, True):
            bf = os.path.basename(f)
            if not RE_exclude.search(bf):
                archive.write(os.path.join(path, f), os.path.join(base, f))

    archname = StringIO()
    archive = PyZipFile(archname, "w", ZIP_DEFLATED)
    archive.writepy(directory)
    _zippy(archive, directory)
    archive.close()
    val = archname.getvalue()
    archname.close()

    if b64enc:
        val = base64.encodestring(val)

    return val

class training_seance_generate_pdf_wizard(osv.osv_memory):
    _name = 'training.seance.generate.zip.wizard'

    _columns = {
        'presence_list_report' : fields.boolean('Presence List Report',
                                                help="If you select this option you will print the report for the presence list. " \
                                                "The file format is Presence_List_DATEOFSEANCE_SEANCEID.pdf"),
        'remuneration_form_report' : fields.boolean('Remuneration Form',
                                                    help="If you select this option, you will print the report for the remuneration " \
                                                    "forms of all contacts. The file format is Request_REQUESTNAME_Invoice_INVOICEID.pdf"),
        'zip_file' : fields.binary('Zip File', readonly=True),
        'zip_file_name' : fields.char('File name', readonly=True, size=64),
        'zip_error': fields.text('Errors'),
        'state' : fields.selection( [ ('selection', 'Selection'), ('result', 'Result') ], 'State', readonly=True, required=True),
    }

    _defaults = {
        'presence_list_report' : lambda *a: 0,
        'remuneration_form_report' : lambda *a: 0,
        'state' : lambda *a: 'selection',
    }

    def action_close(self, cr, uid, ids, context=None):
        return { 'type' : 'ir.actions.act_window.close' }

    def action_generate_zip(self, cr, uid, ids, context=None):
        if context is None:
            context = {}
        values = {}
        try:
            import tempfile
            parent_directory = tempfile.mkdtemp(prefix='openerp_', suffix='_reports')
            directory = os.path.join(parent_directory, 'Reports')
            import codecs
            class log_object(object):
                def __init__(self):
                    self.errors = ''
                def write(self, msg):
                    self.errors += msg
            log = log_object()
            os.mkdir(directory)
            self.add_selections(cr, uid, ids, directory, log, context=context)
            if log.errors:
                fp = codecs.open(os.path.join(directory, 'errors.txt'), encoding='utf-8', mode='w')
                fp.write(log.errors)
                fp.close()
            result = get_zip_from_directory(directory, True)
            fp = file(os.path.join(parent_directory, 'output.zip'), 'w')
            fp.write(result)
            fp.close()

            active_id = context.get('active_id', False)
            seance = self.pool.get('training.seance').browse(cr, uid, active_id, context=context)
            ts = time.strptime(seance.date, '%Y-%m-%d %H:%M:%S')
            date = time.strftime('%Y%m%d', ts)

            values.update({
                'state' : 'result',
                'zip_file' : result,
                'zip_file_name' : 'Seance_Reports_%s_%06d.zip' % (date, seance.id),
                'zip_error': log.errors,
            })
        finally:
            import shutil
            shutil.rmtree(parent_directory)
        return self.write(cr, uid, ids, values, context=context)

    def _get_report(self, cr, uid, oid, reportname, context=None):
        srv = netsvc.LocalService(reportname)
        pdf, _r = srv.create(cr, uid, [oid], {}, context=context)
        return pdf

    def add_selections(self, cr, uid, ids, directory, log, context=None):
        if context is None:
            context = {}
        active_id = context.get('active_id', False)
        seance = self.pool.get('training.seance').browse(cr, uid, active_id, context=context)
        ts = time.strptime(seance.date, '%Y-%m-%d %H:%M:%S')
        date = time.strftime('%Y%m%d', ts)
        for obj in self.browse(cr, uid, ids, context=context):
            if obj.presence_list_report:
                res = self._get_report(cr, uid, active_id, 'report.training.seance.presence.report', context=context)
                filename = os.path.join(directory, 'Presence_List_%s_%06d.pdf' % (date, seance.id,))
                fp = file(filename, 'w')
                fp.write(res)
                fp.close()

            if obj.remuneration_form_report:
                for contact in seance.contact_ids:
                    if contact.state in ('cancelled','refused'):
                        continue
                    if not contact.request_id:
                        raise osv.except_osv(_('Error'),
                                             _('The stakeholder %s %s has not a request') % (contact.job_id.fist_name, contact.job_id.name) )

                    if not contact.request_id.purchase_order_id:
                        raise osv.except_osv(_('Error'),
                                             _('There is no Purchase Order for a request'))

                    if not contact.request_id.purchase_order_id.invoice_ids:
                        raise osv.except_osv(_('Error'),
                                             _('There is no Invoice for the Purchase Order for this request'))

                    for invoice_id in contact.request_id.purchase_order_id.invoice_ids:
                        res = self._get_report(cr, uid, invoice_id.id, 'report.account.invoice', context=context)
                        filename = os.path.join(directory, 'Request_%s_Invoice_%06d.pdf' % (re.sub('/|-', '_', contact.request_id.reference),
                                                                                            invoice_id.id))
                        fp = file(filename, 'w')
                        fp.write(res)
                        fp.close()
        return True

training_seance_generate_pdf_wizard()

class training_group(osv.osv):
    _name = 'training.group'
    _description = 'Group'
    _columns = {
        'name': fields.char('Name', size=64, required=True, select=True, help="The group's name",),
        'session_id' : fields.many2one('training.session', 'Session', select=1, required=True, ondelete='cascade'),
        'seance_ids' : fields.one2many('training.seance', 'group_id', 'Seances', readonly=True),
    }

    _sql_constraints = [
        ('uniq_name_session', 'unique(name, session_id)', 'It already exists a group with this name'),
    ]

training_group()

class training_session(osv.osv):
    _name = 'training.session'
    _description = 'Session'
    _order = 'date desc, name'

    def _has_shared_seances_compute(self, cr, uid, ids, fieldnames, args, context=None):
        res = dict.fromkeys(ids, False)
        for session in self.browse(cr, uid, ids, context=context):
            res[session.id] = any(seance.shared for seance in session.seance_ids)
        return res

    # training.session
    def _store_get_participation(self, cr, uid, ids, context=None):
        result = set()
        for line in self.pool.get('training.subscription.line').browse(cr, uid, ids, context=context):
            result.add(line.session_id.id)
        return list(result)

    def _participant_count(self, cr, uid, ids, fieldnames, args, context=None):
        res = dict.fromkeys(ids, 0)
        cr.execute("""SELECT ts.id, MAX(tsea_summary.count)
                        FROM training_session ts
                   LEFT JOIN (SELECT tssr.session_id, tp.seance_id, CASE WHEN tsea.manual = True THEN tsea.participant_count_manual ELSE COUNT(DISTINCT(tsl.contact_id)) END AS count
                              FROM training_participation tp
                              LEFT JOIN training_subscription_line tsl ON (tp.subscription_line_id = tsl.id)
                              LEFT JOIN training_session_seance_rel tssr ON (tssr.seance_id = tp.seance_id)
                              LEFT JOIN training_seance tsea ON (tssr.seance_id = tsea.id)
                              WHERE tssr.session_id IN %(session_ids)s
                              AND tsl.state IN ('confirmed','done')
                              GROUP BY tp.seance_id, tsea.manual, tsea.participant_count_manual, tssr.session_id) AS tsea_summary ON (tsea_summary.session_id = ts.id)
                        WHERE ts.id IN %(session_ids)s GROUP BY ts.id
        """, {'session_ids': tuple(ids)})
        for session_id, count in cr.fetchall():
            if count is None:
                count = 0
            res[session_id] = int(count)
        return res

    # training.session
    def _confirmed_subscriptions_count(self, cr, uid, ids, fieldnames, args, context=None):
        res = dict.fromkeys(ids, 0)
        sl_proxy = self.pool.get('training.subscription.line')
        for session_id in ids:
            res[session_id] = int(sl_proxy.search_count(cr, uid, [('session_id', '=', session_id),('state', 'in', ['confirmed', 'done'])], context=context))
        return res

    # training.session
    def _available_seats_compute(self, cr, uid, ids, fieldnames, args, context=None):
        res = dict.fromkeys(ids, 0)

        for session in self.browse(cr, uid, ids, context=context):
            if session.manual:
                value_max = session.participant_count_manual
            else:
                value_max = session.participant_count

            res[session.id] = int(session.max_limit) - int(value_max)

        return res

    # training.session
    def _draft_subscriptions_count(self, cr, uid, ids, fieldnames, args, context=None):
        res = dict.fromkeys(ids, 0)

        proxy = self.pool.get("training.subscription.line")
        for session_id in ids:
            res[session_id] = int(proxy.search_count(cr, uid, [('session_id', '=', session_id),('state', '=', 'draft')], context=context))

        return res

    # training.session
    def _limit_all(self, cr, uid, ids, fieldnames, args, context=None):
        res = {}
        for obj in self.browse(cr, uid, ids, context=context):
            res[obj.id] = {'min_limit' : 0, 'max_limit' : 0}
            groups = {}
            def _add_to_group(course_id, group_id):
                if course_id not in groups:
                    groups[course_id] = set()
                groups[course_id].add(group_id)

            if len(obj.seance_ids) > 0:
                seances = iter(obj.seance_ids)
                seance = seances.next()
                if seance.group_id:
                    _c = seance.course_id and seance.course_id.id or 0
                    _add_to_group(_c, seance.group_id.id)
                value_min = seance.min_limit
                value_max = seance.max_limit

                for seance in seances:
                    if seance.group_id:
                        _c = seance.course_id and seance.course_id.id or 0
                        _add_to_group(_c, seance.group_id.id)
                    value_min = max(seance.min_limit, value_min)
                    value_max = min(seance.max_limit, value_max)

                max_groups = 0
                for v in groups.values():
                    if len(v) > max_groups:
                        max_groups = len(v)
                res[obj.id]['min_limit'] = value_min
                res[obj.id]['max_limit'] = value_max * max(max_groups, 1)

        return res

    def _min_limit_reached(self, cr, uid, ids, fn, args, context=None):
        result = dict.fromkeys(ids, False)
        for session in self.browse(cr, uid, ids, context):
            count = ['participant_count', 'participant_count_manual'][session.manual]
            result[session.id] =  int(session[count]) >= int(session.min_limit)
        return result

    # training.session
    def _store_get_seances(self, cr, uid, ids, context=None):
        values = set()
        for obj in self.pool.get('training.seance').browse(cr, uid, ids, context=context):
            for session in obj.session_ids:
                values.add(session.id)
        return list(values)

    _columns = {
        'id' : fields.integer('Seance ID', readonly=True),
        'has_shared_seances' : fields.function(_has_shared_seances_compute, method=True, type='boolean',
                                               string="Has Shared Seances", help="Allows to know if the session has a shared seance"),
        'name' : fields.char('Name', size=64, required=True),
        'state' : fields.selection([('draft', 'Draft'),
                                    ('opened', 'Opened'),
                                    ('opened_confirmed', 'Confirmed'),
                                    ('closed_confirmed', 'Closed Subscriptions'),
                                    ('inprogress', 'In Progress'),
                                    ('closed', 'Closed'),
                                    ('cancelled', 'Cancelled')],
                                   'State', required=True, readonly=True, help="The status of the session"),
        'group_ids' : fields.one2many('training.group', 'session_id', 'Group', readonly=True),
        'done' : fields.boolean('Done'),
        'offer_id' : fields.many2one('training.offer', 'Offer', required=True, help="Allows to select a validated offer for the session",
                                     domain="[('state', '=', 'validated')]"),
        'offer_product_line_id' : fields.related('offer_id', 'product_line_id', type='many2one', relation='training.course_category', string='Product Line'),
        'kind' : fields.related('offer_id', 'kind', type='selection', selection=training_offer_kind_compute, string='Kind', readonly=True),
        'catalog_id' : fields.many2one('training.catalog', 'Catalog', help="Allows to select a published catalog" ),
        'seance_ids' : fields.many2many('training.seance', 'training_session_seance_rel', 'session_id', 'seance_id', 'Seances', ondelete='cascade',
                                        help='List of the events in the session', limit=500),
        'date' : fields.datetime('Date', required=True,  help="The date of the planned session" ),
        'user_id' : fields.many2one('res.users', 'Responsible', required=True),
        'available_seats' : fields.function(_available_seats_compute, method=True, string="Available Seats", type='integer',
                                            help="Available Seats = Maximum Threshold - Total Confirmed Seats"),
        'participant_count' : fields.function(_participant_count, method=True, string='Total Confirmed Seats', type='integer'),
        'confirmed_subscriptions' : fields.function(_confirmed_subscriptions_count, method=True, string='Confirmed Subscriptions', type='integer',
                                                    #store={
                                                    #    'training.subscription.line' : (_store_get_participation, None, 10),
                                                    #}
                                                   ),
        'draft_subscriptions' : fields.function(_draft_subscriptions_count, method=True, string="Draft Subscriptions", type="integer",
                                                help="Draft Subscriptions for this session" ),
        'subscription_line_ids': fields.one2many('training.subscription.line', 'session_id', 'Subscription Lines', readonly=True),
        'participant_count_manual' : fields.integer('Manual Confirmed Seats', help="The quantity of supports, catering, ... relative to the number of participants coming from the confirmed seats"),
        'manual' : fields.boolean('Manual', help="Allows to the user to specify the number of participants"),
        'min_limit' : fields.function(_limit_all, method=True, string='Mininum Threshold', readonly=True,
                                      #store={
                                      #    'training.seance' : (_store_get_seances, None, 10),
                                      #},
                                      type='integer', multi='limit', help="The minimum threshold is the minimum of the minimum threshold of each seance"),
        'max_limit' : fields.function(_limit_all, method=True, string='Maximum Threshold', readonly=True,
                                      #store={
                                      #    'training.seance' : (_store_get_seances, None, 10),
                                      #},
                                      type='integer', multi='limit', help="The maximum threshold is the minimum of the maximum threshold of each seance"),
        'min_limit_reached': fields.function(_min_limit_reached, method=True, string='Minimum Threshold Reached', type='boolean', readonly=True,
                                             store={
                                                'training.session': (lambda self, cr, uid, ids, *a, **kw: ids, ['participant_count_manual', 'manual'], 20),
                                                'training.seance' : (_store_get_seances, None, 10),
                                                'training.subscription.line' : (_store_get_participation, None, 10),
                                             }),
        'request_ids': fields.one2many('training.participation.stakeholder.request', 'session_id', 'Requests'),
        'stylegroup_id': fields.many2one('training.email.stylegroup', 'Style Group'),
    }

    _order = "date asc"

    def _check_date_before_now(self, cr, uid, ids, context=None):
        if not ids:
            return False
        obj = self.browse(cr, uid, ids[0], context=context)
        return obj.date >= time.strftime('%Y-%m-%d %H:%M:%S')

    def _check_date_holiday(self, cr, uid, ids, context=None):
        if not ids:
            return False
        obj = self.browse(cr, uid, ids[0], context=context)
        date = time.strftime('%Y-%m-%d', time.strptime(obj.date, '%Y-%m-%d %H:%M:%S'))
        return not self.pool.get('training.holiday.period').is_in_period(cr, date)

    def _check_date_of_seances(self, cr, uid, ids, context=None):
        for session in self.browse(cr, uid, ids, context=context):
            for seance in session.seance_ids:
                if seance.date < session.date:
                    return False
        return True

    _constraints = [
        #(_check_date_before_now, "You cannot create a date before now", ['date']),
        #(_check_date_holiday, "You cannot assign a date in a public holiday", ['date']),

        (_check_date_of_seances, "You have a seance with a date inferior to the session's date", ['date']),
    ]

    def _find_catalog_id(self, cr, uid, context=None):
        ids = self.pool.get('training.catalog').search(cr, uid, [('year', '=', datetime.datetime.today().year)], context=context)
        return ids and ids[0]

    _defaults = {
        'catalog_id' : _find_catalog_id,
        'state' : lambda *a: 'draft',
        'user_id' : lambda obj,cr,uid,context: uid,
        'done' : lambda *a: 0,
        'manual' : lambda *a: 0,
        'min_limit' : lambda *a: 1,
        'max_limit' : lambda *a: 1,
    }

    def _create_seance(self, cr, uid, session, context=None):
        seance_ids = []
        seance_proxy = self.pool.get('training.seance')

        holiday_proxy = self.pool.get('training.holiday.year')

        if not holiday_proxy.search(cr, uid, [('year', '=', time.strftime('%Y'))], context=context):
            raise osv.except_osv(_('Warning'),
                                 _('Please, Can you configure the holidays ?'))

        def get_list_of_courses(lst, course):
            if course.course_ids:
                for child in course.course_ids:
                    get_list_of_courses(lst, child)
            else:
                lst.append(course)

        group_proxy = self.pool.get('training.group')
        group_ids = group_proxy.search(cr, uid, [('session_id', '=', session.id)])
        if group_ids:
            group_id = group_ids[0]
        else:
            group_id = group_proxy.create(cr, uid, {'name' : _('Class %d') % (1,), 'session_id': session.id}, context=context)

        def _inner_create_seance(item, session, date, duration, master_seance_id = None):
            values = {
                'name' : self.pool.get('training.seance').on_change_course(cr, uid, [], item.id, session.kind, context=context)['value']['name'],
                'original_session_id' : session.id,
                'course_id' : item.id,
                'kind': item.kind,
                'min_limit' : item.course_type_id.min_limit,
                'max_limit' : item.course_type_id.max_limit,
                'user_id' : session.user_id.id,
                'date' : date.strftime('%Y-%m-%d %H:%M:%S'),
                'master_id' : master_seance_id,
                'duration' : duration,
                'group_id': group_id,
            }
            if session.manual:
                values['manual'] = session.manual
                values['participant_count_manual'] = session.participant_count_manual
            return seance_proxy.create(cr, uid, values, context=context)



        planned_seance_ids = []
        planned_course_ids = set()
        if session.seance_ids:
            proxy_seance = self.pool.get('training.seance')

            for seance in session.seance_ids:
                planned_course_ids.add(seance.course_id.id)
                if seance.master_id:
                    planned_seance_ids.extend(proxy_seance.search(cr, uid, [('master_id', '=', seance.master_id.id)], context=context))
                else:
                    planned_seance_ids.append(seance.id)
                    planned_seance_ids.extend(proxy_seance.search(cr, uid, [('master_id', '=', seance.id)], context=context))

        date = datetime.datetime.strptime(session.date, '%Y-%m-%d %H:%M:%S')

        seance_counter = 0

        lst = []
        for course in session.offer_id.course_ids:
            if course.course_id.id in planned_course_ids:
                continue
            tmp_lst = []
            get_list_of_courses(tmp_lst, course.course_id)

            splitted_by = int(course.course_id.splitted_by) or 8
            for item in tmp_lst:
                duration = item.duration
                while duration > 0:
                    seance_counter += 1
                    duration -= splitted_by

            lst.extend(tmp_lst)

        sdate_tuple = time.strptime(session.date, '%Y-%m-%d %H:%M:%S')
        sdate = datetime.datetime(*sdate_tuple[:7])
        sdate_incr = datetime.timedelta(days=1)

        dates = []

        # Day by day search for a valid working day
        rec = 0
        max_rec = 365
        while len(dates) < seance_counter and rec < max_rec:
            try_session_date = sdate.strftime('%Y-%m-%d')
            cr.execute(
                "SELECT count(*) "
                "FROM training_holiday_period p "
                "WHERE date(%s) >= p.date_start "
                "  AND date(%s) <= p.date_stop ",
                (try_session_date, try_session_date))
            r = cr.fetchall()
            if len(r) == 1 and r[0][0] == 0:
                dates.append(sdate)
            sdate += sdate_incr
            rec += 1

        if not dates:
            cr.execute(
                "SELECT date(%s) + s.t AS date FROM generate_series(0,%s) AS s(t)"
                , (session.date, seance_counter+1))

            for x in cr.fetchall():
                dates.append(datetime.datetime.strptime(x[0] + " " + date.strftime('%H:%M:%S'), '%Y-%m-%d %H:%M:%S'))

        # later we will use date.pop() so we need to reverse date,
        # so that first date are a end of array, at poped first
        dates.reverse()

        def create_purchase_lines(purchase_line_ids, seance_id, procurement_quantity):
            proxy = self.pool.get('training.seance.purchase_line')
            for pl in purchase_line_ids:
                if pl.procurement_quantity == procurement_quantity:
                    if pl.attachment_id:
                        product_price = pl.attachment_price
                        description = "%s (%s)" % (pl.product_id.name, pl.attachment_id.datas_fname)
                    else:
                        product_price = pl.product_price
                        description = pl.description or pl.product_id.name

                    if pl.description:
                        description = "%s - %s" % (description, pl.description,)

                    values = {
                        'seance_id' : seance_id,
                        'course_id': pl.course_id.id,
                        'product_id' : pl.product_id.id,
                        'description' : description,
                        'product_qty' : pl.product_qty,
                        'product_uom' : pl.product_uom.id,
                        'product_price' : product_price,
                        'fix' : pl.fix,
                        'attachment_id' : pl.attachment_id and pl.attachment_id.id,
                    }

                    purchase_line_id = proxy.create(cr, uid, values, context=context)

        seance_id = None
        first_seance_id = None
        for item in lst:
            duration = item.duration
            splitted_by = int(item.splitted_by) or 8

            master_seance_id = None
            counter_part = 0
            while duration > 0:
                date = dates.pop()
                tmp_id = _inner_create_seance(item, session, date, duration <= splitted_by and duration or splitted_by, master_seance_id)

                proxy = self.pool.get('training.seance.purchase_line')
                for pl in session.offer_id.purchase_line_ids:
                    if pl.procurement_quantity == 'on_all_seances':
                        if pl.attachment_id:
                            product_price = pl.attachment_price
                            description = "%s (%s)" % (pl.product_id.name, pl.attachment_id.datas_fname)
                        else:
                            product_price = pl.product_price
                            description = pl.product_id.name

                        if pl.description:
                            description = "%s - %s" % (description, pl.description,)

                        seance_info = self.pool.get('training.seance').read(cr, uid, tmp_id, ['course_id'], context=context)
                        if seance_info:
                            pl_course_id = seance_info['course_id'] and seance_info['course_id'][0] or False
                        else:
                            pl_course_id = pl.course_id.id

                        values = {
                            'seance_id' : tmp_id,
                            'course_id': pl_course_id,
                            'product_id' : pl.product_id.id,
                            'description' : description,
                            'product_qty' : pl.product_qty,
                            'product_uom' : pl.product_uom.id,
                            'product_price' : product_price,
                            'fix' : pl.fix,
                            'attachment_id' : pl.attachment_id and pl.attachment_id.id,
                        }

                        purchase_line_id = proxy.create(cr, uid, values, context=context)

                if master_seance_id is None:
                    master_seance_id = tmp_id

                if first_seance_id is None:
                    first_seance_id = tmp_id

                seance_ids.append(tmp_id)

                duration -= splitted_by

            seance_id = master_seance_id

            if master_seance_id:
                seance = self.pool.get('training.seance').browse(cr, uid, master_seance_id, context=context)

                proxy = self.pool.get('training.seance.purchase_line')
                for pl in session.offer_id.purchase_line_ids:
                    if pl.procurement_quantity == 'on_seance_course' and pl.course_id and pl.course_id.id == seance.course_id.id:
                        if pl.attachment_id:
                            product_price = pl.attachment_price
                            description = "%s (%s)" % (pl.product_id.name, pl.attachment_id.datas_fname)
                        else:
                            product_price = pl.product_price
                            description = pl.product_id.name

                        if pl.description:
                            description = "%s - %s" % (description, pl.description,)

                        values = {
                            'seance_id' : master_seance_id,
                            'course_id': pl.course_id.id,
                            'product_id' : pl.product_id.id,
                            'description' : description,
                            'product_qty' : pl.product_qty,
                            'product_uom' : pl.product_uom.id,
                            'product_price' : product_price,
                            'fix' : pl.fix,
                            'attachment_id' : pl.attachment_id and pl.attachment_id.id,
                        }

                        purchase_line_id = proxy.create(cr, uid, values, context=context)

        if first_seance_id:
            self.pool.get('training.seance').write(cr, uid, [first_seance_id], {'is_first_seance' : 1}, context=context)
            create_purchase_lines(session.offer_id.purchase_line_ids, first_seance_id, 'on_first_seance')

        last_seance_id = len(seance_ids) > 0 and seance_ids[-1] or None
        if last_seance_id:
            create_purchase_lines(session.offer_id.purchase_line_ids, last_seance_id, 'on_last_seance')

        return list(set(seance_ids + planned_seance_ids))

    # training.session
    def action_create_seances(self, cr, uid, ids, context=None):
        for session in self.browse(cr, uid, ids, context=context):
            seance_ids = self._create_seance(cr, uid, session, context)
            self.write(cr, uid, session.id, {'seance_ids' : [(6, 0, seance_ids)]}, context=context)
        return True

    def on_change_offer(self, cr, uid, ids, offer_id, context=None):
        if not offer_id:
            return False

        offer_proxy = self.pool.get('training.offer')
        offer = offer_proxy.browse(cr, uid, offer_id, context=context)

        return {
            'value' : {
                'kind' : offer.kind,
                'name' : offer.name
            }
        }

    def on_change_date(self, cr, uid, ids, date, offer_id, context=None):
        if not ids:
            return False
        old_date = ids and self.browse(cr, uid, ids[0], context=context).date or 0

        if self.pool.get('training.holiday.period').is_in_period(cr, date):
            return {
                'value' : {
                    'date' : old_date,
                },
                'warning' : {
                    'title' : _("Selection Date"),
                    'message' : _("You can not select this date because it is a public holiday"),
                },
            }
        return {}


    # training.session
    def _create_participation(self, cr, uid, ids, subscription_line, context=None):
        proxy = self.pool.get('training.participation')
        proxy_seance = self.pool.get('training.seance')

        if subscription_line.session_id.group_ids:
            for group in subscription_line.session_id.group_ids:
                if len(group.seance_ids) > 0:
                    for seance in group.seance_ids:
                        participation_id = proxy_seance._create_participation(cr, uid, seance, subscription_line, context=context)
                        #if seance.state == 'confirmed':
                        #    proxy.create_procurements(cr, uid, [participation_id], delayed=True, context=context)
                    break
        else:
            for seance in subscription_line.session_id.seance_ids:
                participation_id = proxy_seance._create_participation(cr, uid, seance, subscription_line, context=context)
                #if seance.state == 'confirmed':
                #    proxy.create_procurements(cr, uid, [participation_id], delayed=True, context=context)
        return True

    # training.session
    def action_workflow_draft(self, cr, uid, ids, context=None):
        return self.write(cr, uid, ids, {'state' : 'draft'}, context=context)

    # training.session
    def test_workflow_open(self, cr, uid, ids, context=None):
        for obj in self.browse(cr, uid, ids, context=context):
            if not len(obj.seance_ids):
                raise osv.except_osv(_('Warning'), _("Please, do not forget to have the seances in your session"))
            else:
                min_date = obj.date
                for seance in obj.seance_ids:
                    if seance.state == 'draft':
                        raise osv.except_osv(_('Warning'), _('Please, you have at least a draft seance'))
                    else:
                        if seance.date < obj.date:
                            raise osv.except_osv(_('Warning'), _("Please, Check the date of your seances because there is one seance with a date inferior to the session's date"))

                    min_date = min(min_date, seance.date)

        return True

    # training.session
    def action_workflow_open(self, cr, uid, ids, context=None):
        return self.write(cr, uid, ids, {'state' : 'opened'}, context=context)

    # training.session
    def action_workflow_open_confirm(self, cr, uid, ids, context=None):
        email_proxy = self.pool.get('training.email')
        proxy = self.pool.get('training.subscription.line')
        subscription_line_ids = proxy.search(cr, uid, [('session_id', 'in', ids), ('state', '=', 'confirmed')], context=context)
        proxy.send_email(cr, uid, subscription_line_ids, 'session_open_confirmed', context)

        proxy = self.pool.get('training.participation.stakeholder')
        for session in self.browse(cr, uid, ids, context=context):
            objs = {}
            for seance in session.seance_ids:
                for contact in seance.contact_ids:
                    if contact.state == 'accepted':
                        objs.setdefault(contact.id, {}).setdefault('seances', []).append(seance)

            proxy.send_email(cr, uid, objs.keys(), 'session_open_confirmed', session, context, objs)
            if session.offer_id.internal_session_notif_active:
                email_proxy.send_email(cr, uid, 'session_open_confirmed', 'i', '_', session=session, context=context)

        return self.write(cr, uid, ids, {'state' : 'opened_confirmed'}, context=context)

    # training.session
    def test_workflow_open_confirm(self, cr, uid, ids, context=None):
        return True

    # training.session
    def action_workflow_close_confirm(self, cr, uid, ids, context=None):
        return self.write(cr, uid, ids, {'state' : 'closed_confirmed'}, context=context)

    # training.session
    def action_create_invoice(self, cr, uid, ids, context=None):
        sl_proxy = self.pool.get('training.subscription.line')
        for session in self.browse(cr, uid, ids, context=context):
            sl_ids = sl_proxy.search(cr, uid, [('session_id', '=', session.id),('invoice_line_id', '=', False),('state', 'in', ('confirmed', 'done'))], context=context)
            sl_proxy.action_create_invoice(cr, uid, sl_ids, context=context)
        return True

    # training.session
    def action_workflow_inprogress(self, cr, uid, ids, context=None):
        self.action_create_invoice(cr, uid, ids, context=context)
        return self.write(cr, uid, ids, {'state' : 'inprogress'}, context=context)

    # training.session
    def action_workflow_close(self, cr, uid, ids, context=None):
        workflow = netsvc.LocalService('workflow')
        proxy = self.pool.get('training.subscription.line')
        for session in self.browse(cr, uid, ids, context):
            subscription_line_ids = proxy.search(cr, uid, [('session_id', '=', session.id), ('state', '=', 'confirmed')], context=context)
            for sl_id in subscription_line_ids:
                workflow.trg_validate(uid, 'training.subscription.line', sl_id, 'signal_done', cr)
        return self.write(cr, uid, ids, {'state' : 'closed'}, context=context)

    # trainin.session
    def test_workflow_close(self, cr, uid, ids, context=None):
        return all(seance.state in ('done','cancelled') for session in self.browse(cr, uid, ids, context=context)
                                          for seance in session.seance_ids)

    # training.session
    def action_cancellation_session(self, cr, uid, ids, context=None):
        # just send emails...
        proxy = self.pool.get('training.subscription.line')
        subscription_line_ids = proxy.search(cr, uid, [('session_id', 'in', ids), ('state', '=', 'confirmed')], context=context)
        proxy.send_email(cr, uid, subscription_line_ids, 'session_confirm_cancelled', context)

        proxy = self.pool.get('training.participation.stakeholder')
        for session in self.browse(cr, uid, ids, context=context):
            objs = {}
            for seance in session.seance_ids:
                for contact in seance.contact_ids:
                    if contact.state == 'accepted':
                        objs.setdefault(contact.id, {}).setdefault('seances', []).append(seance)

            proxy.send_email(cr, uid, objs.keys(), 'session_confirm_cancelled', session, context, objs)
        return True

    # training.session
    def action_workflow_cancel(self, cr, uid, ids, context=None):
        self.write(cr, uid, ids, {'state' : 'cancelled'}, context=context)

        workflow = netsvc.LocalService('workflow')
        for session in self.browse(cr, uid, ids, context=context):
            if not session.has_shared_seances:
                for request in session.request_ids:
                    workflow.trg_validate(uid, 'training.participation.stakeholder.request', request.id, 'pshr_cancel', cr)
            else:
                ### What to do with requests to shared seances ?
                pass

            for seance in session.seance_ids:
                seance_cancellable = all([ s.state == 'cancelled' and True or False for s in seance.session_ids ])
                if seance_cancellable:
                    workflow.trg_validate(uid, 'training.seance', seance.id, 'signal_cancel', cr)

            for subline in session.subscription_line_ids:
                workflow.trg_validate(uid, 'training.subscription.line', subline.id, 'signal_cancel', cr)

        return True


    def copy(self, cr, uid, object_id, values, context=None):
        raise osv.except_osv(_("Error"),
                             _("You can not duplicate a session"))

training_session()

class training_mass_subscription_wizard(osv.osv_memory):
    _name = 'training.subscription.mass.wizard'
    _description = 'Mass Subscription Wizard'

    def action_cancel(self, cr, uid, ids, context=None):
        return {'type':'ir.actions.act_window_close'}

    def action_apply(self, cr, uid, ids, context=None):
        if context is None:
            context = {}
        subscription_form_view = context.get('subscription_form_view', False)
        record_id = context.get('record_id', False)

        this = self.browse(cr, uid, ids)[0]

        subscription_proxy = self.pool.get('training.subscription')
        subscription_line_proxy = self.pool.get('training.subscription.line')
        subscription_line_second_proxy = self.pool.get('training.subscription.line.second')

        subscriptions = {}

        if record_id:
            for job in this.job_ids:
                for subscription_mass_line in this.session_ids:
                    sl_id = subscription_line_proxy._create_from_wizard(cr, uid, this, record_id, job, subscription_mass_line, context=context)

            return {
                'type' : 'ir.actions.act_window_close',
            }

        for job in this.job_ids:
            # if the job hasn't a partner, we put this subscription in waiting mode
            if not job.name:
                for subscription_mass_line in this.session_ids:
                    subscription_line_second_proxy._create_from_wizard(cr, uid, this, job, subscription_mass_line, context=context)

            else:
                for subscription_mass_line in this.session_ids:
                    subscriptions.setdefault(job.name.id, []).append((job, subscription_mass_line,))

        subscription_ids = []

        # We create all subscription where there is a partner associated to the job
        for partner_id, lines in subscriptions.iteritems():
            values = subscription_proxy.on_change_partner(cr, uid, [], partner_id)['value']
            values.update({
                'partner_id' : partner_id,
            })

            subscription_id = subscription_proxy.create(cr, uid, values, context=context)

            for job, subscription_mass_line in lines:
                subscription_line_proxy._create_from_wizard(cr, uid, this, subscription_id, job, subscription_mass_line, context=context)

            subscription_ids.append(subscription_id)
        if context.get('menu',False):
            return {
                'domain' : "[('id', 'in', [%s])]" % ','.join(map(str,subscription_ids)),
                'name' : 'Subscriptions',
                'view_type' : 'form',
                'view_mode' : 'tree,form',
                'res_model' : 'training.subscription',
                'type' : 'ir.actions.act_window',
            }
        else:
            return  {'type' : 'ir.actions.act_window_close'}

    _columns = {
        'partner_id' : fields.many2one('res.partner', 'Partner'),
        'job_ids' : fields.many2many('res.partner.job', 'tms_contact_job_rel', 'ms_id', 'job_id', 'Contacts'),
        'session_ids' : fields.one2many('training.subscription.mass.line', 'wizard_id', 'Sessions'),
    }

    def default_get(self, cr, uid, fields, context=None):
        if context is None:
            context = {}
        record_id = context.get('record_id', False)

        res = super(training_mass_subscription_wizard, self).default_get(cr, uid, fields, context=context)

        if record_id:
            partner_id = self.pool.get('training.subscription').browse(cr, uid, record_id, context=context).partner_id.id
            res['partner_id'] = partner_id

        return res

    def fields_view_get(self, cr, uid, view_id=None, view_type='form', context=None, toolbar=False, submenu=False):
        if context is None:
            context = {}
        record_id = context.get('record_id', False)

        res = super(training_mass_subscription_wizard, self).fields_view_get(cr, uid, view_id=view_id, view_type=view_type, context=context, toolbar=toolbar, submenu=submenu)
        if record_id:
            if 'fields' in res and 'partner_id' in res['fields']:
                res['fields']['partner_id']['readonly'] = True

        return res

training_mass_subscription_wizard()

class mass_subscription_line(osv.osv_memory):
    _name = 'training.subscription.mass.line'

    _columns = {
        'wizard_id' : fields.many2one('training.subscription.mass.wizard', 'Wizard'),
        'session_id' : fields.many2one('training.session', 'Session',
                                       domain="[('state', 'in', ('opened','opened_confirmed', 'closed_confirmed', 'inprogress'))]", required=True),
        'allow_closed_session': fields.boolean('Allow Closed Session'),
        'kind' : fields.related('session_id','offer_id', 'kind', type='selection', selection=training_offer_kind_compute, string='Kind', readonly=True),
    }

    def on_change_allow_closed_session(self, cr, uid, ids, new_allow, context=None):
        if new_allow:
            return {
                'domain': {'session_id': []}
            }
        else:
            return {
                'domain': {'session_id': [('state', 'in', ('opened','opened_confirmed', 'closed_confirmed', 'inprogress'))]}
            }


    def on_change_session(self, cr, uid, ids, session_id, context=None):
        return {}

mass_subscription_line()

class training_subscription_line_second(osv.osv):
    _name = 'training.subscription.line.second'
    _rec_name='partner_id'
    _columns = {
        'job_id' : fields.many2one('res.partner.job', 'Contact', required=True),
        'partner_id' : fields.related('job_id', 'name', string='Partner', type='many2one', relation='res.partner', store=True),
        'session_id' : fields.many2one('training.session', 'Session', required=True),
    }

    def _create_from_wizard(self, cr, uid, the_wizard, job, subscription_mass_line, context=None):
        proxy = self.pool.get('training.subscription.line.second')
        values = {
            'job_id' : job.id,
            'session_id' : subscription_mass_line.session_id.id,
        }
        return proxy.create(cr, uid, values, context=context)

training_subscription_line_second()


class training_subscription(osv.osv):
    _name = 'training.subscription'

training_subscription()

class training_subscription_line(osv.osv):
    _name = 'training.subscription.line'

training_subscription_line()

class training_participation(osv.osv):
    _name = 'training.participation'
    _description = 'Participation'
    _rec_name = 'job_id'
    _order = 'seance_date, seance_id, contact_lastname, contact_firstname'

    def _store_get_sublines(self, cr, uid, sl_ids, context=None):
        """get all participations related to this subscription line"""
        if context is None:
            context = {}
        search_context = context.copy()
        # do no limit search to only active records
        search_context['active_test'] = False

        part_pool = self.pool.get('training.participation')
        part_ids = part_pool.search(cr, uid, [('subscription_line_id', 'in', sl_ids)], context=context)
        return part_ids

    def _store_get_contacts(self, cr, uid, contact_ids, context=None):
        """get all participations related to the modified contact ids supplied
           (trigger on 'res.partner.contact' for every contact modification)"""
        if context is None:
            context = {}
        part_pool = self.pool.get('training.participation')

        search_context = context.copy()
        # do no limit search to only active records
        search_context['active_test'] = False

        part_ids = part_pool.search(cr, uid, [('job_id.contact_id', 'in', contact_ids)], context=search_context)
        return part_ids

    def _store_get_jobs(self, cr, uid, job_ids, context=None):
        """get all participations related to the modified job ids supplied
           (trigger on 'res.partner.job' when 'contact_id' field is modified)"""
        if context is None:
            context = {}
        part_pool = self.pool.get('training.participation')

        search_context = context.copy()
        # do not limit search to only active records
        search_context['active_test'] = False

        # get all participations of jobs
        part_ids = part_pool.search(cr, uid, [('job_id', 'in', job_ids)], context=search_context)
        return part_ids

    def _store_get_own(self, cr, uid, job_ids, context=None):
        """get all participations"""
        return job_ids

    _columns = {
        'id' : fields.integer('Database ID', readonly=True),
        'seance_id' : fields.many2one('training.seance', 'Seance', required=True, readonly=True, ondelete='cascade', select=True),
        'seance_date' : fields.related('seance_id', 'date', type='datetime', store=True, string="Seance Date", select=True),
        'group_id' : fields.related('seance_id', 'group_id', string='Group', type='many2one', relation='training.group', store=True, readonly=True),
        'subscription_line_id' : fields.many2one('training.subscription.line', 'Subscription Line', required=True, readonly=True, ondelete='cascade', select=True),
        'session_id' : fields.related('subscription_line_id', 'session_id', type='many2one', relation='training.session', string='Session', store=True),
        'offer_id': fields.related('session_id', 'offer_id', type='many2one', relation='training.offer', string='Offer', readonly=True),
        'course_id' : fields.related('seance_id', 'course_id', 'name', type='char', size=64, readonly=True, string="Course"),
        'duration' : fields.related('seance_id', 'duration', string='Duration', type='float', readonly=True, store=True),
        'kind' : fields.related('seance_id', 'kind', type='selection', selection=training_course_kind_compute, string='Kind'),
        'present' : fields.boolean('Present', help="Allows to know if a participant was present or not"),
        'subscription_id' : fields.related('subscription_line_id', 'subscription_id', type='many2one', relation='training.subscription', string='Subscription', readonly=True),
        'job_id' : fields.related('subscription_line_id', 'job_id', type='many2one', relation='res.partner.job',
                                  string='Participant', readonly=True,
                                  store={
                                      'training.participation': (_store_get_own, ['subscription_line_id'], 10),
                                      'training.subscription.line' : (_store_get_sublines, ['job_id'], 9),
                                  }, select=True,
                                 ),
        'contact_id' : fields.related('subscription_line_id', 'job_id', 'contact_id', type='many2one', relation='res.partner.contact',
                                      string='Contact', readonly=True,
                                      select=True,
                                      store={
                                          'training.subscription.line': (_store_get_sublines, None, 10),
                                          'res.partner.job': (_store_get_jobs, ['contact_id'], 12),
                                      }
                                      ),
        'contact_lastname': fields.related('contact_id', 'name', readonly=True, type='char', size=64, string='Contact Last Name',
                                            select=True,
                                            store={
                                                'training.subscription.line': (_store_get_sublines, None, 11),
                                                'res.partner.contact': (_store_get_contacts, ['name'], 12),
                                                'res.partner.job': (_store_get_jobs, ['contact_id'], 13),
                                            }
                                            ),

        'contact_firstname': fields.related('contact_id', 'first_name', readonly=True, type='char', size=64, string='Contact First Name',
                                            select=True,
                                            store={
                                                'training.subscription.line': (_store_get_sublines, None, 11),
                                                'res.partner.contact': (_store_get_contacts, ['first_name'], 12),
                                                'res.partner.job': (_store_get_jobs, ['contact_id'], 13),
                                            }
                                            ),
        'function_id': fields.related('subscription_line_id', 'job_id', 'function_id', type='many2one', relation='res.partner.function', string='Function', readonly=True),

        'partner_id' : fields.related('subscription_line_id', 'partner_id',
                                      type='many2one', relation='res.partner',
                                      string='Partner', readonly=True),
        'date' : fields.related('seance_id', 'date', type='datetime', string='Date', select=True, readonly=True, store=True),
        'purchase_ids': fields.many2many('purchase.order.line',
                                         'training_participation_purchase_rel',
                                         'participation_id',
                                         'purchase_id',
                                         'Purchases'),
        'purchase_state': fields.selection([('draft','To Do'),
                                            ('done', 'Done'),
                                            ('cancelled', 'Cancelled')
                                           ], 'Purchase State', required=True, readonly=True),

        'summary': fields.text('Summary'),
    }

    _defaults = {
        'present' : lambda *a: 0,
        'purchase_state' : lambda *a: 'draft',
    }

    _sql_constraints = [
        ('uniq_seance_sl', 'unique(seance_id, subscription_line_id)', "The subscription and the seance must be unique !"),
    ]

    def on_change_seance(self, cr, uid, ids, seance_id, context=None):
        if not seance_id:
            return {'value' : {'group_id' : 0}}

        seance = self.pool.get('training.seance').browse(cr, uid, seance_id, context=context)

        return {
            'value' : {
                'group_id' : seance.group_id and seance.group_id.id,
                'date' : seance.date,
            }
        }


    def name_get(self, cr, uid, ids, context=None):
        res = []
        for obj in self.browse(cr, uid, list(set(ids)), context=context):
            sl = obj.subscription_line_id
            oid = obj.id
            if sl.contact_id:
                name = "%s %s (%s)" % (sl.job_id.contact_id.first_name, sl.job_id.contact_id.name, sl.partner_id.name,)
            else:
                name = super(training_participation, self).name_get(cr, uid, [oid], context=context)[0][1]
            res.append((oid, name,))
        return res

    # training.participation
    def create_procurements(self, cr, uid, participation_ids, delayed=False, context=None):
        purchase_order_pool = self.pool.get('purchase.order')
        products = {}
        for participation in self.browse(cr, uid, participation_ids, context=context):
            if participation.seance_id and participation.seance_id.purchase_line_ids:
                for purchase_line in participation.seance_id.purchase_line_ids:
                    products.setdefault(purchase_line, [0.0, []])
                    products[purchase_line][0] = purchase_line.product_qty

                    if purchase_line.fix == 'by_subscription':
                        products[purchase_line][0] = purchase_line.product_qty * len(participation_ids)

                    products[purchase_line][1].append(participation)

        default_warehouse_id = purchase_order_pool.default_get(cr, uid, ['warehouse_id'], context=context).get('warehouse_id')
        if not default_warehouse_id:
            raise osv.except_osv(_('Error'), _('No default warehouse set on purchase order, please define one'))
        location_id = self.pool.get('stock.warehouse').browse(cr, uid, default_warehouse_id, context=context).lot_input_id.id


        participations = {}
        for po_line, (quantity, parts) in products.items():
            # Create purchase order from this po_line ('seance.purchase.line')
            purchase_id = purchase_order_pool.create_from_procurement_line(cr, uid, po_line, quantity, location_id, context=context)
            purchase = purchase_order_pool.browse(cr, uid, purchase_id, context=context)
            # Then get ids of all create purchase.order.line
            purchase_order_line_ids = [ pol.id for pol in purchase.order_line ]

            for part in parts:
                participations.setdefault(part, []).extend(purchase_order_line_ids)

        # write relate purchase.order.line on each participations
        for participation, purchase_ids in participations.items():
            participation.write({'purchase_ids' : [(6, 0, purchase_ids)]}, context=context)

        # mark the purchase as done for this participations
        return self.write(cr, uid, participation_ids, {'purchase_state' : 'done'}, context=context)

    def unlink(self, cr, uid, ids, context=None):
        # TODO cancel the procurements ??
        return super(training_participation, self).unlink(cr, uid, ids, context=context)

training_participation()

class training_seance(osv.osv):
    _name = 'training.seance'
    _description = 'Seance'
    _order = 'date ASC'

    def _shared_compute(self, cr, uid, ids, fieldnames, args, context=None):
        res = dict.fromkeys(ids, 0)
        for seance in self.browse(cr, uid, ids, context=context):
            res[seance.id] = len(seance.session_ids) > 1
        return res

    # training.seance
    def _available_seats_compute(self, cr, uid, ids, fieldnames, args, context=None):
        res = dict.fromkeys(ids, 0)
        for seance in self.browse(cr, uid, ids, context=context):
            count = ['participant_count', 'participant_count_manual'][seance.manual]
            res[seance.id] = int(seance.max_limit - int(seance[count]))
        return res

    # training.seance
    def _draft_seats_compute(self, cr, uid, ids, fieldnames, args, context=None):
        if not ids:
            return False
        res = dict.fromkeys(ids, 0)

        cr.execute("SELECT rel.seance_id, count(1) "
                   "FROM training_subscription_line sl, training_session_seance_rel rel "
                   "WHERE sl.state = 'draft' "
                   "AND sl.session_id = rel.session_id "
                   "AND rel.seance_id IN (" + ",".join(['%s'] * len(ids)) + ") "
                   "GROUP BY rel.seance_id ", ids)

        for seance_id, count in cr.fetchall():
            res[seance_id] = int(count)

        return res

    # training.seance
    def _participant_count(self, cr, uid, ids, name, args, context=None):
        if not ids:
            return False
        res = dict.fromkeys(ids, 0)

        cr.execute('SELECT tp.seance_id, COUNT(DISTINCT(tsl.contact_id)) '
                   'FROM training_participation tp, training_subscription_line tsl '
                   'WHERE tp.subscription_line_id = tsl.id '
                   'AND tp.seance_id in (' + ",".join(['%s'] * len(ids)) + ") "
                   "AND tsl.state in ('confirmed', 'done') "
                   'GROUP BY tp.seance_id',
                   ids
                  )
        for seance_id, count in cr.fetchall():
            res[seance_id] = int(count)

        return res

    _order = "date asc"

    def _confirmed_lecturer_compute(self, cr, uid, ids, fieldnames, args, context=None):
        res = dict.fromkeys(ids, 'no')
        proxy = self.pool.get('training.participation.stakeholder')
        for seance in self.browse(cr, uid, ids, context=context):
            #if seance.kind == 'standard':
                has = len(seance.contact_ids) > 0 and any(c.state in ['accepted', 'done'] for c in seance.contact_ids)
                res[seance.id] = ['no', 'yes'][has]

        return res

    def _get_stakeholders(self, cr, uid, ids, context=None):
        values = set()
        for part in self.pool.get('training.participation.stakeholder').browse(cr, uid, ids, context=context):
            values.add(part.seance_id.id)

        return list(values)

    def _get_sessions_type(self, cr, uid, ids, fieldnames, args, context=None):
        res = []
        for seance in self.browse(cr, uid, ids, context=context):
            types = set()
            for session in seance.session_ids:
                if session.offer_id:
                    types.add(session.offer_id.kind.capitalize())
            res.append((seance.id, types))
        res = dict((x[0], ' / '.join(map(_, x[1]))) for x in res)
        return res

    def _contact_names_compute(self, cr, uid, ids, fieldnames, args, context=None):
        res = dict.fromkeys(ids, 'None')
        for seance in self.browse(cr, uid, ids, context=context):
            name = []
            for contact in seance.contact_ids:
                # skip lecturer request which has been cancelled
                if contact.state in ['cancelled','refused']:
                    continue
                if contact.job_id:
                    lecturer_name = "%s %s" % (contact.job_id.contact_id.name, contact.job_id.contact_id.first_name,)
                    if contact.state == 'draft':
                        name.append("[%s]" % (lecturer_name))
                    else:
                        name.append(lecturer_name)
            res[seance.id] = ", ".join(name)
        return res

    # training.seance
    def name_get(self, cr, uid, ids, context=None):
        return [(obj.id, "%s (%s)" % (obj.name, obj.group_id.name or _('Class %d') % (1,))) for obj in self.browse(cr, uid, list(set(ids)), context=context)]

    def on_change_course(self, cr, uid, ids, course_id, kind, context=None):
        if not course_id:
            return {
                'value' : {
                    'min_limit' : 0,
                    'max_limit' : 0,
                    'duration' : 0.0,
                }
            }

        course = self.pool.get('training.course').browse(cr, uid, course_id, context=context)

        return {
            'value':{
                'name' : course.name,
                'min_limit' : course.course_type_id.min_limit,
                'max_limit' : course.course_type_id.max_limit,
                'duration' : course.duration,
            }
        }


    _columns = {
        'id' : fields.integer('Database ID', readonly=True),
        'is_first_seance' : fields.boolean('First Seance'),
        'name' : fields.char('Name', size=64, required=True),
        'session_ids' : fields.many2many('training.session', 'training_session_seance_rel', 'seance_id', 'session_id', 'Sessions', ondelete='cascade'),
        'sessions_type': fields.function(_get_sessions_type, method=True, string='Session(s) Type', type='char', size=32),
        'forced_lecturer' : fields.boolean('Forced Lecturer(s)'),
        'confirmed_lecturer' : fields.function(_confirmed_lecturer_compute, method=True, string="Confirmed Lecturer",
                                               store={
                                                   'training.participation.stakeholder' : (_get_stakeholders, None, 10),
                                               }, type='selection', selection=[('no', 'No'),('yes','Yes')]),
        'lecturer_note': fields.text('Lecturer Note'),
        'original_session_id' : fields.many2one('training.session', 'Original Session', ondelete='cascade'),
        'original_offer_id': fields.related('original_session_id', 'offer_id', string="Original Offer", type='many2one', relation='training.offer'),
        'original_offer_kind': fields.related('original_offer_id','kind', type='selection', selection=training_offer_kind_compute, string='Original Offer Kind', readonly=True),
        'duplicata' : fields.boolean('Duplicata', required=True),
        'duplicated' : fields.boolean('Duplicated', required=True),
        'date' : fields.datetime('Date', required=True, select=True,help="The create date of seance"),
        'duration' : fields.float('Duration', help="The duration of the seance"),
        'participant_ids' : fields.one2many('training.participation', 'seance_id', 'Participants'),
        'group_id' : fields.many2one('training.group', 'Group', help='The group of participants'),
        'state' : fields.selection([('opened', 'Opened'),
                                    ('confirmed', 'Confirmed'),
                                    ('inprogress', 'In Progress'),
                                    ('closed', 'Closed'),
                                    ('cancelled', 'Cancelled'),
                                    ('done', 'Done')],
                                   'State', required=True, readonly=True, help="The status of the Seance"),
        'contact_ids' : fields.one2many('training.participation.stakeholder', 'seance_id', 'Lecturers', readonly=True),
        'contact_names' : fields.function(_contact_names_compute, method=True, type='char', size=256, string='Lecturers'),
        'course_id' : fields.many2one('training.course', 'Course', domain="[('state_course', '=', 'validated')]"),
        'state_course' : fields.related('course_id', 'state_course', string="Course's State", type='selection',
                                        selection=[('draft', 'Draft'),
                                                   ('pending', 'Pending'),
                                                   ('deprecated', 'Deprecated'),
                                                   ('validated', 'Validated')],
                                        readonly=True),
        'purchase_line_ids' : fields.one2many('training.seance.purchase_line', 'seance_id', 'Supplier Commands'),
        'min_limit' : fields.integer("Minimum Threshold",help='The Minimum of Participants in Seance'),
        'max_limit' : fields.integer("Maximum Threshold",help='The Maximum of Participants in Seance'),
        'user_id' : fields.many2one('res.users', 'Responsible', required=True),
        'available_seats' : fields.function(_available_seats_compute, method=True, string='Available Seats', type='integer', help='Available seats in Seance'),
        'draft_seats' : fields.function(_draft_seats_compute, method=True, string='Draft Subscriptions', type='integer', help='Draft Subscriptions'),
        'presence_form' : fields.selection([('yes', 'Yes'),
                                            ('no', 'No')],
                                           'Presence Form Received', help='The training center has received the presence list from the lecturer'),
        'shared' : fields.function(_shared_compute, method=True, string='Shared', type='boolean', help="Allows to know if the seance is linked with a lot of sessions"),
        'kind': fields.selection(training_course_kind_compute, 'Kind', required=True),
        'master_id' : fields.many2one('training.seance', 'Master Seance'),
        'participant_count' : fields.function(_participant_count, method=True, type="integer",
                                              string="Confirmed Seats", help="Confirmed Subscriptions for this seance"),
        'participant_count_manual' : fields.integer('Manual Confirmed Seats', help="The quantity of supports, catering, ... relative to the number of participants coming from the confirmed seats"),
        'manual' : fields.boolean('Manual', help="Allows to the user to specify the number of participants"),
    }

    def _check_limits(self, cr, uid, ids, context=None):
        obj = self.browse(cr, uid, ids)[0]
        return obj.min_limit <= obj.max_limit

    def _check_date_before_now(self,cr,uid,ids,context=None):
        obj = self.browse(cr, uid, ids[0])
        return obj.date > time.strftime('%Y-%m-%d %H:%M:%S')

    def _check_date_holiday(self, cr, uid, ids, context=None):
        obj = self.browse(cr, uid, ids[0], context=context)
        date = time.strftime('%Y-%m-%d', time.strptime(obj.date, '%Y-%m-%d %H:%M:%S'))
        return not self.pool.get('training.holiday.period').is_in_period(cr, date)

    def _check_date_of_sessions(self, cr, uid, ids, context=None):
        for seance in self.browse(cr, uid, ids, context=context):
            for session in seance.session_ids:
                if seance.date < session.date:
                    return False
        return True

    _constraints = [
        #(_check_date_before_now, "You cannot create a date before now", ['date']),
        #(_check_date_holiday, "You cannot assign a date in a public holiday", ['date']),
        (_check_limits, 'The minimum limit is greater than the maximum limit', ['min_limit', 'max_limit']),
        (_check_date_of_sessions, "You have a session with a date inferior to the seance's date", ['date']),
    ]

    _defaults = {
        'min_limit' : lambda *a: 0,
        'max_limit' : lambda *a: 0,
        'user_id' : lambda obj,cr,uid,context: uid,
        'presence_form' : lambda *a: 'no',
        'confirmed_lecturer' : lambda *a: 'no',
        'state' : lambda *a: 'opened',
        'date' : lambda *a: time.strftime('%Y-%m-%d %H:%M:%S'),
        'kind' : lambda *a: 'standard',
        'duplicata' : lambda *a: 0,
        'duplicated' : lambda *a: 0,
        'is_first_seance' : lambda *a: 0,
        'duration' : lambda *a: 2.0,
        'forced_lecturer' : lambda *a: 0,
    }

    # training.seance
    def action_workflow_open(self, cr, uid, ids, context=None):
        return self.write(cr, uid, ids, {'state' : 'opened'}, context=context)

    # training.seance
    def test_workflow_confirm(self, cr, uid, ids, context=None):
        for seance in self.browse(cr, uid, ids, context=context):
            if any(session.state in ('draft', 'opened') for session in seance.session_ids):
                raise osv.except_osv(_('Warning'),
                                     _('There is at least a session in the "Draft" or "Confirmed" state'))

        return True

    # training.seance
    def action_do_confirm(self, cr, uid, ids, context=None):
        if context is None:
            context = {}
        proxy = self.pool.get('training.participation')
        emails = self.pool.get('training.email')
        report = netsvc.LocalService('report.training.seance.support.delivery.report')

        report_ctx = context.copy()

        # make preventive check that all seance purchase line product does have a supplier.
        # this is necessary as some code issue commit during the process, so we don't blow
        # up worfklow ('ir.attachment'.copy(), workflow...)
        for seance in self.browse(cr, uid, ids, context=context):
            for po_line in seance.purchase_line_ids:
                if not po_line.product_id.seller_ids:
                    raise osv.except_osv(_('Error'), _("Product %s doesn't have any supplier defined") % (po_line.product_id.name))

        for seance in self.browse(cr, uid, ids, context=context):
            if not seance.manual:
                proxy.create_procurements(cr, uid, [x.id for x in seance.participant_ids], context=context)
            else:
                self.create_procurements(cr, uid, [seance.id], context=context)

            # send email to suppliers
            partners = set()
            for po_line in seance.purchase_line_ids:
                for seller in po_line.product_id.seller_ids:
                    partners.add(seller.name)


            for partner in partners:
                to = None
                for address in partner.address:
                    if not address.email:
                        continue
                    if address.type == 'delivery':
                        to = address.email
                        break
                    elif address.type == 'default':
                        to = address.email

                if to is None:
                    continue

                report_ctx['partner'] = partner
                pdf, _r = report.create(cr, uid, [seance.id], {}, context=report_ctx)
                filename = seance.name.replace('/', ' ') + '.pdf'
                emails.send_email(cr, uid, 'procurements', 's', to=to, attachments=[(filename, pdf),], context=context, seance=seance, partner=partner)

            original_session = seance.original_session_id
            if seance.is_first_seance and original_session \
                    and original_session.offer_id.internal_session_notif_active:
                emails.send_email(cr, uid, 'session_open_confirmed', 'i', '_', session=original_session, context=context)

        return True
    # training.seance
    def action_workflow_confirm(self, cr, uid, ids, context=None):
        self.action_do_confirm(cr, uid, ids, context=context)
        return self.write(cr, uid, ids, {'state' : 'confirmed'}, context=context)

    # training.seance
    def action_workflow_inprogress(self, cr, uid, ids, context=None):
        workflow = netsvc.LocalService('workflow')
        for seance in self.browse(cr, uid, ids, context=context):
            for session in seance.session_ids:
                workflow.trg_validate(uid, 'training.session', session.id, 'signal_inprogress', cr)
        return self.write(cr, uid, ids, {'state' : 'inprogress'}, context=context)

    # training.seance
    def action_workflow_close(self, cr, uid, ids, context=None):
        return self.write(cr, uid, ids, {'state' : 'closed'}, context=context)

    # training.seance
    def test_workflow_done(self, cr, uid, ids, context=None):
        return True

    # training.seance
    def action_workflow_done(self, cr, uid, ids, context=None):
        workflow = netsvc.LocalService('workflow')
        self.write(cr, uid, ids, {'state' : 'done'}, context=context)

        for seance in self.browse(cr, uid, ids, context=context):
            for participation in seance.contact_ids:
                workflow.trg_validate(uid, 'training.participation.stakeholder', participation.id, 'signal_done', cr)
            for session in seance.session_ids:
                workflow.trg_validate(uid, 'training.session', session.id, 'signal_close', cr)

        return True

    # training.seance
    def test_workflow_cancel(self, cr, uid, ids, context=None):
        sessions = [ session for seance in self.browse(cr, uid, ids, context)
                             for session in seance.session_ids ]
        can_be_cancelled = not len(sessions) or any(( session.state in ('cancelled', 'inprogress') for session in sessions ))
        if not can_be_cancelled:
            raise osv.except_osv(_("Warning"),
                                 _("You must be cancel all related session !"))
        return can_be_cancelled

    # training.seance
    def action_workflow_cancel(self, cr, uid, ids, context=None):
        workflow = netsvc.LocalService('workflow')

        # Annulation des commandes, des formateurs et des procurements
        mrp_products = {}
        part_ids = []
        for seance in self.browse(cr, uid, ids, context=context):

            for line in seance.purchase_line_ids:
                mrp_products[(seance.id, line.product_id.id,)] = line.fix

            for participation in seance.participant_ids:
                part_ids.append(participation.id)
                for purchase in participation.purchase_ids:
                    key = (seance.id, purchase.product_id.id,)
                    if purchase.state == 'confirmed' and key in mrp_products:
                        workflow.trg_validate(uid, 'purchase.order', purchase.order_id.id, 'purchase_cancel', cr)

            for participation in seance.contact_ids:
                ### if not participation.participation_sh_session_id:
                ###    # participation on this seance only...
                workflow.trg_validate(uid, 'training.participation.stakeholder', participation.id, 'signal_cancel', cr)

            for session in seance.session_ids:
                workflow.trg_validate(uid, 'training.session', session.id, 'signal_close', cr)

        self.pool.get('training.participation').unlink(cr, uid, part_ids, context=context)

        return self.write(cr, uid, ids, {'state' : 'cancelled'}, context=context)

    # training.seance
    def _create_participation(self, cr, uid, seance, subscription_line, context=None):
        proxy = self.pool.get('training.participation')
        values = {
            'seance_id' : seance.id,
            'subscription_line_id' : subscription_line.id,
        }
        return proxy.create(cr, uid, values, context=context)

    def on_change_date(self, cr, uid, ids, date, context=None):
        old_date = ids and self.browse(cr, uid, ids[0], context=context).date or 0

        if self.pool.get('training.holiday.period').is_in_period(cr, date):
            return {
                'value' : {
                    'date' : old_date,
                },
                'warning' : {
                    'title' : _("Selection Date"),
                    'message' : _("You can not select this date because it is a public holiday"),
                },
            }
        return {}

    # training.seance
    def create_procurements(self, cr, uid, ids, context=None):
        purchase_order_pool = self.pool.get('purchase.order')
        default_warehouse_id = purchase_order_pool.default_get(cr, uid, ['warehouse_id'], context=context).get('warehouse_id')
        if not default_warehouse_id:
            raise osv.except_osv(_('Error'), _('No default warehouse set on purchase order, please define one'))
        location_id = self.pool.get('stock.warehouse').browse(cr, uid, default_warehouse_id, context=context).lot_input_id.id

        for seance in self.browse(cr, uid, ids, context=context):
            if seance.manual:
                for po_line in seance.purchase_line_ids:
                    quantity = po_line.product_qty
                    if po_line.fix == 'by_subscription':
                        quantity = quantity * seance.participant_count_manual

                    procurement_id = purchase_order_pool.create_from_procurement_line(cr, uid, po_line, quantity, location_id, context=context)

        return True

    def unlink(self, cr, uid, ids, context=None):
        for seance in self.browse(cr, uid, ids, context=context):
            if seance.state == 'confirmed':
                for pl in seance.purchase_line_ids:
                    if pl.procurement_id:
                        raise osv.except_osv(_("Warning"),
                                             _("You can not suppress a seance with a confirmed procurement"))
            else:
                for participant in seance.participant_ids:
                    if participant.subscription_line_id.invoice_line_id:
                        raise osv.except_osv(_('Warning'),
                                             _("You can not suppress a seance with a invoiced subscription"))

        return super(training_seance, self).unlink(cr, uid, ids, context=context)

    def copy(self, cr, uid, object_id, values, context=None):
        if not 'is_first_seance' in values:
            values['is_first_seance'] = 0

        return super(training_seance, self).copy(cr, uid, object_id, values, context=context)

    def search(self, cr, uid, domain, offset=0, limit=None, order=None, context=None, count=False):
        if context is None:
            context = {}

        offer_id = context.get('offer_id', False)
        if offer_id:
            date = context.get('date', False)
            if not date:
                date = time.strftime('%Y-%m-%d')
            search_domain = [
                ('course_id.offer_ids.offer_id','in',[offer_id]),
                ('state','=','opened'),
                ('date','>=',date),
                ('duplicated','=',False),
            ]
            return super(training_seance, self).search(cr, 1, search_domain, offset=offset, limit=limit, order=order, context=context)

        job_id = context.get('job_id', False)
        request_session_id = context.get('request_session_id', False)
        if job_id and request_session_id:
            search_domain = [
                ('session_ids','in',[request_session_id]),
                ('course_id.lecturer_ids','in',[job_id]),
            ]
            return super(training_seance, self).search(cr, 1, search_domain, offset=offset, limit=limit, order=order, context=context)

        return super(training_seance, self).search(cr, uid, domain, offset=offset,
                                                   limit=limit, order=order, context=context, count=count)

    def _get_product(self, cr, uid, ids, context=None):
        assert len(ids) == 1
        seance = self.browse(cr, uid, ids[0], context)
        return  seance.course_id.course_type_id and seance.course_id.course_type_id.product_id or False

training_seance()

class training_seance_purchase_line(osv.osv):
    _inherit = 'training.base.purchase_line'
    _name = 'training.seance.purchase_line'
    _rec_name = 'product_id'

    def _proc_qty_compute(self, cr, uid, ids, fieldnames, args, context=None):
        res = dict.fromkeys(ids, 0)
        for pl in self.browse(cr, uid, ids, context=context):
            if pl.fix == 'fix':
                multi = 1
            elif pl.seance_id.manual:
                multi = pl.seance_id.participant_count_manual
            else:
                multi = pl.seance_id.participant_count
            res[pl.id] = pl.product_qty * multi
        return res

    _columns = {
        'seance_id' : fields.many2one('training.seance', 'Seance', required=True, ondelete="cascade"),
        'seance_date': fields.related('seance_id', 'date', string="Sceance Date", type="datetime"),
        'course_id': fields.many2one('training.course', 'For Course'),
        'description' : fields.char('Description', size=128, required=True),
        'fix' : fields.selection(_get_product_procurement_methods, 'Invoiced Quantity', required=True),
        'procurement_id' : fields.many2one('procurement.order', 'Procurement'),
        'proc_qty' : fields.function(_proc_qty_compute, method=True, string='Ordered Quantity', type='integer'),
        'auto_update': fields.boolean('Auto. Update'),
    }

    _defaults = {
        'product_qty' : lambda *a: 1,
        'fix' : lambda *a: 'fix',
        'auto_update': lambda *a: True,
    }

    def on_change_product_or_attachment(self, cr, uid, ids, product_id, attachment_id, context=None):
        super_val = super(training_seance_purchase_line, self).on_change_product_or_attachment(cr, uid, ids, product_id, attachment_id, context=context)
        if super_val and super_val.get('value', False):
            ret_val = super_val['value']
        else:
            ret_val = {'product_uom': 0, 'product_price': 0.0, 'attachment_price': 0.0}
        ret_val['description'] = ''

        if not product_id:
            return {'value' : ret_val}

        # most part are done in training.base.purchase_line
        # we only need to care about training.seance.purchase_line specific fields
        product = self.pool.get('product.product').browse(cr, uid, product_id)
        ret_val.update({
            'fix': product.procurement_method,
            'procurement_quantity': product.procurement_trigger,
            'description': product.name,
        })

        return {'value': ret_val}

    def on_change_product_qty(self, cr, uid, ids, product_qty, fix, manual, participant_manual, participant):
        if product_qty and fix:
            if fix == 'fix':
                multi = 1
            elif manual:
                multi = participant_manual
            else:
                multi = participant
            return {'value' : {'proc_qty': product_qty * multi}}
        else:
            return {}

    def _get_analytic_account_id(self, cr, uid, ids, context=None):
        po_line = self.browse(cr, uid, ids[0], context=context)
        if not po_line.course_id:
            # course not defined on purchase line, use seance one.
            if po_line.seance_id.course_id:
                return po_line.seance_id.course_id.analytic_account_id.id
        else:
            return po_line.course_id.analytic_account_id.id

training_seance_purchase_line()

class training_subscription(osv.osv):
    _name = 'training.subscription'
    _description = 'Subscription'

    def check_notification_mode(self, cr, uid, ids, context=None):
        for subr in self.browse(cr, uid, ids, context=context):
            if not subr.partner_id.notif_contact_id \
                and not subr.partner_id.notif_participant:
                raise osv.except_osv(_('Error'),
                        _('No notification mode (HR and/or Participant) for this partner "%s", please choose at least one') % (subr.partner_id.name))
        return True

    def _notification_text_compute_one(self, cr, uid, partner_id, context=None):
        partner_pool = self.pool.get('res.partner')
        if isinstance(partner_id, osv.orm.browse_record):
            partner = partner_id
        else:
            partner = partner_pool.browse(cr, uid, partner_id, context=context)
        notifications = []
        if partner.notif_contact_id:
            notifications.append(_('Partner'))
        if partner.notif_participant:
            notifications.append(_('Participant'))
        res = _(" and ").join(notifications)
        return res

    def _notification_text_compute(self, cr, uid, ids, name, args, context=None):
        res = dict.fromkeys(ids, '')
        for obj in self.browse(cr, uid, ids):
            res[obj.id] = self._notification_text_compute_one(cr, uid, obj.partner_id, context=context)
        return res

    def _get_partner(self, cr, uid, ids, context=None):
        return self.pool.get('training.subscription').search(cr, uid, [('partner_id', 'in', ids),('state', '=', 'draft')], context=context)

    _columns = {
        'name' : fields.char('Reference', size=32, required=True, readonly=True, help='The unique identifier is generated by the system (customizable)'),
        'create_date' : fields.datetime('Creation Date', readonly=True),
        'state' : fields.selection([('draft', 'Draft'), ('confirmed','Request Sent'), ('cancelled','Cancelled'), ('done', 'Done') ], 'State', readonly=True, required=True, help='The state of the Subscription'),
        'partner_id' : fields.many2one('res.partner', 'Partner', required=True,help='The Subscription name', select=True, **WRITABLE_ONLY_IN_DRAFT),
        'partner_rh_email' : fields.char('Subscription Contact', size=64),
        'address_id' : fields.many2one('res.partner.address', 'Invoice Address', required=True,help='The Subscription invoice address', **WRITABLE_ONLY_IN_DRAFT),
        'subscription_line_ids' : fields.one2many('training.subscription.line', 'subscription_id',
                                                  'Subscription Lines', **WRITABLE_ONLY_IN_DRAFT),

        'pricelist_id' : fields.related('partner_id', 'property_product_pricelist', string='Pricelist', type='many2one', relation='product.pricelist', readonly=True),
        'payment_term_id' : fields.many2one('account.payment.term', 'Payment Term', **WRITABLE_ONLY_IN_DRAFT),
        'responsible_id' : fields.many2one('res.users', 'Creator', required=True, readonly=True, select=1),
        'origin' : fields.char('Origin', size=64, **WRITABLE_ONLY_IN_DRAFT),
        'custom_ref': fields.char('Custom Reference', size=64, **WRITABLE_ONLY_IN_DRAFT),
        'notification_active' : fields.boolean('Active', **WRITABLE_ONLY_IN_DRAFT),
        'notification_text' : fields.function(_notification_text_compute, method=True,
                                              string='Kind', type='char',
                                              store={
                                                  'res.partner' : (_get_partner, None, 10),
                                                  'training.subscription' : (lambda self, cr, uid, ids, context=None: ids, None, 10),
                                              },
                                              size=64),
        'is_from_web': fields.boolean('Is from Web?', help='Is this subscription come from an online order', readonly=True),
    }

    def create(self, cr, uid, vals, context):
        if vals.get('name', '/') == '/':
            vals['name'] = self.pool.get('ir.sequence').get(cr, uid, 'training.subscription')
        if not vals.get('notification_text', '') and vals.get('partner_id', None):
            vals['notification_text'] = self._notification_text_compute_one(cr, uid, vals['partner_id'], context=context)
        return super(training_subscription, self).create(cr, uid, vals, context)

    _defaults = {
        'state' : lambda *a: 'draft',
        'name' : lambda *args: '/',
        'responsible_id' : lambda obj, cr, uid, context: uid,
        'notification_active' : lambda *a: 1,
    }

    def copy(self, cr, uid, subscription_id, default_values, context=None):
        default_values.update({
            'name' : '/',
            'subscription_line_ids' : [],
        })

        return super(training_subscription, self).copy(cr, uid, subscription_id, default_values, context=context)

    def on_change_partner(self, cr, uid, ids, partner_id):
        """
        This function returns the invoice address for a specific partner, but if it didn't find an
        address, it takes the first address from the system
        """
        if not partner_id:
            return {'value' :
                    {'address_id' : 0,
                     'partner_rh_email' : '',
                     'pricelist_id' : 0,
                     'payment_term_id' : 0,
                     'notification_text': '',
                    }
                   }

        values = {}

        proxy = self.pool.get('res.partner')
        partner = proxy.browse(cr, uid, partner_id)

        values['notification_text'] = self._notification_text_compute_one(cr, uid, partner)

        partner_rh = partner.notif_contact_id
        if partner_rh and partner_rh.email:
            partner_rh_email = partner_rh.email
            values['partner_rh_email'] = partner_rh.email

        price_list_id = partner.property_product_pricelist
        if price_list_id:
            values['pricelist_id'] = price_list_id.id

        payment_term_id = partner.property_payment_term
        if payment_term_id:
            values['payment_term_id'] = payment_term_id.id

        proxy = self.pool.get('res.partner.address')
        ids = proxy.search(cr, uid, [('partner_id', '=', partner_id),('type', '=', 'invoice')])

        if not ids:
            ids = proxy.search(cr, uid, [('partner_id', '=', partner_id)])

        if ids:
            values['address_id'] = ids[0]

        return {'value' : values}

    # training.subscription
    def action_workflow_draft(self, cr, uid, ids, context=None):
        return self.write(cr, uid, ids, {'state' : 'draft'}, context=context)

    def trg_validate_workflow(self, cr, uid, ids, signal, context=None):
        workflow = netsvc.LocalService("workflow")
        for subscription in self.browse(cr, uid, ids, context=context):
            for sl in subscription.subscription_line_ids:
                workflow.trg_validate(uid, 'training.subscription.line', sl.id, signal, cr)
        return True

    # training.subscription
    def action_workflow_cancel(self, cr, uid, ids, context=None):
        workflow = netsvc.LocalService('workflow')
        sl_proxy = self.pool.get('training.subscription.line')
        sl_ids = []
        stack_holder_pool = self.pool.get('training.participation.stakeholder')
        for subscription in self.browse(cr, uid, ids, context=context):
            if subscription.state == 'draft':
                for sl in subscription.subscription_line_ids:
                    workflow.trg_validate(uid, sl._name, sl.id, 'signal_cancel', cr)
            elif subscription.state == 'confirmed':
                for sl in subscription.subscription_line_ids:
                    if sl.state == 'draft':
                        sl_ids.append(sl.id)
                    elif sl.state != 'confirmed':
                        continue
                    objs = {}
                    for seance in sl.session_id.seance_ids:
                        for contact in seance.contact_ids:
                            if contact.state == 'confirmed':
                                objs.setdefault(contact.id, {}).setdefault('seances', []).append(seance)

                    stack_holder_pool.send_email(cr, uid, objs.keys(), 'sub_cancelled', sl.session_id, context, objs)
                    sl_ids.append(sl.id)

        sl_proxy.action_workflow_invoice_and_send_emails(cr, uid, sl_ids, context)
        workflow = netsvc.LocalService('workflow')
        for oid in sl_ids:
            workflow.trg_validate(uid, 'training.subscription.line', oid, 'signal_cancel', cr)

        return self.write(cr, uid, ids, {'state' : 'cancelled'}, context=context)

    # training.subscription
    def action_workflow_confirm(self, cr, uid, ids, context=None):
        self.check_notification_mode(cr, uid, ids, context=context)
        return self.write(cr, uid, ids, {'state' : 'confirmed'}, context=context)

    # training.subscription
    def action_workflow_done(self, cr, uid, ids, context=None):
        return self.write(cr, uid, ids, {'state' : 'done'}, context=context)

    def test_wkf_done(self, cr, uid, ids, context=None):
        subs = self.browse(cr, uid, ids, context)
        lines = [line for sub in subs for line in sub.subscription_line_ids]
        return any(line.state != 'cancelled' for line in lines) and \
               all(line.state in ('cancelled', 'done') for line in lines)

    def test_wkf_cancel(self, cr, uid, ids, context=None):
        subs = self.browse(cr, uid, ids, context)
        lines = [line for sub in subs for line in sub.subscription_line_ids]
        return all(line.state == 'cancelled' for line in lines)

    _order = 'create_date desc'

    _sql_constraints=[
        ('uniq_name', 'unique(name)', 'The name of the subscription must be unique !'),
    ]

training_subscription()

class training_subscription_line(osv.osv):
    _name = 'training.subscription.line'
    _description = 'Subscription Line'

    def on_change_job(self, cr, uid, ids, job_id, context=None):
        if not job_id:
            return False

        return {'value' : {'job_email' : self.pool.get('res.partner.job').browse(cr, uid, job_id, context=context).email}}

    def on_change_subscription(self, cr, uid, ids, subscription_id, context=None):
        if not subscription_id:
            return False

        return {'value' : {'partner_id' : self.pool.get('training.subscription').browse(cr, uid, subscription_id, context=context).partner_id.id}}


    def _paid_compute(self, cr, uid, ids, fieldnames, args, context=None):
        res = dict.fromkeys(ids, 0)
        for obj in self.browse(cr, uid, ids, context=context):
            if obj.invoice_line_id:
                res[obj.id] = obj.invoice_line_id.invoice_id.state == 'paid'
        return res

    def _theoritical_disponibility_compute(self, cr, uid, ids, fieldnames, args, context=None):
        res = dict.fromkeys(ids, 0)
        for obj in self.browse(cr, uid, ids, context=context):
            res[obj.id] = int(obj.session_id.available_seats) - int(obj.session_id.draft_subscriptions)
        return res

    def _was_present_compute(self, cr, uid, ids, fieldnames, args, context=None):
        res = dict.fromkeys(ids, False)
        for sl in self.browse(cr, uid, ids, context=context):
            res[sl.id] = any(part.present for part in sl.participation_ids)
        return res

    def _store_get_jobs(self, cr, uid, job_ids, context=None):
        """get all subscription line related to the modified job ids supplied
           (trigger on 'res.partner.job' when 'contact_id' field is modified)"""
        if context is None:
            context = {}
        subl_pool = self.pool.get('training.subscription.line')

        search_context = context.copy()
        # do not limit search to only active records
        search_context['active_test'] = False

        # get all participations of jobs
        subl_ids = subl_pool.search(cr, uid, [('job_id', 'in', job_ids)], context=search_context)
        return subl_ids

    def _store_get_own(self, cr, id, subl_ids, context=None):
        return subl_ids

    _columns = {
        'name' : fields.char('Reference', size=64, required=True, readonly=True),
        'create_uid': fields.many2one('res.users', 'Created by', readonly=True),
        'create_date': fields.datetime('Created at', readonly=True),
        'validation_uid': fields.many2one('res.users', 'Validated by', readonly=True),
        'validation_date': fields.datetime('Validated at', readonly=True),
        'cancellation_uid': fields.many2one('res.users', 'Cancelled by', readonly=True),
        'cancellation_date': fields.datetime('Cancelled at', readonly=True),
        'subscription_id' : fields.many2one('training.subscription', 'Subscription', required=True, ondelete='cascade',
                                            select=1,help='Select the subscription', **WRITABLE_ONLY_IN_DRAFT),
        'subscription_state' : fields.related('subscription_id', 'state', type='selection',
                                              selection=[
                                                  ('draft', 'Draft'),
                                                  ('confirmed','Confirmed'),
                                                  ('cancelled','Cancelled'),
                                                  ('done', 'Done'),
                                              ], string='State', readonly=True, required=True, help='The state of the Subscription'),
        'price_list_id' : fields.many2one('product.pricelist', 'Pricelist', required=True, domain="[('type', '=', 'sale')]", **WRITABLE_ONLY_IN_DRAFT),
        'partner_hr_email' : fields.related('subscription_id', 'partner_rh_email', type='char', size=64, string='HR Email', readonly=True),
        'notification_text' : fields.related('subscription_id', 'notification_text', type='char', size=64, store=True, string='Notification (Kind)', readonly=True),
        'session_id' : fields.many2one('training.session', 'Session', select=1, required=True,
                                       domain="[('state', 'in', ('opened','opened_confirmed', 'closed_confirmed', 'inprogress'))]",
                                       help='Select the session', **WRITABLE_ONLY_IN_DRAFT),
        'session_state': fields.related('session_id', 'state', readonly=True, type='selection',
                                        selection= [
                                            ('draft', 'Draft'),
                                            ('opened', 'Opened'),
                                            ('opened_confirmed', 'Confirmed'),
                                            ('closed_confirmed', 'Closed Subscriptions'),
                                            ('inprogress', 'In Progress'),
                                            ('closed', 'Closed'),
                                            ('cancelled', 'Cancelled'),
                                        ], string='Session State'),
        'session_date': fields.related('session_id', 'date', readonly=True, type="datetime", string="Session Date"),
        'offer_id' : fields.related('session_id', 'offer_id', type='many2one', relation='training.offer', string='Offer', store=True, readonly=True),
        'offer_product_line_id' : fields.related('session_id', 'offer_id', 'product_line_id', type='many2one', relation='training.course_category', string='Product Line'),
        'price' : fields.float('Sales Price', digits_compute=dp.get_precision('Account'), required=True, write=['base.group_user'], **WRITABLE_ONLY_IN_DRAFT),
        'partner_id' : fields.related('subscription_id', 'partner_id', type='many2one', store=True, select=True,
                                      relation='res.partner', string="Partner", readonly=True),
        'job_id' : fields.many2one('res.partner.job', 'Participant', select=1, required=True,
                                   domain="[('name', '=', parent.partner_id),('state', '=', 'current')]",
                                   help='Select the Participant', **WRITABLE_ONLY_IN_DRAFT),
        'job_email' : fields.char('Participant Email', size=64, help='Participant Email Address'),
        'contact_id' : fields.related('job_id', 'contact_id', type='many2one', relation='res.partner.contact', string='Contact', readonly=True,
                                        store = {
                                            'training.subscription.line': (_store_get_own, ['job_id'], 9),
                                            'res.partner.job': (_store_get_jobs, ['contact_id'], 10),
                                        }),
        'contact_firstname' : fields.related('contact_id', 'first_name', string='Contact Firstname', type='char', size=64, readonly=True),
        'contact_lastname' : fields.related('contact_id', 'name', string='Contact Lastname', type='char', size=64, readonly=True),
        'partner_rh' : fields.related('partner_id', 'notif_contact_id', type='many2one', relation='res.partner.job', readonly=True, string='Subscription Contact', store=True),
        'invoice_line_id' : fields.many2one('account.invoice.line', 'Invoice Line', readonly=True),
        'invoice_id' : fields.related('invoice_line_id', 'invoice_id', type='many2one', relation='account.invoice', string="Invoice", readonly=True, store=True),
        'paid' : fields.function(_paid_compute, method=True, string='Paid', type='boolean', readonly=True),
        'kind' : fields.related('session_id', 'kind', string="Kind", type="selection", selection=training_offer_kind_compute, readonly=True),
        'state' : fields.selection([('draft', 'Draft'),
                                    ('confirmed','Confirmed'),
                                    ('cancelled','Cancelled'),
                                    ('done', 'Done') ],
                                   'State', required=True, readonly=True, select=1,help='The state of participant'),
        'date' : fields.related('session_id', 'date', type='datetime', string='Date', readonly=True),
        'available_seats' : fields.related('session_id', 'available_seats', type='integer', readonly=True, string='Available Seats'),
        'draft_subscriptions' : fields.related('session_id', 'draft_subscriptions', type='integer', readonly=True, string='Draft Subscriptions'),
        'has_certificate' : fields.boolean('Has Certificate', readonly=True),
        'reason_cancellation' : fields.text('Reason of Cancellation', readonly=True),
        'theoritical_disponibility' : fields.function(_theoritical_disponibility_compute, method=True, string='Theoritical Disponibility', type='integer'),
        'max_limit' : fields.related('session_id', 'max_limit', string='Maximum Threshold', type='integer', store=True, readonly=True),
        'confirmed_subscriptions' : fields.related('session_id', 'confirmed_subscriptions', string='Confirmed Subscriptions', type='integer', readonly=True),
        'participation_ids' : fields.one2many('training.participation', 'subscription_line_id', 'Participations', readonly=True),
        'internal_note' : fields.text("Internal Note"),
        'email_note': fields.text("Email Note"),
        'was_present' : fields.function(_was_present_compute, method=True, type='boolean', string='Was Present'),
    }

    def _default_name(self, cr, uid, context=None):
        return self.pool.get('ir.sequence').get(cr, uid, 'training.subscription.line')

    _defaults = {
        'state' : lambda *a: 'draft',
        'name' : _default_name,
        'has_certificate' : lambda *a: 0,
    }

    _order = 'name desc'

    def copy(self, cr, uid, object_id, values, context=None):
        if 'name' not in values:
            values['name'] = self._default_name(cr, uid, context=context)

        if 'participation_ids' not in values:
            values['participation_ids'] = []

        if 'session_id' in values:
            session = self.pool.get('training.session').browse(cr, uid, values['session_id'], context=context)
            values['session_state'] = session.state
            values['session_date'] = session.date

        return super(training_subscription_line, self).copy(cr, uid, object_id, values, context=context)

    # training.subscription.line
    def _check_subscription(self, cr, uid, ids, context=None):
        for sl in self.browse(cr, 1, ids, context=context):
            session = sl.session_id
            contact = sl.job_id.contact_id
            name = "%s %s" % (contact.first_name, contact.name)

            same_contact_same_session = ['&', '&', '&', ('contact_id', '=', contact.id), ('session_id', '=', session.id), ('id', '!=', sl.id), ('state', '!=', 'cancelled')]

            sl_ids = self.search(cr, 1, same_contact_same_session, context=context)
            if sl_ids:
                raise osv.except_osv(_('Error'), _('%(participant)s is already following the session "%(session)s"') % {'participant': name, 'session': session.name})

        return True

    _constraints = [
        (lambda self, *args, **kwargs: self._check_subscription(*args, **kwargs), 'Invalid Subscription', ['session_id', 'contact_id', 'state']),
    ]

    def _get_price_list_from_product_line(self, session, partner_id, default_price_list):
        pl = default_price_list
        if session.offer_id.product_line_id and session.offer_id.product_line_id.price_list_id:
            if any(partner.id == partner_id for partner in session.offer_id.product_line_id.partner_ids):
                pl = session.offer_id.product_line_id.price_list_id.id

        if not pl:
            raise osv.except_osv(_('Warning'),
                                 _("Please, Can you check the price list of the partner ?"))
        return pl

    # training.subscription.line
    def on_change_session(self, cr, uid, ids, session_id, price_list_id, partner_id, context=None):
        if not session_id:
            return False

        session = self.pool.get('training.session').browse(cr, uid, session_id, context=context)
        if (not price_list_id) and partner_id:
            part = self.pool.get('res.partner').browse(cr, uid, partner_id)
            price_list_id = part.property_product_pricelist and part.property_product_pricelist.id or False

        if not session.offer_id.product_id:
            if session.kind == 'intra':
                return {
                    'value': {
                        'price': 0.0,
                        'kind': session.kind,
                        'price_list_id': price_list_id,
                        'session_date': session.date,
                    }
                }
            else:
                return False

        price_list = self._get_price_list_from_product_line(session, partner_id, price_list_id)

        price = self.pool.get('product.pricelist').price_get(cr, uid, [price_list], session.offer_id.product_id.id, 1.0)[price_list]

        values = {
            'value' : {
                'price' : price,
                'kind' : session.kind,
                'price_list_id' : price_list,
                'session_date' : session.date,
            }
        }

        return values

    def on_change_price_list(self, cr, uid, ids, session_id, price_list_id, context=None):
        if not price_list_id or not session_id:
            return False

        pricelist_proxy = self.pool.get('product.pricelist')
        session = self.pool.get('training.session').browse(cr, uid, session_id, context=context)

        if not session.offer_id.product_id:
            return False

        return {
            'value' : {
                'price' : pricelist_proxy.price_get(cr, uid, [price_list_id], session.offer_id.product_id.id, 1.0)[price_list_id]
            }
        }

    # training.subscription.line
    def _get_values_from_wizard(self, cr, uid, subscription_id, job, subscription_mass_line, context=None):
        # this method can easily surcharged by inherited modules
        subscription = self.pool.get('training.subscription').browse(cr, uid, subscription_id, context=context)
        session = subscription_mass_line.session_id

        def_pricelist_id = job.name.property_product_pricelist.id

        values = {
            'subscription_id' : subscription_id,
            'job_id' : job.id,
            'job_email': job.email,
            'session_id' : session.id,
            'kind': session.kind,
            'price_list_id': def_pricelist_id,
            'price': 0,
        }
        ocv = self.on_change_session(cr, uid, [], session.id, def_pricelist_id, job.name.id, context=context)
        if ocv and 'value' in ocv:
            values.update(ocv['value'])
        ocv = self.on_change_price_list(cr, uid, [], values['session_id'], values.get('price_list_id', False), context=context)

        if ocv and 'value' in ocv:
            values.update(ocv['value'])
        return values

    # training.subscription.line
    def _create_from_wizard(self, cr, uid, the_wizard, subscription_id, job, subscription_mass_line, context=None):
        values = self._get_values_from_wizard(cr, uid,
                                              subscription_id, job,
                                              subscription_mass_line,
                                              context=context)
        return self.create(cr, uid, values, context=context)

    # training.subscription.line
    def action_workflow_draft(self, cr, uid, ids, context=None):
        return self.write(cr, uid, ids, {'state' : 'draft'}, context=context)

    # training.subscription.line
    def test_workflow_confirm(self, cr, uid, ids, context=None):
        subs = set()
        for subl in self.read(cr, uid, ids, ['subscription_id'], context=context):
                if subl['subscription_id']:
                    subs.add(subl['subscription_id'][0])
        self.pool.get('training.subscription').check_notification_mode(cr, uid, list(subs), context=context)
        return True

    # training.subscription.line
    def action_create_participation(self, cr, uid, ids, context=None):
        for sl in self.browse(cr, uid, ids, context=context):
            sl.session_id._create_participation(sl, context=context)
            sl.write({'state' : sl.state})

        return True

    # training.subscription.line
    def action_workflow_send_confirm_emails(self, cr, uid, ids, context=None):
        # the confirm function will send an email to the participant and/or the HR Manager
        lines = {}

        for sl in self.browse(cr, uid, ids, context=context):
            sl.session_id._create_participation(sl, context=context)

            # opened -> opened; opened_confirmed and closed_confirmed -> confirmed
            state = sl.session_id.state.rsplit('_', 1)[-1]

            if state in ('opened', 'confirmed', 'inprogress'):
                lines.setdefault(state, []).append(sl.id)

        if 'opened' in lines:
            self.send_email(cr, uid, lines['opened'], 'sub_confirm_open', context)
        if 'confirmed' in lines:
            # TODO: change signal name (as closed confirmed sessions are also possible)
            self.send_email(cr, uid, lines['confirmed'], 'sub_confirm_openconf', context)
        if 'inprogress' in lines:
            self.send_email(cr, uid, lines['inprogress'], 'sub_confirm_openconf', context)

        return True

    # training.subscription.line
    def action_workflow_confirm(self, cr, uid, ids, context=None):
        self.write(cr, uid, ids, {'state': 'confirmed', 'validation_date': time.strftime('%Y-%m-%d %H:%M:%S'), 'validation_uid': uid}, context=context)
        return True

    def _subscription_refund_prepare_invoice(self, cr, uid, invoice, context=None):
        journal_obj = self.pool.get('account.journal')
        get_refund_journal_by_type = partial(journal_obj._get_journal_id_by_type,
                                            cr, uid, refund=True, context=context)
        if invoice.type in ('in_invoice', 'in_refund'):
            refund_journal_id = get_refund_journal_by_type('purchase')
        else:
            refund_journal_id = get_refund_journal_by_type('sale')

        invoice_values = {
            'name' : "[REFUND] %s - %s" % (invoice.name, _('Cancellation')),
            'origin' : invoice.origin,
            'type' : 'out_refund',
            'reference' : "[INVOICE] %s" % (invoice.reference,),
            'partner_id' : invoice.partner_id.id,
            'address_contact_id' : invoice.address_contact_id and invoice.address_contact_id.id,
            'address_invoice_id' : invoice.address_invoice_id and invoice.address_invoice_id.id,
            # NOTE: Use the same journal as orignial invoice, otherwise
            #       we have to create a new analytic distribution for
            #       each lines
            # FIXME: this is not correct anymore!
            'journal_id': refund_journal_id,
            'account_id' : invoice.account_id.id,
            'fiscal_position' : invoice.fiscal_position.id,
            'date_invoice' : time.strftime('%Y-%m-%d'),
        }
        return invoice_values

    def _subscription_refund_prepare_line(self, cr, uid, refund_value, invoice_line, subscription_line, context=None):
        """Prepare a new refund line based on existing invoice line
        :param cr: database cursor
        :param uid: current user id
        :param refund_value: dictionnary of new refund invoice values
        :param invoice_line: browse_record instance of original invoice line
        :param subscription_line: browse_record instance of original subscription line
        :param context: context dictionnary
        """
        if context is None:
            context = {}
        refund_line = {
            #'invoice_id': invoice_id, # this is forced later on
            'product_id': invoice_line.product_id.id,
            'quantity': invoice_line.quantity,
            'name': invoice_line.name,
            'origin': subscription_line.name,
            'account_id': invoice_line.account_id.id,
            'discount': context.get('discount', 0.0),
            'price_unit': invoice_line.price_unit,
            'invoice_line_tax_id': [(6, 0, [ t.id for t in invoice_line.invoice_line_tax_id])],
            # NOTE: reuse the same analytic distribution as orignal
            #       invoice. ;-)
            'analytics_id': invoice_line.analytics_id.id,
        }
        return refund_line

    def action_create_refund(self, cr, uid, ids, context=None):
        if context is None:
            context = {}
        invoice_proxy = self.pool.get('account.invoice')
        invoice_line_proxy = self.pool.get('account.invoice.line')
        seance_proxy = self.pool.get('training.seance')

        invoices = {}

        for sl in self.browse(cr, uid, ids, context=context):
            if sl.price >= -0.00001 and sl.price <= 0.00001:
                # sale price is ZERO, nothing to refund
                continue
            invoices.setdefault(sl.invoice_id, []).append( sl )

        for invoice, subscription_lines in invoices.iteritems():
            invoice_values = self._subscription_refund_prepare_invoice(cr, uid, invoice, context=context)
            invoice_lines = []

            for sl in subscription_lines:
                if not sl.invoice_line_id:
                     raise osv.except_osv(_('Error'),
                                     _("Subscription line %s doesn't have any corresponding invoice line"))

                invoice_line = invoice_line_proxy.browse(cr, uid, sl.invoice_line_id.id, context=context)

                refund_line = self._subscription_refund_prepare_line(cr, uid, invoice_values, invoice_line, sl, context=context)

                _to_pay = refund_line['price_unit'] * float(100.0 - (refund_line['discount'] or 0.0) / 100.0)
                # we only create the refunded line if amount to pay is not "zero"
                if not (-0.0001 < _to_pay < 0.0001):
                    invoice_lines.append(refund_line)

            if invoice_lines:
                invoice_id = invoice_proxy.create(cr, uid, invoice_values, context=context)
                for l in invoice_lines:
                    l['invoice_id'] = invoice_id
                    invoice_line_proxy.create(cr, uid, l, context=context)
                invoice_proxy.button_compute(cr, uid, [invoice_id], context=context)
                invoice = invoice_proxy.browse(cr, uid, invoice_id, context=context)
                if not invoice.amount_untaxed:
                    invoice.unlink()

        return True


    # training.subscription.line
    def action_workflow_invoice_and_send_emails(self, cr, uid, ids, context=None):
        """invoice subscription line during 'cancel with penalities' call"""
        return self.action_create_invoice_and_refund(cr, uid, ids, context)

    # training.subscription.line
    def _delete_participations(self, cr, uid, ids, context=None):
        proxy = self.pool.get('training.participation')
        part_ids = []
        mrp_products = {}

        for sl in self.browse(cr, uid, ids, context=context):
            for part in sl.participation_ids:
                part_ids.append(part.id)

                seance = part.seance_id
                for line in seance.purchase_line_ids:
                    mrp_products[(seance.id, line.product_id.id,)] = line.fix

                for purchase in part.purchase_ids:
                    key = (seance.id, purchase.product_id.id,)
                    # before deleting participations we need to check
                    # for relate purchases. If still not acceped or refused
                    # by supplier, we can still modify it
                    if purchase.state == 'confirmed' and key in mrp_products:
                        if mrp_products[key] != 'fix':
                            # we need to use 'read' to bypass 'browse' cache
                            prod_qty = purchase.read(['product_qty'])[0]['product_qty']
                            purchase.write({'product_qty': prod_qty - 1})

                        prod_qty = purchase.read(['product_qty'])[0]['product_qty']
                        # if final quantity is null, we can simply cancel the purchase
                        if prod_qty == 0:
                            workflow = netsvc.LocalService('workflow')
                            workflow.trg_validate(uid, 'purchase.order', purchase.order_id.id, 'purchase_cancel', cr)
        proxy.unlink(cr, uid, part_ids, context=context)
        return True

    # training.subscription.line
    def action_create_invoice_and_refund(self, cr, uid, ids, context=None):
        if context is None:
            context = {}
        config_pool = self.pool.get('ir.config')
        DISCOUNT_DAYS = config_pool.get(cr, uid, 'training.subscription.cancel.discount.days') # calendar days

        # below x calendar days have a discount of y percent...
        DISCOUNT_REFUND_PERCENT = config_pool.get(cr, uid, 'training.subscription.cancel.discount.rate')
        DISCOUNT_PERCENT = (100.0 - DISCOUNT_REFUND_PERCENT)

        full_invoices = []
        discount_invoices = []
        full_refunds = []
        discount_refunds = []

        # Selection de toutes les lignes d'inscriptions
        for sl in self.browse(cr, uid, ids, context=context):
            today = datetime.datetime.today()
            session_date_minus_x = datetime.datetime.strptime(sl.session_id.date[:10], '%Y-%m-%d') - relativedelta(days=DISCOUNT_DAYS)
            before_x_days = today < session_date_minus_x
            if sl.price >= -0.00001 and sl.price <= 0.00001:
                # Ignore subscription line with a sale price of ZERO
                continue
            if sl.invoice_line_id:
                if sl.partner_id.no_penalties:
                    # No pernalties, full refund
                    full_refunds.append(sl.id)
                elif before_x_days:
                    # Create a refund excepts for exams
                    if sl.kind != 'exam':
                        discount_refunds.append(sl.id)
                else:
                    # No refund at all
                    continue
            else:
                if not sl.partner_id.no_penalties:
                    if sl.kind == 'exam' or not before_x_days:
                        full_invoices.append(sl.id)
                    else:
                        discount_invoices.append(sl.id)

        ctx = context.copy()
        ctx['cancellation'] = True
        ctx['discount'] = 0.0

        if full_refunds:
            self.action_create_refund(cr, uid, full_refunds, context=ctx)

        if full_invoices:
            self.action_create_invoice(cr, uid, full_invoices, context=ctx)

        if discount_refunds:
            ctx['discount'] = DISCOUNT_REFUND_PERCENT
            self.action_create_refund(cr, uid, discount_refunds, context=ctx)

        if discount_invoices:
            ctx['discount'] = DISCOUNT_PERCENT
            self.action_create_invoice(cr, uid, discount_invoices, context=ctx)
        return True

    def _invoice_min_max(self, cr, uid, values, context=None):
        if context is None:
            context = {}
        # NOTE: replace values in place
        if not context.get('cancellation', False):
            return True

        config_pool = self.pool.get('ir.config')
        MIN_AMOUNT = config_pool.get(cr, uid, 'training.subscription.invoice.threshold.minimum')
        MAX_AMOUNT = config_pool.get(cr, uid, 'training.subscription.invoice.threshold.maximum')

        price = values['price_unit']
        discount = values['discount'] or 0.0

        amount = price * (1.0 - discount / 100.0)
        if amount < MIN_AMOUNT:
            price = min(price, MIN_AMOUNT)
            discount = 0.0
        elif amount > MAX_AMOUNT:
            price = MAX_AMOUNT
            discount = 0.0

        values['price_unit'] = price
        values['discount'] = discount
        return values

    def _get_courses(self, cr, uid, ids, context=None):
        res = dict.fromkeys(ids, [])
        for sl in self.browse(cr, uid, ids, context=context):
            res[sl.id] = set(seance.course_id for seance in sl.session_id.seance_ids if seance.course_id)
        return res

    # training.subscription.line
    def _get_invoice_line_taxes(self, cr, uid, subline, fiscal_position, partner, session, context=None):
        if session.offer_id.product_id:
            if session.offer_id.product_id.taxes_id:
                taxes = session.offer_id.product_id.taxes_id
                tax_id = self.pool.get('account.fiscal.position').map_tax(cr, uid, fiscal_position, taxes)
                return [(6, 0, [t for t in tax_id])]
        return []

    def _invoicing_get_analytic_distribution_by_record(self, cr, uid, subline, courses, journal, context=None):
        """
        Create a new analytic distribution from subline, with a pro-rata line of each courses
        :param cr: database cursor
        :param uid: current user id
        :param subline: browse_record of training.subscription.line object
        :param courses: list of browse_record of training.course object
        :param journal: browse_record of account.journal object
        :param context: current request context
        :return: new analytic distribution id
        """
        total_duration = 0.0
        for c in courses:
            if c.duration == 0.0:
                raise osv.except_osv(_('Error'),
                                    _("The following course has not a valid duration \"%s\" (%d)") % (c.name, c.id))
            total_duration += c.duration

        def _cleanup_line(course, total):
            aaid = course.analytic_account_id
            if not isinstance(aaid, (int, long, bool)):
                aaid = aaid.id
            return {
                'analytic_account_id': aaid,
                'rate': course.duration * 100 / total
            }

        # Create 'analytic distribution instance'
        adist_id = self.pool.get('account.analytic.plan.instance').create(cr, uid, {
            'journal_id': journal.analytic_journal_id.id,
            'account_ids': [ (0, 0, _cleanup_line(course, total_duration)) for course in courses ],
        })
        return adist_id

    # training.subscription.line
    def _invoicing_get_grouping_key(self, cr, uid, ids, subline, context=None):
        if context is None:
            context = {}
        if context.get('group_by_subscription') is True:
            return (subline.subscription_id.partner_id.id,
                    subline.subscription_id.payment_term_id.id,
                    subline.subscription_id.id)
        else:
            return (subline.subscription_id.partner_id.id,
                    subline.subscription_id.payment_term_id.id,
                    subline.session_id.id)

    # training.subscription.line
    def _invoicing_get_grouping_data(self, cr, uid, ids, key, subline, journal=None, context=None):
        if context is None:
            context = {}
        # setup context
        session = subline.session_id
        partner = subline.partner_id
        subscription = subline.subscription_id
        if journal is None:
            journal = self.pool.get('account.journal')._get_journal_record_by_type(cr, uid, type='sale')

        # generic fields
        data = {
            'type' : 'out_invoice',
            'partner_id' : partner.id,
            'address_contact_id' : partner.notif_contact_id and partner.notif_contact_id.address_id.id,
            'address_invoice_id' : subscription.address_id.id,
            'journal_id': journal.id,
            'account_id' : partner.property_account_receivable.id,
            'fiscal_position' : partner.property_account_position.id,
            'date_invoice' : time.strftime('%Y-%m-%d'),
            'payment_term' : subscription.payment_term_id.id,
        }

        if context.get('group_by_subscription') is True:
            data.update({
                'name': (subscription.custom_ref or '').strip() or subscription.name,
                'origin': subscription.name,
                'reference': '%s - %s' % (partner.name, subscription.name)
            })
        else:
            data.update({
                'name': session.name,
                'origin': session.name,
                'reference': "%s - %s" % (partner.name, session.name),
            })
        if context.get('cancellation', False) is True:
            data['name'] += ' - ' + _('Cancellation')
        return data

    # training.subscription.line
    def _invoicing_get_line_data(self, cr, uid, ids, subline, invoice, context=None):
        if context is None:
            context = {}
        account_code = self.pool.get('ir.config').get(cr, uid, 'training.subscription.invoice.default.account')
        if not account_code:
            raise osv.except_osv(_('Error!'), _("Not default invoice account specified, please configure it inside system options"))
        account_id = self.pool.get('account.account').search(cr, uid, [('code', '=', account_code)])[0]
        # setup context
        session  = subline.session_id
        offer_product = session.offer_id.product_id
        partner = subline.partner_id
        contact = subline.contact_id
        fpos = invoice.fiscal_position
        # create data needed from account.invoice.line creation
        data = {
            'invoice_id' : invoice.id,
            'product_id' : offer_product.id,
            'quantity' : 1,
            'name' : '%s %s %s' % (session.name, contact.first_name, contact.name),
            'account_id' : offer_product.property_account_income and offer_product.property_account_income.id or account_id,
            'origin' : subline.name,
            'price_unit' : subline.price,
            'discount' : context and context.get('discount', 0.0) or 0.0,
            'invoice_line_tax_id': self._get_invoice_line_taxes(cr, uid, subline, fpos, partner, session, context=context),
            'account_analytic_id': '',
            'analytics_id': '',  # Analytic Distribution
        }
        return data

    # training.subscription.line
    def action_create_invoice(self, cr, uid, ids, context=None):
        # Get journal
        journal_sales = self.pool.get('account.journal')._get_journal_record_by_type(cr, uid, type='sale')

        proxy_seance = self.pool.get('training.seance')
        proxy_invoice = self.pool.get('account.invoice')
        proxy_invoice_line = self.pool.get('account.invoice.line')
        workflow = netsvc.LocalService('workflow')

        partners = {}
        for sl in self.browse(cr, uid, ids, context=context):
            if sl.price >= -0.00001 and sl.price <= 0.00001:
                # Ignore subscription line with a sale price of ZERO
                continue
            if sl.invoice_line_id:
                continue
            key = self._invoicing_get_grouping_key(cr, uid, [], sl, context)
            partners.setdefault(key, []).append(sl)

        invoice_ids = []

        for k, subscription_lines in partners.items():
            invoice_values = self._invoicing_get_grouping_data(cr, uid, [], k, subscription_lines[0], journal_sales, context)

            invoice_id = proxy_invoice.create(cr, uid, invoice_values)
            invoice = proxy_invoice.browse(cr, uid, invoice_id, context=context)

            global_courses = self._get_courses(cr, uid, [sl.id for sl in subscription_lines], context=context)
            for sl in subscription_lines:
                values = self._invoicing_get_line_data(cr, uid, [], sl, invoice, context)

                self._invoice_min_max(cr, uid, values, context)

                adist_id = self._invoicing_get_analytic_distribution_by_record(cr, uid, sl, global_courses[sl.id], journal_sales, context=context)
                values['analytics_id'] = adist_id

                invoice_line_id = proxy_invoice_line.create(cr, uid, values, context=context)
                sl.write({'invoice_line_id' : invoice_line_id})

            proxy_invoice.button_compute(cr, uid, [invoice_id])
            invoice_ids.append(invoice_id)

        return invoice_ids

    # training.subscription.line
    def action_workflow_done(self, cr, uid, ids, context=None):
        self.write(cr, uid, ids, {'state' : 'done'}, context=context)

        workflow = netsvc.LocalService('workflow')
        for line in self.browse(cr, uid, ids, context):
            workflow.trg_validate(uid, 'training.subscription', line.subscription_id.id, 'signal_done_cancel', cr)
        return True

    # training.subscription.line
    def action_workflow_cancel(self, cr, uid, ids, context=None):
        """cancellation of the subscription line - cleanup"""
        workflow = netsvc.LocalService('workflow')

        self.send_email(cr, uid, ids, 'sub_cancelled', context)
        self.write(cr, uid, ids, {'state': 'cancelled', 'cancellation_date': time.strftime('%Y-%m-%d %H:%M:%S'), 'cancellation_uid': uid}, context=context)

        self._delete_participations(cr, uid, ids, context)

        for line in self.browse(cr, uid, ids, context):
            workflow.trg_validate(uid, 'training.subscription', line.subscription_id.id, 'signal_done_cancel', cr)
        return True

    def send_email(self, cr, uid, ids, trigger, context=None, **objects):
        """Send email to participant and to HR (grouped by session)"""
        email_proxy = self.pool.get('training.email')
        groups = {}
        email_to_subs = {}
        for sl in self.browse(cr, uid, ids, context=context):
            if sl.subscription_id.notification_active:
                partner = sl.job_id.name
                if partner.notif_participant:
                    email_proxy.send_email(cr, uid, trigger, 'p', sl.job_email, session=sl.session_id, stylegroup=sl.session_id.stylegroup_id, context=context, partner=partner, subline=sl, **objects)

                hremail = sl.partner_hr_email
                # do not group HR emails...
                email_proxy.send_email(cr, uid, trigger, 'hr', hremail, session=sl.session_id, stylegroup=sl.session_id.stylegroup_id, context=context, subline=sl, **objects)
        return True

training_subscription_line()

class training_participation_stakeholder(osv.osv):
    _name = 'training.participation.stakeholder'
training_participation_stakeholder()

class training_request_payment_mode(osv.osv):
    _name = 'training.request.payment.mode'

    _invoice_methods_selection = [
        ('manual','Manual'),
        ('order','From Order'),
        ('picking','From Picking'),
    ]

    _columns = {
        'code': fields.char('Code', size=16, required=True),
        'name': fields.char('Payment Mode', size=32, required=True, translate=True),
        'fiscal_position_id': fields.many2one('account.fiscal.position', 'Fiscal Position',
                                help='The fiscal position will be set on the request purchase order'),
        'invoice_method': fields.selection(_invoice_methods_selection, 'Invoice Method',
                                help='The invoice method will be set on the request purchase order'),
    }
training_request_payment_mode()

class training_participation_stakeholder_purchase(osv.osv):
    _name = 'training.participation.stakeholder.purchase'
    _rec_name = 'seance_id'

    def product_id_change(self, cr, uid, ids, payment_mode_id, product_id):
        if not product_id:
            return False

        if not payment_mode_id:
            raise osv.except_osv(_('Error'), _('Please choose a Payment Mode on the request, before adding extra purchase line'))

        payment_mode = self.pool.get('training.request.payment.mode').browse(cr, uid, payment_mode_id)

        fiscalpos_obj = self.pool.get('account.fiscal.position')
        product_obj = self.pool.get('product.product')
        product =  product_obj.browse(cr, uid, [product_id])[0]
        result = {
            'value': {
                'price_unit': product.list_price or 0.0,
                'desc': product.product_tmpl_id.name or'',
                'product_uom' : product.uom_po_id.id or False,
            }
        }
        if product.supplier_taxes_id:
            fpos = payment_mode.fiscal_position_id or False
            result['value']['tax_ids'] = fiscalpos_obj.map_tax(cr, uid, fpos, product.supplier_taxes_id)

        return result

    def _total_compute(self, cr, uid, ids, fieldnames, args, context=None):
        res = dict.fromkeys(ids, 0.0)
        for this in self.browse(cr, uid, ids, context=context):
            res[this.id] = this.product_qty * this.price_unit
        return res

    _columns = {
        'request_id': fields.many2one('training.participation.stakeholder.request'),
        'request_session_id': fields.related('request_id', 'session_id', type='many2one', relation='training.session', readonly=True),
        'job_id': fields.related('request_id', 'job_id', string="Contact", type='many2one', relation='res.partner.job', readonly=True),
        'product_id': fields.many2one('product.product', 'Product'),
        'product_qty': fields.float('Quantity', required=True, digits=(16,2)),
        'product_uom': fields.many2one('product.uom', 'Product UOM', required=True),
        'desc': fields.char('Description', size=256, required=True),
        'price_unit': fields.float('Unit Price', required=True, digits_compute= dp.get_precision('Purchase Price')),
        'seance_id' : fields.many2one('training.seance', 'Seance', required=True, help='Select the Seance', ondelete='cascade',
                                      domain="[('date', '>=', time.strftime('%Y-%m-%d'))]"),
        'total' : fields.function(_total_compute, method=True, string='Total', type='float', digits_compute=dp.get_precision('Account')),
        'tax_ids': fields.many2many('account.tax', 'training_participation_stakeholder_purchase_tax_rel',
                                    'request_purchase_id', 'tax_id', 'Taxes', domain=[('parent_id','=',False),('type_tax_use','in',['purchase','all'])]),
    }

training_participation_stakeholder_purchase()

class training_participation_stakeholder_request(osv.osv):
    _name = 'training.participation.stakeholder.request'
    _rec_name = 'reference'

    def _store_get_requests(self, cr, uid, ids, context=None):
        keys = set()
        for this in self.pool.get('training.participation.stakeholder').browse(cr, uid, ids, context=context):
            if this.request_id:
                keys.add(this.request_id.id)

        return list(keys)

    def _store_purchase_get_requests(self, cr, uid, ids, context=None):
        keys = set()
        purchaseline_obj = self.pool.get('training.participation.stakeholder.purchase')
        for purchaseline in purchaseline_obj.browse(cr, uid, ids, context=context):
            if purchaseline.request_id:
                keys.add(purchaseline.request_id.id)
        return list(keys)

    def _price_compute(self, cr, uid, ids, fieldnames, args, context=None):
        res = dict.fromkeys(ids, 0.0)
        for this in self.browse(cr, uid, ids, context=context):
            res[this.id] = sum(p.price for p in this.participation_ids) + sum(p.total for p in this.purchase_ids)
        return res

    def _date_compute(self, cr, uid, ids, fieldnames, args, context=None):
        res = dict.fromkeys(ids, False)
        for this in self.browse(cr, uid, ids, context=context):
            if this.participation_ids:
                res[this.id] = min(p.seance_id.date for p in this.participation_ids)
        return res

    def _amount_to_pay(self, cr, uid, ids, name, args, context=None):
        res = dict.fromkeys(ids, 'N/A')
        if not ids:
            return res

        invoice_pool = self.pool.get('account.invoice')
        request_invoices = {}

        for request in self.browse(cr, uid, ids, context=context):
            try:
                invoice_ids = request.purchase_order_id.invoice_ids
                if invoice_ids:
                    for invoice_id in request.purchase_order_id.invoice_ids:
                        request_invoices[invoice_id.id] = request.id
            except AttributeError:
                # catch: "bool object doesn't have 'x' attributte" errors
                pass

        invoice_residual_amounts = invoice_pool._amount_to_pay(cr, uid, request_invoices.keys(), None, None, context)
        for invoice_id, residual_amount in invoice_residual_amounts.items():
            res[request_invoices[invoice_id]] = unicode(residual_amount)
        return res

    _columns = {
        'reference': fields.char('Reference', size=16, required=True, **WRITABLE_ONLY_IN_DRAFT),
        'session_id': fields.many2one('training.session', 'Session', required=True, **WRITABLE_ONLY_IN_DRAFT),
        'date': fields.function(_date_compute, method=True, string='First Seance Date', type='datetime',
                                store={
                                    'training.participation.stakeholder': (_store_get_requests, None, 25),
                                }),
        'kind' : fields.related('session_id', 'offer_id', 'kind', type='char', string="Offer's Kind", **WRITABLE_ONLY_IN_DRAFT),
        'job_id': fields.many2one('res.partner.job', 'Contact', required=True, **WRITABLE_ONLY_IN_DRAFT),
        'contact_id': fields.related('job_id', 'contact_id', type='many2one', relation='res.partner.contact', readonly=True, store=True),
        'email' : fields.char('Email', size=128, **WRITABLE_ONLY_IN_DRAFT),
        'payment_mode_id': fields.many2one('training.request.payment.mode', 'Payment Mode', **WRITABLE_ONLY_IN_DRAFT),
        'participation_ids': fields.one2many('training.participation.stakeholder', 'request_id', 'Participations', **WRITABLE_ONLY_IN_DRAFT),
        'purchase_ids': fields.one2many('training.participation.stakeholder.purchase', 'request_id', 'Purchases', **WRITABLE_ONLY_IN_DRAFT),
        'notes' : fields.text('Notes', **WRITABLE_ONLY_IN_DRAFT),
        'state': fields.selection([('draft', 'Draft'),
                                   ('valid', 'Validated'),
                                   ('requested', 'Requested'),
                                   ('accepted', 'Accepted'),
                                   ('refused', 'Refused'),
                                   ('cancelled', 'Cancelled'),
                                   ('done', 'Done'),
                                  ], 'State', readonly=True, required=True),
        'purchase_order_id': fields.many2one('purchase.order', 'Purchase Order', readonly=True),
        'purchase_paid': fields.related('purchase_order_id', 'invoice_ids', 'reconciled', type='boolean', string='Invoice Paid', readonly=True,),
        'amount_to_pay': fields.function(_amount_to_pay, string='Amount to pay', type='char', size=20, method=True),
        'invoiced': fields.related('purchase_order_id', 'invoiced', type='boolean', string='Invoiced', readonly=True),
        'price' : fields.function(_price_compute, method=True, string='Remuneration', type='float', digits_compute=dp.get_precision('Account'),
                                  store={
                                      'training.participation.stakeholder': (_store_get_requests, None, 25),
                                      'training.participation.stakeholder.purchase': (_store_purchase_get_requests, None, 25),
                                  }),
    }

    _defaults = {
        'state': lambda *a: 'draft',
        'reference' : lambda *args: '/',
    }

    def create(self, cr, uid, values, context=None):
        if values.get('reference', '/') == '/':
            values['reference'] = self.pool.get('ir.sequence').get(cr, uid, 'training.pshr')
        return super(training_participation_stakeholder_request, self).create(cr, uid, values, context)

    def copy(self, cr, uid, id, default=None, context=None):
        if not default:
            default = {}
        default = default.copy()
        default.update({
            'reference': '/',
            'purchase_order_id': False,
            'participation_ids' : [],
        })
        return super(training_participation_stakeholder_request, self).copy(cr, uid, id, default=default,
                    context=context)


    def on_change_job(self, cr, uid, ids, job_id, context=None):
        if not job_id:
            return False

        job = self.pool.get('res.partner.job').browse(cr, uid, job_id, context=context)
        return {
            'value' : {
                'email': job.email,
                'manual_price': False,
            }
        }

    @tools.cache(skiparg=4)
    def _get_stock_location(self, cr, uid, context=None):
        purchase_order_pool = self.pool.get('purchase.order')
        default_warehouse_id = purchase_order_pool.default_get(cr, uid, ['warehouse_id'], context=context).get('warehouse_id')
        if not default_warehouse_id:
            return False
        location_id = self.pool.get('stock.warehouse').browse(cr, uid, default_warehouse_id, context=context).lot_input_id.id
        return location_id

    def _prepare_purchase_order(self, cr, uid, request, context=None):
        values = {
            'name': "Session - %s" % (request.session_id.name),
            'partner_id': request.job_id.name.id,
            'contact_id': request.job_id.contact_id.id,
            'partner_address_id': request.job_id.address_id.id,
            'pricelist_id': request.job_id.name.property_product_pricelist_purchase.id,
            'location_id': self._get_stock_location(cr, uid, context),
            'invoice_method' : request.payment_mode_id.invoice_method,
            'fiscal_position': request.payment_mode_id.fiscal_position_id.id,
        }
        return values

    def action_create_purchase_order(self, cr, uid, ids, context=None):
        purchase_obj = self.pool.get('purchase.order')
        purchase_line_obj = self.pool.get('purchase.order.line')
        stkh_participation_obj = self.pool.get('training.participation.stakeholder')

        for request in self.browse(cr, uid, ids, context=context):
            if not request.payment_mode_id:
                raise osv.except_osv(_('Error'),
                                     _('The Payment Mode is not specified'))
            if not request.job_id.name.property_product_pricelist_purchase:
                raise osv.except_osv(_('No Pricelist'),_('You have to select a pricelist for this contact'))

            purchase_values = self._prepare_purchase_order(cr, uid, request, context=context)
            new_order_id = purchase_obj.create(cr, uid, purchase_values, context=context)
            request.write({'purchase_order_id' : new_order_id})
            order = purchase_obj.browse(cr, uid, new_order_id, context=context)

            for part in request.participation_ids:
                values = stkh_participation_obj._prepare_purchase_order_line(cr, uid, part, order, context=context)
                new_order_line_id = purchase_line_obj.create(cr, uid, values, context=context)
                part.write({'purchase_order_line_id' : new_order_line_id})

            for purch in request.purchase_ids:
                tax_ids = False
                if purch.tax_ids:
                    tax_ids = [(6, 0, [t.id for t in purch.tax_ids])]

                purchase_line_obj.create(cr, uid, {
                    'order_id': values['order_id'],
                    'product_id': purch.product_id.id,
                    'product_qty': purch.product_qty,
                    'product_uom': purch.product_uom.id,
                    'name': purch.desc,
                    'price_unit': purch.price_unit,
                    'seance_id': purch.seance_id.id,
                    'date_planned': purch.seance_id.date,
                    'taxes_id': tax_ids,
                    }, context=context)
        return True

    def _trigger_purchase_order_signal(self, cr, uid, ids, signal_name, context=None):
        wf_service = netsvc.LocalService('workflow')
        for this in self.browse(cr, uid, ids, context=context):
            wf_service.trg_validate(uid, 'purchase.order', this.purchase_order_id.id, signal_name, cr)
        return True

    def _send_request_email(self, cr, uid, ids, email_type, context=None, **kwargs):
        email_proxy = self.pool.get('training.email')
        for request in self.browse(cr, uid, ids, context=context):
            seances = [ sh.seance_id for sh in request.participation_ids ]
            seances.sort(cmp=self.sh_sort_by_date)
            email_proxy.send_email(cr, uid, email_type, 'sh', request.email, session=request.session_id, context=context, seances=seances, request=request, **kwargs)
        return True

    def _spread_wkf_signal(self, cr, uid, ids, signal, context=None):
        workflow = netsvc.LocalService('workflow')
        for this in self.browse(cr, uid, ids, context):
            for sh in this.participation_ids:
                workflow.trg_validate(uid, 'training.participation.stakeholder', sh.id, signal, cr)
        return True

    def action_wkf_cancel(self, cr, uid, ids, context=None):
        self.write(cr, uid, ids, {'state': 'cancelled'}, context=context)
        self._trigger_purchase_order_signal(cr, uid, ids, 'purchase_cancel', context=context)
        self._spread_wkf_signal(cr, uid, ids, 'signal_cancel', context)
        return True

    def test_wkf_valid(self, cr, uid, ids, context=None):
        res = all(bool(request.purchase_order_id) != False for request in self.browse(cr, uid, ids, context=context))
        if not res:
            raise osv.except_osv(_('Warning'),_('You can not validate request without Purchase Order'))
        return res

    def action_wkf_valid(self, cr, uid, ids, context=None):
        self._trigger_purchase_order_signal(cr, uid, ids, 'purchase_confirm', context=context)
        return self.write(cr, uid, ids, {'state': 'valid'}, context=context)

    def test_wkf_request(self, cr, uid, ids, context=None):
        return all(bool(request.purchase_order_id) != False for request in self.browse(cr, uid, ids, context=context))

    def action_wkf_request(self, cr, uid, ids, context=None):
        return self.write(cr, uid, ids, {'state': 'requested'}, context=context)

    def test_wkf_accept(self, cr, uid, ids, context=None):
        return all(bool(request.purchase_order_id) != False for request in self.browse(cr, uid, ids, context=context))

    def sh_sort_by_date(self, x, y):
        """Used for sorting list of browse_record() by date
           Usage: List.sort(cmp=sh_short_by_date)"""
        if x.date < y.date:
            return -1
        if x.date > y.date:
            return 1
        return 0

    def action_wkf_accept(self, cr, uid, ids, context=None):
        report_invoice = netsvc.LocalService('report.account.invoice')
        self.write(cr, uid, ids, {'state': 'accepted'}, context=context)
        self._trigger_purchase_order_signal(cr, uid, ids, 'purchase_approve', context=context)
        self._spread_wkf_signal(cr, uid, ids, 'signal_accept', context)

        email_proxy = self.pool.get('training.email')
        for this in self.browse(cr, uid, ids, context=context):
            email_attachments = []
            if this.purchase_order_id and this.purchase_order_id.invoice_ids:
                invoice_ids = []
                # attach the draft invoice for this stakeholder request
                for invoice_id in this.purchase_order_id.invoice_ids:
                    invoice_ids.append(invoice_id.id)
                pdf, _r = report_invoice.create(cr, uid, invoice_ids, {}, context=context)
                email_attachments.append(('Invoice.pdf', pdf))
            self._send_request_email(cr, uid, ids, 'sh_accept', context=context, attachments=email_attachments)
            if this.session_id.state in ('opened_confirmed', 'closed_confirmed'):
                # session already confirmed, force sending of 'session confirmation'
                self._send_request_email(cr, uid, ids, 'session_open_confirmed', context=context)
        return True

    def action_wkf_send_request_email(self, cr, uid, ids, context=None):
        return self._send_request_email(cr, uid, ids, 'sh_request', context=context)

    def action_wkf_send_cancellation_email(self, cr, uid, ids, context=None):
        return self._send_request_email(cr, uid, ids, 'sh_cancel', context=context)

    def action_wkf_refuse(self, cr, uid, ids, context=None):
        self.write(cr, uid, ids, {'state': 'refused'}, context=context)
        self._trigger_purchase_order_signal(cr, uid, ids, 'purchase_cancel', context=context)
        self._spread_wkf_signal(cr, uid, ids, 'signal_refuse', context)
        return self._send_request_email(cr, uid, ids, 'sh_refuse', context=context)

    def test_wkf_done(self, cr, uid, ids, context=None):
        return all(participation.state == 'done' for request in self.browse(cr, uid, ids, context) for participation in request.participation_ids)

    def action_wkf_done(self, cr, uid, ids, context=None):
        return self.write(cr, uid, ids, {'state': 'done'}, context=context)

training_participation_stakeholder_request()

class training_participation_stakeholder(osv.osv):
    _name = 'training.participation.stakeholder'
    _rec_name = 'job_id'

    def name_get(self, cr, user, ids, context=None):
        rec_ffield = lambda x: isinstance(x[self._rec_name], (tuple,list)) and x[self._rec_name][1] or x[self._rec_name]
        res = [(x['id'],rec_ffield(x)) for x in self.read(cr, user, ids, [self._rec_name], context, load='_classic_write')]
        return res

    def on_change_job(self, cr, uid, ids, job_id, context=None):
        if not job_id:
            return False

        job = self.pool.get('res.partner.job').browse(cr, uid, job_id, context=context)
        return {'value' : {
            'manual_price': False,
        }}

    def on_change_seance(self, cr, uid, _, job_id, seance_id, context=None):
        seance = seance_id and self.pool.get('training.seance').browse(cr, uid, seance_id, context=context) or False
        return {'value': {
            'group_id': seance and seance.group_id.id or False,
            'date': seance and seance.date or False,
            'kind': seance and seance.kind or False,
            'course_id': seance and seance.course_id.id or False,
            'duration': seance and seance.duration or False,
            'state_seance': seance and seance.state or False,
            'product_id': seance and  seance._get_product() and seance._get_product().id or False,

            'price': self._default_price_compute(cr, uid, job_id, seance, context=context)
        }}

    def on_change_manual(self, cr, uid, ids, manual, job_id, seance_id, context=None):
        if not manual:
            return {}
        return {'value': {
            'forced_price': self._default_price_compute(cr, uid, job_id, seance_id, context=context),
        }}

    def on_change_product(self, cr, uid, ids, job_id, seance_id, product_id, context=None):
        return {'value': {
            'price': self._default_price_compute(cr, uid, job_id, seance_id, product_id, context)
        }}

    def _default_price_compute(self, cr, uid, job, seance, product_id=None, context=None):
        if not job or not seance:
            return False

        if isinstance(seance, (int, long)):
            seance = self.pool.get('training.seance').browse(cr, uid, seance, context=context)

        course = seance.course_id
        if not course:
            return False

        if product_id and isinstance(product_id, (int,long)):
            product = self.pool.get('product.product').browse(cr, uid, product_id, context=context)
        else:
            product = seance._get_product()
        if not product:
            raise osv.except_osv(_('Error'),
                                 _("The type of the course (%s) of this seance has no product defined") % course.name)

        if isinstance(job, (int, long)):
            job = self.pool.get('res.partner.job').browse(cr, uid, job, context=context)

        pricelist = job.name.property_product_pricelist_purchase
        if not pricelist:
            # no pricelist available: use the product cost price
            unit_price = product.standard_price
        else:
            pricelists = self.pool.get('product.pricelist')
            unit_price = pricelists.price_get(cr, uid, [pricelist.id], product.id, seance.duration, job.name)[pricelist.id]

        return unit_price * seance.duration

    def _get_price(self, cr, uid, ids, fieldnames, args, context=None):
        res = dict.fromkeys(ids, 0.0)
        for this in self.browse(cr, uid, ids, context=context):
            if this.manual_price:
                res[this.id] = this.forced_price
            else:
                prod_id = this.product_id and this.product_id.id or None
                res[this.id] = self._default_price_compute(cr, uid, this.job_id, this.seance_id, product_id=prod_id, context=context)
        return res

    _columns = {
        'request_id': fields.many2one('training.participation.stakeholder.request', 'Request', required=True, ondelete='cascade'),
        'request_session_id': fields.related('request_id', 'session_id', type='many2one', relation='training.session', readonly=True),
        'seance_id' : fields.many2one('training.seance', 'Seance', required=True, help='Select the Seance', ondelete='cascade',
                                      domain="[('date', '>=', time.strftime('%Y-%m-%d'))]"),
        'group_id' : fields.related('seance_id', 'group_id', type='many2one', relation='training.group', readonly=True, store=True, string='Group'),
        'job_id': fields.related('request_id', 'job_id', string="Contact", type='many2one', relation='res.partner.job', readonly=True),
        'contact_id' : fields.related('job_id', 'contact_id', type='many2one', relation='res.partner.contact', readonly=True, store=True),
        'partner_id' : fields.related('job_id', 'name', type='many2one', relation='res.partner', readonly=True, store=True, string="Partner"),
        'date' : fields.related('seance_id', 'date', type='datetime', string='Date', readonly=True, store=True),
        'kind' : fields.related('seance_id', 'kind', type='selection', selection=[('standard', 'Course')], string='Kind', readonly=True),
        'course_id' : fields.related('seance_id', 'course_id', type='many2one', relation='training.course', string='Course', readonly=True),
        'state' : fields.selection([('draft', 'Draft'),
                                    ('accepted', 'Accepted'),
                                    ('refused', 'Refused'),
                                    ('cancelled', 'Cancelled'),
                                    ('done', 'Done'),
                                   ], 'State', required=True, readonly=True),
        'duration' : fields.related('seance_id', 'duration', type='float', string='Duration', readonly=True, store=True),
        'state_seance' : fields.related('seance_id', 'state', type='selection',
                                        selection=[('opened', 'Opened'),
                                                   ('confirmed', 'Confirmed'),
                                                   ('inprogress', 'In Progress'),
                                                   ('closed', 'Closed'),
                                                   ('cancelled', 'Cancelled'),
                                                   ('done', 'Done')
                                                  ], string='State of Seance', readonly=True),
        'purchase_order_line_id' : fields.many2one('purchase.order.line', 'Purchase Order Line', readonly=True),
        'purchase_order_id' : fields.related('purchase_order_line_id', 'order_id', string='Purchase Order', type='many2one', relation='purchase.order', readonly=True),
        'paid' : fields.related('purchase_order_id', 'invoiced', type='boolean', string='Invoiced & Paid', readonly=True, select=1),
        'manual_price': fields.boolean('Manual Price'),
        'forced_price': fields.float('Renumeration', required=True, digits_compute=dp.get_precision('Account')),
        'price' : fields.function(_get_price, method=True, string='Remuneration', type='float', digits_compute=dp.get_precision('Account'), store=True),
        'product_id' : fields.many2one('product.product', 'Product'),
    }

    _defaults = {
        'manual_price': lambda *a: False,
    }

    def _pol_get_analytic_account(self, session_record, seance_record, context=None):
        return seance_record.course_id.analytic_account_id  \
                and seance_record.course_id.analytic_account_id.id or ''

    def _prepare_purchase_order_line(self, cr, uid, stkh_participation, order, context=None):
        fiscal_position_obj = self.pool.get('account.fiscal.position')
        obj = stkh_participation
        product = obj.seance_id._get_product()
        if not product:
            raise osv.except_osv(_('Error'),
                                 _("The type of the course (%s) of this seance hasn't a product") % obj.seance_id.course_id.name)

        price = obj.price
        qty = obj.seance_id.duration * 1.0

        analytics_id = False
        account_analytic_id = self._pol_get_analytic_account(obj.request_id.session_id, obj.seance_id, context=context)
        if account_analytic_id:
            prsh_analytic_journal_ids = self.pool.get('account.analytic.journal').search(cr, uid, [('type','=','purchase')])
            if not prsh_analytic_journal_ids:
                raise osv.except_osv(_('Error'), _('No purchase analytic journal defined'))
            analytics_id = self.pool.get('account.analytic.plan.instance').create(cr, uid, {
                'journal_id': prsh_analytic_journal_ids[0],
                'account_ids': [
                    (0, 0, {
                        'analytic_account_id': account_analytic_id,
                        'rate': 100.0,
                    }),
                ],
            })
        # update purchase order line taxes
        supplier_tax_ids = fiscal_position_obj.map_tax(cr, uid, order.fiscal_position, product.supplier_taxes_id, context=context)

        values = {
            'name' : 'Seance %s - %s' % (obj.seance_id.name, obj.seance_id.date),
            'date_planned' : obj.seance_id.date,
            'order_id' : order.id,
            'price_unit' : price / qty,
            'seance_id' : obj.seance_id.id,
            'product_qty' : qty,
            'product_id' : product.id,
            'product_uom' : product.uom_id.id,
            'taxes_id': [(6, 0, supplier_tax_ids)],
#           'account_analytic_id': self._pol_get_analytic_account(obj.request_id.session_id, obj.seance_id, context=context),
            'analytics_id': analytics_id,
        }
        return values

    def _check_disponibility(self, cr, uid, ids, context=None):
        for this in self.browse(cr, uid, ids, context=context):
            contact = this.job_id.contact_id
            cr.execute("""SELECT s.id
                            FROM training_participation_stakeholder sh INNER JOIN training_seance s ON sh.seance_id = s.id,
                                 training_participation_stakeholder this INNER JOIN training_seance this_seance ON this.seance_id = this_seance.id
                           WHERE sh.id != %s
                             AND this.id = %s
                             AND sh.state in ('requested', 'confirmed')
                             AND sh.contact_id = %s
                             AND (     (this_seance.date BETWEEN s.date AND s.date + (s.duration * interval '1 hour'))
                                  OR (this_seance.date + (this_seance.duration * interval '1 hour') BETWEEN s.date AND s.date + (s.duration * interval '1 hour'))
                                  OR (this_seance.date < s.date AND this_seance.date + (this_seance.duration * interval '1 hour') > s.date + (s.duration * interval '1 hour'))
                                  )
                       """, (this.id, this.id, contact.id))
            res = cr.fetchone()
            if res:
                name = '%s %s' % (contact.first_name, contact.name)
                other_seance = self.pool.get('training.seance').browse(cr, uid, res[0], context=context)

                raise osv.except_osv(_('Error'), _('%(stakeholder)s is not available for seance "%(this_seance)s" because (s)he is already requested or confirmed for the seance "%(other_seance)s"') % {'stakeholder':name, 'this_seance': this.seance_id.name, 'other_seance': other_seance.name})

        return True

    def send_email(self, cr, uid, ids, trigger, session, context=None, objects_by_ids=None):
        if objects_by_ids is None:
            objects_by_ids = {}
        email_proxy = self.pool.get('training.email')
        send_mail = False
        for sh in self.browse(cr, uid, ids, context=context):
            if sh.request_id and sh.request_id.email:
                send_mail = email_proxy.send_email(cr, uid, trigger, 'sh', sh.request_id.email, session=session, context=context, sh=sh, **objects_by_ids.get(sh.id, {}))
        return send_mail

    def _test_wkf(self, cr, uid, ids, state, context=None):
        for this in self.browse(cr, uid, ids, context):
            if this.request_id:
                if this.request_id.state != state:
                    return False
        return True

    def _action_wkf(self, cr, uid, ids, state, purchase_order_signal, context=None):
        # First, validate or cancel the purcchase order line (at least try to)
        wkf = netsvc.LocalService('workflow')
        for this in self.browse(cr, uid, ids, context):
            po = this.purchase_order_id
            if po:
                if purchase_order_signal == 'purchase_cancel' and po.invoice_ids:
                    for invoice_id in po.invoice_ids:
                        wkf.trg_validate(uid, 'account.invoice', invoice_id.id, 'invoice_cancel', cr)
                    wkf.trg_validate(uid, 'purchase.order', po.id, 'cancel', cr)
                else:
                    wkf.trg_validate(uid, 'purchase.order', this.purchase_order_id.id, purchase_order_signal, cr)

        # Second, change the participation state
        self.write(cr, uid, ids, {'state': state}, context=context)
        return True

    def action_wkf_cancel(self, cr, uid, ids, context=None):
        return self._action_wkf(cr, uid, ids, 'cancelled', 'purchase_cancel', context)

    def test_wkf_accept(self, cr, uid, ids, context=None):
        return self._test_wkf(cr, uid, ids, 'accepted', context)

    def action_wkf_accept(self, cr, uid, ids, context=None):
        return self._action_wkf(cr, uid, ids, 'accepted', 'purchase_approve', context)

    def test_wkf_refuse(self, cr, uid, ids, context=None):
        return self._test_wkf(cr, uid, ids, 'refused', context)

    def action_wkf_refuse(self, cr, uid, ids, context=None):
        return self._action_wkf(cr, uid, ids, 'refused', 'purchase_cancel', context)

    def test_wkf_done(self, cr, uid, ids, context=None):
        return all(participation.state_seance == 'done' for participation in self.browse(cr, uid, ids, context))

    def action_wkf_done(self, cr, uid, ids, context=None):
        self.write(cr, uid, ids, {'state': 'done'}, context=context)

        # spread the signal to the request
        wkf = netsvc.LocalService('workflow')
        for this in self.browse(cr, uid, ids, context):
            wkf.trg_validate(uid, 'training.participation.stakeholder.request', this.request_id.id, 'pshr_done', cr)

        return True

    def _check_request(self, cr, uid, ids, context=None):
        return all(this.request_id.session_id in this.seance_id.session_ids for this in self.browse(cr, uid, ids, context))

    def _check_unique_seance_lecturer(self, cr, uid, ids, context=None):
        for part in self.browse(cr, uid, ids, context=context):
            if part.state == 'cancelled':
                continue
            part_unique_domain = [
                ('seance_id','=',part.seance_id.id),
                ('contact_id','=',part.job_id.contact_id.id),
                ('state','!=','cancelled'),
                ('id','!=',part.id),
            ]
            if self.search(cr, uid, part_unique_domain, context=context, count=True) > 0:
                # another line already exist
                return False
        return True

    _constraints = [
#        (lambda self, *args, **kwargs: self._check_disponibility(*args, **kwargs), "Please, Can you check the disponibility of the stakeholder", ['date', 'seance_id', 'contact_id']),
         (_check_request, "The request is not linked to a session of the seance", ['seance_id', 'request_id']),
         (_check_unique_seance_lecturer, "The contact is already associated to the seance", ['seance_id', 'contact_id', 'state']),
    ]

    def _state_default(self, cr, uid, context=None):
        return 'draft'

    _defaults = {
        #'state' : lambda *a: 'draft',
        'state' : _state_default,
    }

    _order = "date asc"

training_participation_stakeholder()

class training_contact_course(osv.osv):
    _name = 'training.contact.course'
    _auto = False

    _columns = {
        'function' : fields.char('Function', size=64, readonly=True),
        'course_id' : fields.many2one('training.course', 'Course', readonly=True),
        'contact_id' : fields.many2one('res.partner.contact', 'Contact', readonly=True),
    }

    def init(self, cr):
        tools.drop_view_if_exists(cr, 'training_contact_course')
        cr.execute("CREATE OR REPLACE VIEW training_contact_course as ( "
                   "SELECT rel.id as id, course.id AS course_id, contact.id AS contact_id, COALESCE(func.name,'') AS function "
                   "FROM res_partner_contact AS contact "
                   "LEFT JOIN res_partner_job job ON (job.contact_id = contact.id) "
                   "LEFT JOIN training_course_job_rel rel ON (rel.job_id = job.id) "
                   "LEFT JOIN training_course course ON (rel.course_id = course.id) "
                   "LEFT JOIN res_partner_function func ON (job.function_id = func.id) "
                   "GROUP BY rel.id, course.id, contact.id, func.name "
                   "HAVING rel.id IS NOT NULL "
                   ")")

training_contact_course()

class res_partner_contact(osv.osv):
    _inherit = 'res.partner.contact'

    _columns = {
        'course_ids' : fields.one2many('training.contact.course', 'contact_id', 'Courses', readonly=True),
        # participation_ids: needed for ir.rule usability!
        'participation_ids': fields.one2many('training.participation', 'contact_id', 'Participations', readonly=True),
    }

res_partner_contact()

class training_session_duplicate_wizard(osv.osv_memory):
    _name = 'training.session.duplicate.wizard'

    _columns = {
        'session_id': fields.many2one('training.session', 'Session', required=True, readonly=True, domain=[('state', 'in', ['opened', 'opened_confirmed'])]),
        'group_id' : fields.many2one('training.group', 'Group', domain="[('session_id', '=', session_id)]"),
        'subscription_line_ids' : fields.many2many('training.subscription.line',
                                                   'training_sdw_participation_rel',
                                                   'wizard_id',
                                                   'participation_id',
                                                   'Participations',
                                                   domain="[('session_id', '=', session_id),('state', '=', 'confirmed')]"),
    }

    def action_cancel(self, cr, uid, ids, context=None):
        return {'type' : 'ir.actions.act_window_close'}

    def action_apply(self, cr, uid, ids, context=None):
        if not ids:
            return False
        this = self.browse(cr, uid, ids[0], context=context)

        if len(this.subscription_line_ids) == 0:
            raise osv.except_osv(_('Error'),
                                 _('You have not selected a participant of this session'))

        seances = []

        if any(len(seance.session_ids) > 1 for seance in this.session_id.seance_ids):
            raise osv.except_osv(_('Error'),
                                 _('You have selected a session with a shared seance'))

        #if not all(seance.state == 'opened' for seance in this.session_id.seance_ids):
        #    raise osv.except_osv(_('Error'),
        #                         _('You have to open all seances in this session'))

        lengths = [len(group.seance_ids)
                   for group in this.session_id.group_ids
                   if group != this.group_id]

        if len(lengths) == 0:
            raise osv.except_osv(_('Error'),
                                 _('There is no group in this session !'))

        minimum, maximum = min(lengths), max(lengths)

        if minimum != maximum:
            raise osv.except_osv(_('Error'),
                                 _('The defined groups for this session does not have the same number of seances !'))

        group_id = this.session_id.group_ids[0]

        seance_sisters = {}
        for group in this.session_id.group_ids:
            for seance in group.seance_ids:
                seance_sisters.setdefault((seance.date, seance.duration, seance.course_id or False, seance.kind,), {})[seance.id] = None

        seance_ids = []

        if len(this.group_id.seance_ids) == 0:
            proxy_seance = self.pool.get('training.seance')

            for seance in group_id.seance_ids:
                values = {
                    'group_id' : this.group_id.id,
                    'presence_form' : 'no',
                    'manual' : 0,
                    'participant_count_manual' : 0,
                    'contact_ids' : [(6, 0, [])],
                    'participant_ids' : [],
                    'duplicata' : 1,
                    'duplicated' : 1,
                    'is_first_seance' : seance.is_first_seance,
                }

                seance_ids.append( proxy_seance.copy(cr, uid, seance.id, values, context=context) )
        else:
            # If the there are some seances in this group
            seance_ids = [seance.id for seance in this.group_id.seance_ids]

        for seance in self.pool.get('training.seance').browse(cr, uid, seance_ids, context=context):
            key = (seance.date, seance.duration, seance.course_id or False, seance.kind,)
            if key in seance_sisters:
                for k, v in seance_sisters[key].items():
                    seance_sisters[key][k] = seance.id
            else:
                seance_sisters[key][seance.id] = seance.id

        final_mapping = {}
        for key, values in seance_sisters.iteritems():
            for old_seance_id, new_seance_id in values.iteritems():
                final_mapping[old_seance_id] = new_seance_id

        for sl in this.subscription_line_ids:
            for part in sl.participation_ids:
                part.write({'seance_id' : final_mapping[part.seance_id.id]})

        return {'type' : 'ir.actions.act_window_close'}

    def default_get(self, cr, uid, fields, context=None):
        if context is None:
            context = {}
        record_id = context.get('record_id', False)

        res = super(training_session_duplicate_wizard, self).default_get(cr, uid, fields, context=context)

        if record_id:
            res['session_id'] = record_id

        return res

training_session_duplicate_wizard()

class purchase_order(osv.osv):
    _inherit = 'purchase.order'
    _columns = {
        'contact_id': fields.many2one('res.partner.contact', 'Contact'),
        'seance_id': fields.many2one('training.seance', 'Seance',),
        'course_id': fields.many2one('training.course', 'Course',),

        'date_approved' : fields.datetime('Date Approved', readonly=True),
    }

    def action_po_create(self, cr, uid, po_line_ids, location_id, context=None):
        """ function action_po_assign from mrp module, modified to fullfill needs
            or basing training purchase system """
        if context is None:
            context = {}
        purchase_id = False
        partner_pool = self.pool.get('res.partner')
        product_pricelist_pool = self.pool.get('product.pricelist')
        product_pool = product=self.pool.get('product.product')
        anlytic_journal_pool = self.pool.get('account.analytic.journal')
        anlytic_acc_pool = self.pool.get('account.analytic.account')
        ac_plan_inst_pool = self.pool.get('account.analytic.plan.instance')

        company = self.pool.get('res.users').browse(cr, uid, uid, context).company_id
        for po_line in self.pool.get('training.seance.purchase_line').browse(cr, uid, po_line_ids, context=context):
            if not po_line.product_id.seller_ids:
                raise osv.except_osv(_('Error'), _("Product %s doesn't have any supplier defined") % (po_line.product_id.name))
            partner = po_line.product_id.seller_ids[0].name
            partner_id = partner.id
            address_id = partner_pool.address_get(cr, uid, [partner_id], ['delivery'])['delivery']
            pricelist_id = partner.property_product_pricelist_purchase.id

            uom_id = po_line.product_id.uom_po_id.id

            qty = self.pool.get('product.uom')._compute_qty(cr, uid, po_line.product_uom.id, int(po_line.proc_qty), uom_id)
            if po_line.product_id.seller_ids[0].qty:
                qty=max(qty,po_line.product_id.seller_ids[0].qty)

            price = product_pricelist_pool.price_get(cr, uid, [pricelist_id], po_line.product_id.id, qty, False, {'uom': uom_id})[pricelist_id]

            newdate = datetime.datetime.strptime(po_line.seance_id.date, '%Y-%m-%d %H:%M:%S')
            newdate = newdate - relativedelta(days=company.po_lead)
            newdate = newdate - relativedelta(days=po_line.product_id.seller_ids[0].delay)

            #Passing partner_id to context for purchase order line integrity of Line name
            context.update({'lang':partner.lang, 'partner_id':partner_id})

            product = product_pool.browse(cr, uid, po_line.product_id.id, context=context)


            line = {
                'name': product.partner_ref,
                'product_qty': qty,
                'product_id': po_line.product_id.id,
                'product_uom': uom_id,
                'price_unit': po_line.product_price,
                'date_planned': newdate.strftime('%Y-%m-%d %H:%M:%S'),
                'notes':product.description_purchase,
                'seance_id': po_line.seance_id.id,
                'course_id': po_line.course_id.id,
                'seance_purchase_line_id': po_line.id,
            }

            if po_line.course_id:
                journal_ids = anlytic_journal_pool.search(cr, uid, [('type','=','general'),('active','=',True)])
                if journal_ids:
                    journal_id = journal_ids[0]

                    ana_acct_id = po_line._get_analytic_account_id()
                    ana_acct = anlytic_acc_pool.browse(cr, uid, ana_acct_id, context=context)

                    plan_id = ac_plan_inst_pool.create(cr, uid, {
                        'name': ana_acct.name,
                        'journal_id': journal_id,
                        'account_ids': [(0, 0, {'analytic_account_id': ana_acct.id, 'rate': 100.0})],
                    }, context=context)

                    line['analytics_id'] = plan_id

            if po_line.attachment_id:
                if not po_line.description:
                    name = "%s (%s)" % (po_line.product_id.name, po_line.attachment_id.name,)
                else:
                    name = po_line.description

                note = []
                for component in po_line.attachment_id.component_ids:
                    note.append("%sx %s" % (component.product_qty, component.product_id.name,))

                if po_line.attachment_id.support_note:
                    note.append("==================================")
                    note.append(po_line.attachment_id.support_note)

                if po_line.seance_id.location_id:
                   note.append("==================================")
                   note.append(po_line.seance_id.location_id.name)

                line.update({'name': name, 'notes' : "\n".join(note)})

            else:
                line.update({'name' : po_line.description})

            taxes_ids = po_line.product_id.product_tmpl_id.supplier_taxes_id
            taxes = self.pool.get('account.fiscal.position').map_tax(cr, uid, partner.property_account_position, taxes_ids)
            line.update({
                'taxes_id':[(6,0,taxes)]
            })
            purchase_id = self.pool.get('purchase.order').create(cr, uid, {
                'name': 'Seance %s - %s' % (po_line.seance_id.name, po_line.seance_id.date,),
                'origin': '',
                'partner_id': partner_id,
                'partner_address_id': address_id,
                'location_id': location_id,
                'pricelist_id': pricelist_id,
                'order_line': [(0,0,line)],
                'fiscal_position': partner.property_account_position and partner.property_account_position.id or False,
                'seance_id': po_line.seance_id.id,
                'course_id': po_line.course_id.id,
            })

            get_first_po_line = self.pool.get('purchase.order').browse(cr, uid, purchase_id, context=context).order_line[0]
            if po_line.attachment_id:
                attachment = {'res_model': 'purchase.order.line', 'res_id': get_first_po_line.id}
                new_id = self.pool.get('ir.attachment').copy(cr, uid, po_line.attachment_id.id, attachment, context=context)

            workflow = netsvc.LocalService('workflow')
            workflow.trg_validate(uid, 'purchase.order', purchase_id, "purchase_confirm", cr)
        return purchase_id

    def create_from_procurement_line(self, cr, uid, po_line, quantity, location_id, context=None):
        """ proxy function for creating procurment from seance.purchase.line object """
        procurement_id = self.action_po_create(cr, uid, [po_line.id], location_id, context=context)
        return procurement_id

purchase_order()

class purchase_order_line(osv.osv):
    _inherit = 'purchase.order.line'
    _order = 'seance_date ASC'

    def _get_purchase_order_line_attachments(self, cr, uid, ids, name, arg, context=None):
        res = {}
        for id in ids:
            res.setdefault(id, [])
        att_pool = self.pool.get('ir.attachment')
        att_ids = att_pool.search(cr, uid, [('res_model','=','purchase.order.line'),('res_id','in',ids)], context=context)

        for att in att_pool.read(cr, uid, att_ids, ['res_id'], context=context):
            res[att['res_id']].append(att['id'])
        return res

    def __build_value_attachment_set(self, cr, uid, value, context=None):
        value_att_ops = set()
        for act in value:
            if act[0] == 6: # set all
                value_att_ops = set(act[2])
            elif act[0] == 3: # unlink
                value_att_ops.remove(act[1])
            elif act[0] == 4: # link
                value_att_ops.add(act[1])
            elif act[0] == 5: # unlink al
                value_att_ops = set()
        return value_att_ops

    def _save_purchase_order_line_attachments(self, cr, uid, id, fname, fvalue, arg, context=None):
        att_pool = self.pool.get('ir.attachment')
        current_att_ids = set(att_pool.search(cr, uid, [('res_model','=',self._name),('res_id','=',id)], context=context))
        value_att_ops = self.__build_value_attachment_set(cr, uid, fvalue, context=context)

        # check what need to be removed
        value_to_remove = current_att_ids.difference(value_att_ops)
        if value_to_remove:
            att_pool.unlink(cr, uid, list(value_to_remove), context=context)
        # check which attachment need to be copied
        value_to_add = value_att_ops.difference(current_att_ids)
        if value_to_add:
            new_att_ids = []
            for att_id in value_to_add:
                att_default = {'res_model': self._name, 'res_id': id}
                new_att_id = att_pool.copy(cr, uid, att_id, default=att_default, context=context)
                new_att_ids.append(new_att_id)
        return True

    _columns = {
        'seance_id_int' : fields.related('seance_id', 'id', type='integer', readonly=True, string="Seance ID"),
        'seance_id': fields.many2one('training.seance', 'Seance'),
        'seance_date': fields.related('seance_id', 'date', type='datetime', string='Seance Date', readonly=True, store=True),
        'course_id': fields.many2one('training.course', 'Course'),
        'seance_purchase_line_id': fields.many2one('training.seance.purchase_line', 'Purchase Line'),
        'state': fields.related('order_id', 'state',
                                type='selection',
                                selection = [('draft', 'Request for Quotation'),
                                             ('wait', 'Waiting'),
                                             ('confirmed', 'Confirmed'),
                                             ('approved', 'Approved'),
                                             ('except_picking', 'Shipping Exception'),
                                             ('except_invoice', 'Invoice Exception'),
                                             ('done', 'Done'),
                                             ('cancel', 'Cancelled')
                                            ],
                                string='Order Status', readonly=True, select=True,
                                help="The state of the purchase order or the quotation request. A quotation is a purchase order in a 'Draft' state. Then the order has to be confirmed by the user, the state switch to 'Confirmed'. Then the supplier must confirm the order to change the state to 'Approved'. When the purchase order is paid and received, the state becomes 'Done'. If a cancel action occurs in the invoice or in the reception of goods, the state becomes in exception.",
                               ),
        'attachment_ids': fields.function(_get_purchase_order_line_attachments, type='many2many', relation='ir.attachment', string='Attachments', method=True, fnct_inv=_save_purchase_order_line_attachments),
    }

    def approved_cb(self, cr, uid, ids, context=None):
        workflow = netsvc.LocalService('workflow')
        for pol in self.browse(cr, uid, ids, context=context):
            if len(pol.order_id.order_line) in (0, 1):
                workflow.trg_validate(uid, 'purchase.order', pol.order_id.id, 'purchase_approve', cr)
                pol.order_id.write({'date_approved' : time.strftime('%Y-%m-%d %H:%M:%S')})
            else:
                raise osv.except_osv(_('Warning'),
                                     _('You can not approve this purchase line it is not the only one purchase order lines in the purchase order'))

        return True

    def cancel_cb(self, cr, uid, ids, context=None):
        workflow = netsvc.LocalService('workflow')
        for pol in self.browse(cr, uid, ids, context=context):
            if len(pol.order_id.order_line) in (0, 1):
                workflow.trg_validate(uid, 'purchase.order', pol.order_id.id, 'purchase_cancel', cr)
            else:
                raise osv.except_osv(_('Warning'),
                                     _('You can not cancel this purchase line it is not the only one purchase order lines in the purchase order'))

        return True

    def onchange_seance_id(self, cr, uid, ids, seance_id, context=None):
        ocv = {
            'value': {
                'seance_date': False,
                'course_id': False,
            }
        }
        if not seance_id:
            return ocv
        seance = self.pool.get('training.seance').browse(cr, uid, seance_id, context=context)
        if seance.course_id:
            ocv['value']['course_id'] = seance.course_id.id
        ocv['value']['seance_date'] = seance.date
        return ocv

    def onchange_attachment_ids(self, cr, uid, ids, attachment_ids, product_id, context=None):
        if not attachment_ids:
            return False
        proc_price = 'from_attachment'
        product_price = 0.0
        if product_id:
            prod = self.pool.get('product.product').browse(cr, uid, product_id, context=context)
            proc_price = prod.procurement_price
            product_price = prod.standard_price

        if isinstance(attachment_ids, (list,set)):
            att_ids = self.__build_value_attachment_set(cr, uid, attachment_ids, context=context)
            if not att_ids:
                if proc_price == 'from_product':
                    return {'value': {'price_unit': product_price}}
                else:
                    return False
            att = self.pool.get('ir.attachment').browse(cr, uid, list(att_ids)[0], context=context)
            if proc_price == 'from_product':
                return {'value': {'price_unit': product_price }}
            else:
                return {'value': {'price_unit': att.price }}
        return False

    def product_id_change(self, cr, uid, ids, pricelist, product, qty, uom,
            partner_id, date_order=False, fiscal_position=False, date_planned=False,
            name=False, price_unit=False, notes=False, attachment_ids=None):
        r = super(purchase_order_line, self).product_id_change(cr, uid, ids, pricelist, product, qty, uom,
                        partner_id, date_order, fiscal_position, date_planned,
                        name, price_unit, notes)
        if not r:
            r = {}
        if attachment_ids:
            rattach = self.onchange_attachment_ids(cr, uid, ids, attachment_ids, product)
            if rattach:
                r.setdefault('value', {})['price_unit'] = rattach.get('value',{}).get('price_unit', 0.0)
        return r

purchase_order_line()


class res_lang(osv.osv):
    _inherit = 'res.lang'

    _columns = {
        'code' : fields.char('Code', size=5, required=True),
        'name' : fields.char('Name', size=64, required=True),
    }

res_lang()

class training_subscription_cancellation_wizard(osv.osv_memory):
    _name = 'training.subscription.cancellation.wizard'

    _description = 'Training Subscription Cancellation Wizard'

    _columns = {
        'subscription_line_id' : fields.many2one('training.subscription.line', 'Subscription Line', required=True,
                                                 domain="[('state', 'in', ['draft', 'confirmed']),('session_id.state', 'in', ('opened', 'opened_confirmed', 'closed_confirmed'))]"),
        'subscription_id' : fields.related('subscription_line_id', 'subscription_id', type='many2one', relation='training.subscription', string='Subscription', readonly=True),
        'partner_id' : fields.related('subscription_line_id', 'subscription_id', 'partner_id', type='many2one', relation='res.partner', string='Partner', readonly=True),
        'participant_id' : fields.related('subscription_line_id', 'job_id', type='many2one', relation='res.partner.job', string='Participant', readonly=True),
        'session_id' : fields.related('subscription_line_id', 'session_id', type='many2one', relation='training.session', string='Session', readonly=True),
        'session_offer_id' : fields.related('subscription_line_id', 'session_id', 'offer_id', type='many2one', relation='training.offer', string='Session Offer', readonly=True),
        'session_date' : fields.related('subscription_line_id', 'session_id', 'date', type='datetime', string='Session Date', readonly=True),
        'session_state' : fields.related('subscription_line_id', 'session_id', 'state', type='selection',
                                         selection=[
                                             ('draft', 'Draft'),
                                             ('opened', 'Opened'),
                                             ('opened_confirmed', 'Confirmed'),
                                             ('closed_confirmed', 'Closed Subscriptions'),
                                             ('inprogress', 'In Progress'),
                                             ('closed', 'Closed'),
                                             ('cancelled', 'Cancelled')
                                         ], string='Session State', readonly=True),
        'new_participant_id' : fields.many2one('res.partner.job', 'Participant', domain="[('name', '=', partner_id),('id', '!=', participant_id),('state', '=', 'current')]"),
        'new_participant_email' : fields.char('Email', size=128),
        'new_session_id' : fields.many2one('training.session', 'Session', domain="[('state', 'in', ('opened', 'opened_confirmed')),('date', '>', time.strftime('%Y-%m-%d')),('id','!=',session_id),('offer_id', '=', session_offer_id)]"),
        'new_session_date' : fields.related('new_session_id', 'date', type='datetime', string='Session Date', readonly=True),
        'cancellation_reason' : fields.text('Reason'),
        'cancellation_medical_certificate_toggle' : fields.boolean('Has Justification'),
        'cancellation_medical_certificate_name' : fields.char('Filename', size=128),
        'cancellation_medical_certificate' : fields.binary('Justification'),
        'end_user_message': fields.text('End User Message', readonly=True),
        'state' : fields.selection([('init', 'Init'),
                                    ('replacement', 'Replacement'),
                                    ('postponement', 'Postponement'),
                                    ('cancellation', 'Cancellation'),
                                    ('end', 'End')], 'State', required=True, readonly=True),
    }

    _defaults = {
        'subscription_line_id' : lambda obj, cr, uid, context: context['active_id'],
        'state' : lambda *a: 'init',
    }

    def on_change_subscription_line(self, cr, uid, ids, subscription_line_id, context=None):
        if not subscription_line_id:
            return {}
        subscription_line = self.pool.get('training.subscription.line').browse(cr, uid, subscription_line_id, context=context)
        return {
            'value' : {
                'subscription_id' : subscription_line.subscription_id.id,
                'subscription_line_id' : subscription_line.id,
                'session_id' : subscription_line.session_id.id,
                'session_date' : subscription_line.session_id.date,
                'session_state' : subscription_line.session_id.state,
                'partner_id' : subscription_line.subscription_id.partner_id.id,
                'participant_id' : subscription_line.job_id.id,
                'session_offer_id' : subscription_line.session_id.offer_id.id,
            }
        }

    def on_change_new_participant(self, cr, uid, ids, new_participant_id, context=None):
        if not new_participant_id:
            return {}

        job = self.pool.get('res.partner.job').browse(cr, uid, new_participant_id, context=context)
        return {'value' : {'new_participant_email' : job.email }}

    def on_change_new_session(self, cr, uid, ids, new_session_id, context=None):
        if not new_session_id:
            return {}
        session = self.pool.get('training.session').browse(cr, uid, new_session_id, context=context)
        return {
            'value' : {
                'new_session_date' : session.date,
            }
        }


    def action_cancel(self, cr, uid, ids, context=None):
        return {'type':'ir.actions.act_window_close'}

    def action_cancellation(self, cr, uid, ids, context=None):
        return self.write(cr, uid, ids, {'state' : 'cancellation'}, context=context)

    def action_replacement(self, cr, uid, ids, context=None):
        return self.write(cr, uid, ids, {'state' : 'replacement'}, context=context)

    def action_postponement(self, cr, uid, ids, context=None):
        # Assign a new session to the subscription line
        return self.write(cr, uid, ids, {'state' : 'postponement'}, context=context)

    def action_apply(self, cr, uid, ids, context=None):
        if context is None:
            context = {}
        if not ids:
            return False
        this = self.browse(cr, uid, ids[0], context=context)
        old_participant_id = this.participant_id
        workflow = netsvc.LocalService('workflow')
        context2 = context.copy()
        if 'default_state' in context2:
            del context2['default_state']

        end_user_message = _('The operation ends sucessfully !')

        if this.state == 'cancellation':
            if this.cancellation_medical_certificate:
                values = {
                    'name' : 'Medical Certificate',
                    'datas' : this.cancellation_medical_certificate,
                    'datas_fname' : this.cancellation_medical_certificate_name,
                    'description' : 'Medical Certificate',
                    'res_model' : 'training.subscription.line',
                    'res_id' : this.subscription_line_id.id,
                }
                self.pool.get('ir.attachment').create(cr, uid, values, context=context2)
            this.subscription_line_id.write(
                {
                    'has_certificate' : this.cancellation_medical_certificate_toggle,
                    'reason_cancellation' : this.cancellation_reason,
                },
                context=context2
            )
            if this.subscription_line_id.invoice_line_id \
                and this.subscription_line_id.invoice_id.state != 'draft':
                    end_user_message += '\n' + _("NOTE: You need to manually generate a 'Credit Note' for this subscription line")
            workflow.trg_validate(uid, 'training.subscription.line', this.subscription_line_id.id, 'signal_cancel', cr)

            # check invoicing
            # if already invoice, but invoice is 'draft' => remove line from invoice
            for line in [this.subscription_line_id]:
                invoice_line_record = line.invoice_line_id
                invoice_record = line.invoice_line_id and line.invoice_id or False
                if invoice_record and invoice_record.state == 'draft':
                    cancel_invoice = False
                    if len(invoice_record.invoice_line) == 1:
                        cancel_invoice = True
                    invoice_line_record.unlink(context=context)
                    invoice_record.button_compute(context=context)
                    if cancel_invoice:
                        workflow.trg_validate(uid, 'account.invoice', invoice_record.id, 'invoice_cancel', cr)

        elif this.state == 'replacement':
            email_proxy = self.pool.get('training.email')

            objects = {
                'new_participant_id' : this.new_participant_id,
                'old_participant_id' : old_participant_id,
            }

            this.subscription_line_id.write( { 'job_id' : this.new_participant_id.id, 'job_email' : this.new_participant_email, })

            internal_note = []

            if this.subscription_line_id.internal_note:
                internal_note.append(this.subscription_line_id.internal_note)

            internal_note.append(_("Replacement: %s %s -> %s %s") % (old_participant_id.contact_id.first_name,
                                                                     old_participant_id.contact_id.name,
                                                                     this.new_participant_id.contact_id.first_name,
                                                                     this.new_participant_id.contact_id.name))
            this.subscription_line_id.write({'internal_note' : "\n----\n".join(internal_note)})
            email_proxy.send_email(cr, uid,
                             'sub_replacement',
                             'hr',
                             this.subscription_line_id.partner_hr_email,
                             session=this.subscription_line_id.session_id,
                             context=context2,
                             subline=this.subscription_line_id,
                             **objects)

        elif this.state == 'postponement':
            values = {
                'session_id' : this.new_session_id.id,
            }
            sl_proxy = self.pool.get('training.subscription.line')
            new_sl_id = sl_proxy.copy(cr, uid, this.subscription_line_id.id, values, context = context2 or {})
            new_sl = sl_proxy.browse(cr, uid, new_sl_id, context=context2)
            new_sl.write({'internal_note' : _("Created by Postponement of %s") % this.subscription_line_id.name})
            this.subscription_line_id.write({'reason_cancellation' : _("Cancelled by Postponement: %s") % new_sl.name })

            if this.subscription_line_id.state == 'confirmed':
                workflow.trg_validate(uid, 'training.subscription.line', new_sl_id, 'signal_confirm', cr)
            workflow.trg_validate(uid, 'training.subscription.line', this.subscription_line_id.id, 'signal_cancel', cr)
        return self.write(cr, uid, ids, {'state' : 'end', 'end_user_message': end_user_message}, context=context)

    def action_done(self, cr, uid, ids, context=None):
        return {'type' : 'ir.actions.act_window_close'}

training_subscription_cancellation_wizard()

class training_participation_reassign_wizard(osv.osv_memory):
    _name = 'training.participation.reassign.wizard'

    _columns = {
        'participation_id' : fields.many2one('training.participation', 'Participation', required=True),
        'participation_seance_id' : fields.related('participation_id', 'seance_id', type='many2one', relation='training.seance', readonly=True, string='Seance'),
        'participation_seance_date' : fields.related('participation_id', 'seance_id', 'date', type='datetime', readonly=True, string='Date'),
        'participation_sl' : fields.related('participation_id', 'subscription_line_id', type='many2one', relation='training.subscription.line', readonly=True, string='Subscription Line'),
        'participation_session_id' : fields.related('participation_id', 'subscription_line_id', 'session_id', type='many2one', relation='training.session',
                                                    readonly=True, string='Session'),
        'seance_id' : fields.many2one('training.seance', 'Seance', required=True),
    }

    def on_change_seance(self, cr, uid, ids, seance_id, context=None):
        values = {
            'domain' : {
                'participation_id' : not seance_id and [] or [('seance_id', '=', seance_id)],
            }
        }
        return values

    def on_change_participation(self, cr, uid, ids, participation_id, context=None):
        if not participation_id:
            return {
                'value' : {
                    'seance_id' : 0,
                },
                'domain' : {
                    'seance_id' : [],
                },
            }

        p = self.pool.get('training.participation').browse(cr, uid, participation_id, context=context)
        return {
            'value' : {
                'participation_seance_id' : p.seance_id.id,
                'participation_seance_date' : p.seance_id.date,
                'participation_sl' : p.subscription_line_id.id,
                'participation_session_id' : p.subscription_line_id.session_id.id,
            },
            'domain' : {
                'seance_id' : [('id', 'in', [seance.id for seance in p.subscription_line_id.session_id.seance_ids])],
            }
        }

    def close_cb(self, cr, uid, ids, context=None):
        return {'type' : 'ir.actions.act_window_close'}

    def apply_cb(self, cr, uid, ids, context=None):
        if not ids:
            return False
        this = self.browse(cr, uid, ids[0], context=context)

        if this.participation_id.seance_id == this.seance_id:
            raise osv.except_osv(_('Warning'),
                                 _('You have selected the same seance'))
        this.participation_id.write({'seance_id' : this.seance_id.id})
        return {'type' : 'ir.actions.act_window_close'}

training_participation_reassign_wizard()

class training_seance_purchase_errors(osv.osv):
    _auto = False
    _name = 'training.seance.purchase_errors'

    def init(self, cr):
        tools.drop_view_if_exists(cr, 'training_seance_purchase_errors')
        cr.execute("CREATE OR REPLACE VIEW training_seance_purchase_errors as ( "
        """
            SELECT pol.seance_id AS id,
                   SUM(CASE WHEN (CAST (CASE WHEN tspl.fix = 'fix' THEN 1.0 ELSE (CASE WHEN tsea.manual THEN tsea.participant_count_manual ELSE COALESCE(tsea_count.count,0) END) END * tspl.product_qty AS numeric(16,2)) != pol.product_qty) THEN 1 ELSE 0 END) AS have_proc_in_errors
            FROM purchase_order_line pol
            LEFT JOIN training_seance_purchase_line tspl ON (pol.seance_purchase_line_id = tspl.id)
            LEFT JOIN training_seance tsea ON (tspl.seance_id = tsea.id AND tspl.fix = 'by_subscription')
            LEFT JOIN (SELECT tp.seance_id,
                              COUNT(DISTINCT(tsl.contact_id))
                       FROM training_participation tp,
                            training_subscription_line tsl
                       WHERE tp.subscription_line_id = tsl.id
                          AND tsl.state IN ('confirmed','done')
                       GROUP BY tp.seance_id
                    ) AS tsea_count ON (tsea_count.seance_id = tspl.seance_id)
            GROUP BY pol.seance_id
        """
                   " )")

    _columns = {
        #'id': fields.many2one('training.seance', 'Seance', required=True, readonly=True),
        'have_proc_in_errors': fields.float('Procurements In Errors', digits=(16,2), required=True, readonly=True),
    }

training_seance_purchase_errors()

class training_seance_purchase_errors(osv.osv):
    _inherit = 'training.seance'

    def _purchase_in_error(self, cr, uid, ids, fieldname, args, context=None):
        prsh_error_obj = self.pool.get('training.seance.purchase_errors')
        res = dict.fromkeys(ids, False)

        # filter unkown seance ids in training.seance.purchase_errors
        # and get the seance status (w. or wo. errors)
        search_ids = prsh_error_obj.search(cr, uid, [('id','in',ids)], context=context)
        for seance_status in prsh_error_obj.read(cr, uid, search_ids, context=context):
            if seance_status.get('have_proc_in_errors', 0) > 0:
                res[seance_status['id']] = True
            else:
                res[seance_status['id']] = False
        return res

    def _purchase_in_error_search(self, cr, uid, obj, name, args, context=None):
        if not len(args):
            return []
        search_for_errors = False
        for arg in args:
            if arg[0] == 'purchase_in_error' and arg[2] == True:
                search_for_errors = True
        search_op = search_for_errors and '>' or '='
        search_domain = [('have_proc_in_errors', search_op, 0)]
        seance_ids = self.pool.get('training.seance.purchase_errors').search(cr, uid, search_domain, context=context)
        return [('id', 'in', seance_ids)]

    _columns = {
        'purchase_in_error' : fields.function(_purchase_in_error, method=True, fnct_search=_purchase_in_error_search,
                                           type='boolean', string='Have Purchase In Errors'),
    }
training_seance_purchase_errors()

class res_partner_w_themes(osv.osv):
    _inherit = 'res.partner'
    _columns = {
        'theme_ids': fields.many2many('training.course.theme', 'res_partner_themes_rel', 'partner_id', 'theme_id', string='Themes'),
        'target_public_id': fields.many2one('training.offer.public.target', 'Target Audience'),
    }
res_partner_w_themes()

# vim:expandtab:smartindent:tabstop=4:softtabstop=4:shiftwidth=4:
