# -*- coding: utf-8 -*-
##############################################################################
#
#    OpenERP, Open Source Management Solution
#    Copyright (C) 2004-2010 Tiny SPRL (<http://tiny.be>).
#    $Id: account.py 1005 2005-07-25 08:41:42Z nicoe $
#
#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU Affero General Public License as
#    published by the Free Software Foundation, either version 3 of the
#    License, or (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU Affero General Public License for more details.
#
#    You should have received a copy of the GNU Affero General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
##############################################################################

import datetime
import calendar

from osv import osv, fields

from hrself.hrself_datetime import to_string

class hrself_holidays_report_wizard(osv.osv_memory):
    """Common class for wizards that report something."""

    _name = 'hrself.holidays.report.wizard'

    _columns = {
        'employee_ids': fields.many2many('hr.employee', 'summary_dept_emp_rel', 'rep_id', 'emp_id', 'Employee(s)'),
    }

    def _get_employees(self, cr, uid, context=None):
        """Employees of which the current user can see the leave requests."""
        result = []
        user_object = self.pool.get('res.users')
        user = user_object.browse(cr, uid, uid, context=context)
        employee = user.employee_id
        employee_view = user.hrself_view_type
        employee_object = self.pool.get('hr.employee')
        if employee_view == 'user':
            result = [employee.id]
        elif employee_view == 'approver':
            employee_ids = employee_object.search(cr, uid, [], context=context)
            approvers = employee_object.get_leave_approver(cr, uid, employee_ids, datetime.date.today(), context=context)
            for (employee_id, approver_id) in approvers.items():
                if approver_id == employee.id:
                    result.append(employee_id)
            if not employee.id in result:
                result.append(employee.id)
        elif employee_view == 'administrator':
            employee_ids = employee_object.search(cr, uid, [], context=context)
            result = employee_ids
            if uid != 1:
                admin_employee_ids = employee_object.search(cr, uid, [('user_id', '=', 1)], context=context)
                #remove administrator from list of employees
                employee_ids.remove(admin_employee_ids[0])
        return result

    _defaults = {
         'employee_ids': _get_employees,
    }

hrself_holidays_report_wizard()

class hrself_holidays_planned_leaves_wizard(osv.osv_memory):
    """Wizard to report planned leaves."""

    _name = 'hrself.holidays.planned.leaves.wizard'
    _inherit = 'hrself.holidays.report.wizard'

    _columns = {
        'date_from': fields.date('From', required=True),
        'date_to': fields.date('To', required=True),
    }

    def _from_date(self, cr, uid, context=None):
        return to_string(datetime.date.today().replace(day=1))

    def _to_date(self, cr, uid, context=None):
        today = datetime.date.today()
        last_day_of_month = calendar.monthrange(today.year, today.month)[1]
        return to_string(today.replace(day=last_day_of_month))
    
    _defaults = {
         'date_from': _from_date,
         'date_to':_to_date,
    }

    def print_report(self, cr, uid, ids, context=None):
        """Print the report."""
        form_data = self.read(cr, uid, ids, [])[0]
        datas = {
             'ids': [],
             'model': 'ir.ui.menu',
             'form': form_data
        }
        return {
            'type': 'ir.actions.report.xml',
            'report_name': 'hrself.planned.leaves',
            'datas': datas,
        }

hrself_holidays_planned_leaves_wizard()

class hrself_holidays_planned_activities_wizard(osv.osv_memory):
    """Wizard to report planned activities."""

    _name = 'hrself.holidays.planned.activities.wizard'
    _inherit = 'hrself.holidays.planned.leaves.wizard'

    def print_report(self, cr, uid, ids, context=None):
        """Print the report."""
        form_data = self.read(cr, uid, ids, [])[0]
        datas = {
             'ids': [],
             'model': 'ir.ui.menu',
             'form': form_data
        }
        return {
            'type': 'ir.actions.report.xml',
            'report_name': 'hrself.planned.activities',
            'datas': datas,
        }

hrself_holidays_planned_activities_wizard()

class hrself_holidays_counters_balance_wizard(osv.osv_memory):
    """Wizard to report the balance of the counters of leave types."""

    _name = 'hrself.holidays.counters.balance.wizard'
    _inherit = 'hrself.holidays.report.wizard'

    def print_report(self, cr, uid, ids, context=None):
        """Print the report."""
        type_object = self.pool.get('hrself.holidays.type')
        balance_object = self.pool.get('hrself.holidays.balance')
        type_ids = type_object.search(cr, uid, [], context=context)
        wizard = self.browse(cr, uid, ids[0], context=context)
        end_year = datetime.date(datetime.date.today().year, 12, 31)
        balance_ids = []
        for employee in wizard.employee_ids:
            balance_context = {
                'employee_id': employee.id,
            }
            remaining_date_to = datetime.date.today()
            today_balances = type_object.compute_values(cr, uid, type_ids, context=balance_context, remaining_date_to=remaining_date_to)
            services = self.pool.get('hrself.services').hrself_holidays_service(cr, uid, context=context)
            activities = services.get_theoretical_activities(cr, uid, employee.id, end_year, end_year, context=context)
            if activities:
                balance_context['activity_context'] = activities[0]['activity_context']
            end_year_balances = type_object.get_remaining_leaves_for_employee(cr, uid, type_ids, context=balance_context)
            for type in type_object.browse(cr, uid, type_ids, context=context):
                initial_value = type.get_initialization_duration_for_employee(context=balance_context)[type.id]
                search_ids = balance_object.search(cr, uid, [('employee_id', '=', employee.id), ('type_id', '=', type.id)], context=context)
                if search_ids:
                    balance_object.write(cr, uid, search_ids, {
                        'unit': today_balances[type.id]['unit'],
                        'initial_value': initial_value,
                        'today_balance': today_balances[type.id]['remaining_leaves'],
                        'end_year_balance': end_year_balances.get(type.id, 0),
                    }, context=context)
                    balance_id = search_ids[0]
                else:
                    balance_id = balance_object.create(cr, uid, {
                        'employee_id': employee.id,
                        'type_id': type.id,
                        'unit': today_balances[type.id]['unit'],
                        'initial_value': initial_value,
                        'today_balance': today_balances[type.id]['remaining_leaves'],
                        'end_year_balance': end_year_balances.get(type.id, 0),
                    }, context=context)
                balance_ids.append(balance_id)

        model_data_object = self.pool.get('ir.model.data')
        model_data_id = model_data_object._get_id(cr, uid, 'hrself_holidays', 'hrself_holidays_balance_action')
        window_id = model_data_object.browse(cr, uid, model_data_id, context=context).res_id
        result = self.pool.get('ir.actions.act_window').read(cr, uid, window_id, context=context)
        result['domain'] = [('id', 'in', balance_ids)]
        return result

hrself_holidays_counters_balance_wizard()
