# -*- coding: utf-8 -*-
##############################################################################
#    
#    OpenERP, Open Source Enterprise Management Solution
#    Copyright (C) 2004-2010 Tiny SPRL (<http://openerp.com>).
#
#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU Affero General Public License as
#    published by the Free Software Foundation, either version 3 of the
#    License, or (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU Affero General Public License for more details.
#
#    You should have received a copy of the GNU Affero General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.     
#
##############################################################################

import logging
import datetime

from osv import osv, fields

from hrself import hrself_datetime

class hrself_holidays_initialization_wizard(osv.osv_memory):
    """Wizard to generate initialization requests of leave type precisions."""

    _name = 'hrself.holidays.initialization.wizard'

    _columns = {
        'date_from': fields.date('Initialization start Date', required=True),
        'error_ids': fields.many2many('hrself.holidays.theoretical.activity.error', 'wizard_error_rel', 'wizard_id', 'error_id', 'Errors'),
        'initialization_ids': fields.many2many('hrself.holidays.request.initialization', 'wizard_initialization_rel', 'wizard_id', 'initialization_id', 'Generated initializations'),
    }

    _defaults = {
        'date_from': hrself_datetime.to_string(datetime.date.today())
    }

    def generate(self, cr, uid, ids, context=None):
        """Generate initialization requests of leave type precisions."""
        if context is None:
            context = {}

        wizard = self.browse(cr, uid, ids[0], context=context)
        generate_date = datetime.datetime.today()
        initialization_object = self.pool.get('hrself.holidays.request.initialization')
        error_object = self.pool.get('hrself.holidays.theoretical.activity.error')
        employee_ids = self.pool.get('hr.employee').search(cr, uid, [('universe_id', '!=', None)], context=context)
        initialization_ids = []
        error_ids = []
        for precision in self.pool.get('hrself.holidays.type.precision').browse(cr, uid, context['active_ids'], context=context):
            if precision.has_initialization_value:
                services = self.pool.get('hrself.services').hrself_holidays_service(cr, uid, context=context)
                employees_activities = services.get_theoretical_activities(cr, uid, employee_ids, wizard.date_from, wizard.date_from, context=context)
                for employee_id, activities in employees_activities.items():
                    #int activities means there was an error and activities is the error code
                    if isinstance(activities, int):
                        error_id = error_object.create(cr, uid, {
                            'employee_id': employee_id,
                            'message': activities,
                        }, context=context)
                        error_ids.append(error_id)
                    elif len(activities) > 0:
                        if precision.activity_context_id.is_compatible(activities[0]['activity_context']):
                            initialization_id = initialization_object.create(cr, uid, {
                                'employee_id': employee_id,
                                'date_from': wizard.date_from,
                                'duration': precision.init_value,
                                'type_id': precision.type_id.id,
                                'activity_context_id': precision.activity_context_id.id,
                                'name': 'Generated from the leave type precision',
                                'generate_date': generate_date,
                                'generate_source': 'p',
                            }, context=context)
                            initialization_ids.append(initialization_id)

        self.write(cr, uid, ids, {
            'error_ids': [(6, 0, error_ids)],
            'initialization_ids': [(6, 0, initialization_ids)],
        }, context=context)
        model_data_object = self.pool.get('ir.model.data')
        model_data_id = model_data_object._get_id(cr, uid, 'hrself_holidays', 'hrself_holidays_initialization_wizard_result_action')
        window_id = model_data_object.browse(cr, uid, model_data_id, context=context).res_id
        result = self.pool.get('ir.actions.act_window').read(cr, uid, window_id, context=context)
        result['domain'] = "[('id', '=', %d)]" % ids[0]
        return result
        
hrself_holidays_initialization_wizard()

class hrself_annual_initialization_wizard(osv.osv_memory):
    """Wizard for the HRSelf annual initialization."""

    _inherit = 'hrself.annual.initialization.wizard'

    _columns = {
        'initialization_ids': fields.many2many('hrself.holidays.request.initialization', 'annual_initialization_wizard_rel', 'wizard_id', 'initialization_id', 'Initializations'),
        'leave_ids': fields.many2many('hrself.holidays.request.leave', 'annual_initialization_wizard_leave_rel', 'wizard_id', 'leave_id', 'Leaves'),
    }

    def _initialize(self, cr, uid, ids, context=None):
        """Private implementation of the initialization."""
        employee = self.pool.get('res.users').browse(cr, uid, uid, context=context).employee_id
        universe_id = employee.universe_id.id
        universe_object = self.pool.get('hrself.universe')
        initialization_date = universe_object.browse(cr, uid, universe_id, context=context).initialization_date
        initialization_object = self.pool.get('hrself.holidays.request.initialization')
        initialization_ids = initialization_object.search(cr, uid, [
            ('generate_date', '=', initialization_date),
            ('generate_source', '=', 'a'),
        ], context=context)
        initialization_object.unlink(cr, uid, initialization_ids, context=context)
        leave_object = self.pool.get('hrself.holidays.request.leave')
        leave_ids = leave_object.search(cr, uid, [
            ('generate_date', '=', initialization_date),
        ], context=context)
        leave_object.unlink(cr, uid, leave_ids, context=context)
        universe_object.write(cr, uid, universe_id, {'initialization_date': datetime.datetime.today()}, context=context)

        wizard_id = self.create(cr, uid, {}, context=context)
        employee_ids = self.pool.get('hr.employee').search(cr, uid, [('groupes_id', '!=', 0)], context=context)
        precision_object = self.pool.get('hrself.holidays.type.precision')
        precision_ids = precision_object.search(cr, uid, [], context=context)
        counter_type_ids = []
        for precision in precision_object.browse(cr, uid, precision_ids, context=context):
            if precision.counter_type_id and not precision.counter_type_id.id in counter_type_ids:
                counter_type_ids.append(precision.counter_type_id.id)
        services = self.pool.get('hrself.services').hrself_holidays_service(cr, uid, context=context)
        date_from = datetime.date(datetime.date.today().year, 1, 1)
        date_to = datetime.date(datetime.date.today().year, 1, 31)
        balances = services.get_counter_balances(cr, uid, employee_ids, counter_type_ids, date_from, context=context)
        initialization_ids = self._initialization_ids_from_balances(cr, uid, balances, context=context)
        movements = services.get_counter_movements(cr, uid, employee_ids, counter_type_ids, date_from, date_to, context=context)
        leave_ids = self._leave_ids_from_movements(cr, uid, movements, context=context)
        self.write(cr, uid, [wizard_id], {
            'initialization_ids': [(6, 0, initialization_ids)],
            'leave_ids': [(6, 0, leave_ids)]
        }, context=context)

        model_data_object = self.pool.get('ir.model.data')
        model_data_id = model_data_object._get_id(cr, uid, 'hrself_holidays', 'hrself_annual_initialization_wizard_success_action')
        window_id = model_data_object.browse(cr, uid, model_data_id, context=context).res_id
        result = self.pool.get('ir.actions.act_window').read(cr, uid, window_id, context=context)
        result['domain'] = "[('id', '=', %d)]" % wizard_id
        return result

    def _initialization_ids_from_balances(self, cr, uid, counter_balances, context=None):
        """Initialization ids from counter balances."""
        result = []
        precision_object = self.pool.get('hrself.holidays.type.precision')
        universe_id = self.pool.get('res.users').browse(cr, uid, uid, context=context).employee_id.universe_id.id
        initialization_date = self.pool.get('hrself.universe').browse(cr, uid, universe_id, context=context).initialization_date
        for balance in counter_balances:
            precision_ids = precision_object.search(cr, uid, [('counter_type_id', '=', balance['counter_type_id'])], context=context)
            precision = precision_object.browse(cr, uid, precision_ids[0])
            values = dict(balance, 
                          type_id = precision.type_id.id, 
                          activity_context_id = precision.activity_context_id.id,
                          name = 'Generated by the annual initialization',
                          generate_date = initialization_date,
                          generate_source = 'a',
                         )
            del values['counter_type_id']
            initialization_id = self.pool.get('hrself.holidays.request.initialization').create(cr, uid, values, context=context)
            result.append(initialization_id)
        return result

    def _leave_ids_from_movements(self, cr, uid, counter_movements, context=None):
        """Leave ids from counter movements."""
        result = []
        precision_object = self.pool.get('hrself.holidays.type.precision')
        employee = self.pool.get('res.users').browse(cr, uid, uid, context=context).employee_id
        universe_id = employee.universe_id.id
        initialization_date = self.pool.get('hrself.universe').browse(cr, uid, universe_id, context=context).initialization_date
        for movement in counter_movements:
            precision_ids = precision_object.search(cr, uid, [('counter_type_id', '=', movement['counter_type_id'])], context=context)
            precision = precision_object.browse(cr, uid, precision_ids[0])
            values = dict(movement, 
                          type_id = precision.type_id.id, 
                          activity_context_id = precision.activity_context_id.id,
                          state = 'validated',
                          leave_approver_id = employee.id,
                          name = 'Generated by the annual initialization',
                          generate_date = initialization_date,
                         )
            del values['counter_type_id']
            leave_id = self.pool.get('hrself.holidays.request.leave').create(cr, uid, values, context=context)
            result.append(leave_id)
        return result

hrself_annual_initialization_wizard()
