# -*- coding: utf-8 -*-
##############################################################################
#
#    OpenERP, Open Source Management Solution
#    Copyright (C) 2004-2010 Tiny SPRL (<http://tiny.be>).
#
#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU Affero General Public License as
#    published by the Free Software Foundation, either version 3 of the
#    License, or (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU Affero General Public License for more details.
#
#    You should have received a copy of the GNU Affero General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
##############################################################################

import time
import datetime
import calendar

import addons
from report import report_sxw
import tools
from tools.translate import _

from hrself.hrself_datetime import to_date, period

DAY = datetime.timedelta(days=1)
MONTH_NAME = {1: 'January', 2: 'February', 3: 'March', 4: 'April', 5: 'May', 6: 'June', 7: 'July', 8: 'August', 9: 'September', 10: 'October', 11: 'November', 12: 'December'}
DEFAULT_BACKGROUND_COLOR = ''

class ReportData(dict):
    def __getattr__(self, name):
        return self[name]

class planned_leaves(report_sxw.rml_parse):
    def __init__(self, cr, uid, name, context):
        super(planned_leaves, self).__init__(cr, uid, name, context=context)
        self.localcontext.update({
            'time': time,
            'report_name': self.report_name(),
            'title': self.title,
            'month_name': self.month_name,
            'month_length': self.month_length,
            'get_employees_leave_data': self.get_employees_leave_data,
            'holidays_types': self.holidays_types,
            'universe': self.pool.get('hr.employee').browse(cr, uid, self.pool.get('hr.employee').search(cr, uid, [('user_id', '=', uid)]))[0].universe_id,
        })
        
    def report_name(self):
        return _("Planned Leaves")
    
    def title(self, form):
        return _("From: %s To: %s") % (self.formatLang(form['date_from'], date=True), self.formatLang(form['date_to'], date=True))
    
    def month_name(self, month_number):
        return _(MONTH_NAME[month_number])
    
    def month_length(self, date):
        return calendar.monthrange(date.year, date.month)[1]

    def _itermonths(self, start, end):
        """Returns an iterator for the months between two dates. The iterator
        yields a date and a duration in days for each month (partial or complete,
        depending on start and end).
        
        >>> list(_itermonths(datetime.datetime(2010, 12, 25), datetime.datetime(2011, 2, 1)))
        [(datetime.datetime(2010, 12, 25, 0, 0), 7), (datetime.datetime(2011, 1, 1, 0, 0), 31), (datetime.datetime(2011, 2, 1, 0, 0), 1)]
        """
        
        while start <= end:
            if start.month == end.month:
                duration = end.day
            else:
                duration = calendar.monthrange(start.year, start.month)[1]
            if start.day != 1:
                duration -= start.day - 1
            result = (start, duration)
            start += result[1] * DAY
            yield result

    def get_employees_leave_data(self, form):
        """Returns all data necessary to print the report about planned leaves
        for selected employees between two dates.
        @param form: the data returned by the wizard form, implicitly injected in the .rml report context as data['form']."""
       
        employee_object = self.pool.get('hr.employee')
        leave_object = self.pool.get('hrself.holidays.request.leave')

        employees = employee_object.browse(self.cr, self.uid, form['employee_ids'])
        start_date = to_date(form['date_from'])
        end_date = to_date(form['date_to'])

        datas = []
        # Performance notice: if something is slow, then consider reordering the
        # loops below to minimize DB hits.  Loops order have been chosen to simplify
        # code and to be sufficiently fast for the main use case of this report 
        # (show planned leaves of my direct subordinates for the current month).
        for date, duration in self._itermonths(start_date, end_date):
            emps_leaves = []
            for employee in employees:
                leave_ids = leave_object.search_planned_leaves_by_employee_in_date_range(self.cr, self.uid, employee.id, date, date + duration * DAY)
                day_range = range(date.day, date.day + duration)
                from collections import defaultdict
                leave_color_by_day = defaultdict(lambda: DEFAULT_BACKGROUND_COLOR)
                for leave in leave_object.browse(self.cr, self.uid, leave_ids):
                    # a leave could span across previous and/or next month
                    for day in day_range:
                        if date.replace(day=day) in period(leave.date_from, leave.date_to):
                            leave_color_by_day[day] = leave.type_id.color_name
                emps_leaves.append(ReportData({'employee_name': employee.name_get()[0][1], 'leave_color_by_day': leave_color_by_day}))
            datas.append(ReportData({'date': date, 'emps_leaves': emps_leaves}))
        return datas            

    def holidays_types(self):
        """HRSelf holidays types."""
        type_object = self.pool.get('hrself.holidays.type')
        type_ids = type_object.search(self.cr, self.uid, [])
        return type_object.browse(self.cr, self.uid, type_ids)

report_sxw.report_sxw(
    'report.hrself.planned.leaves',
    'hrself.holidays.request.leave',
    addons.get_module_resource('hrself_holidays', 'report', 'planned_leaves.rml'),
    parser=planned_leaves,
    header=False,
)

class planned_activities(planned_leaves):

    def report_name(self):
        return _("Planned Activities")
    
    def get_employees_leave_data(self, form):
        """Returns all data necessary to print the report about planned activities
        for selected employees between two dates.
        @param form: the data returned by the wizard form, implicitly injected in the .rml report context as data['form']."""
       
        employee_object = self.pool.get('hr.employee')
        activity_object = self.pool.get('hrself.holidays.theoretical.activity')

        employees = employee_object.browse(self.cr, self.uid, form['employee_ids'])
        start_date = to_date(form['date_from'])
        end_date = to_date(form['date_to'])

        datas = []
        for date, duration in self._itermonths(start_date, end_date):
            emps_data = []
            for employee in employees:
                activity_ids = activity_object.search_planned_activities_by_employee_in_date_range(self.cr, self.uid, employee.id, date, date + duration * DAY)
                activities = activity_object.browse(self.cr, self.uid, activity_ids)
                monthly_activities = dict()
                for a in activities:
                    key = to_date(a.date_from).day
                    if a.duration:
                        value = "%s %dh" % (a.name, a.duration)
                    else:
                        value = a.name
                    monthly_activities[key] = value
                emps_data.append(ReportData({'employee_name': employee.name_get()[0][1], 'monthly_activities': monthly_activities}))
            datas.append(ReportData({'date': date, 'emps_data': emps_data}))
        return datas            

report_sxw.report_sxw(
    'report.hrself.planned.activities',
    'hrself.holidays.request.leave',
    addons.get_module_resource('hrself_holidays', 'report', 'planned_activities.rml'),
    parser=planned_activities,
    header=False,
)
# vim:expandtab:smartindent:tabstop=4:softtabstop=4:shiftwidth=4:

