# -*- coding: utf-8 -*-
##############################################################################
#
#    OpenERP, Open Source Management Solution
#    Copyright (C) 2004-2010 Tiny SPRL (<http://tiny.be>).
#
#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU Affero General Public License as
#    published by the Free Software Foundation, either version 3 of the
#    License, or (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU Affero General Public License for more details.
#
#    You should have received a copy of the GNU Affero General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
##############################################################################

from __future__ import with_statement
import logging
import xmlrpclib
import operator

# specific to the stub
import random

import tools
from osv import osv, fields
from tools.translate import _
from hrself import hrself_datetime, hrself, hrself_services

THEORETICAL_ACTIVITY_ERRORS = {
    -1000: 'Unknown regime',
    -1001: 'Variable regime',
    -1002: 'Unknown regime cycle start',
    -2000: 'Unknown contract',
}

logger = logging.getLogger('hrself.services')

class hrself_holidays_services(osv.osv_memory):
    """HRSelf holidays services factory."""

    _inherit = 'hrself.services'

    def hrself_holidays_service(self, cr, uid, universe_id=None, context=None):
        """Returns the HRSelf holidays service associated to the given arguments."""
        return self._service(cr, uid, ProxyService, ServiceStub, universe_id=universe_id, context=context)
        
hrself_holidays_services()

class Service(object):
    """Interface to services needed by the HRSelf project."""

    def __init__(self, url, pool):
        self.logger = logging.getLogger(self.__class__.__name__)
        self.url = url
        self.pool = pool

    def get_theoretical_activities(self, cr, uid, ids, date_from, date_to, enforce_uniqueness=True, context=None):
        """Returns theoretical activities for a set of employees and a period.
        @param ids: employee(s) identifier(s)
        @type ids: list of int or int
        @param date_from: leave start date
        @param date_to: leave end date
        @param enforce_uniqueness: enforce uniqueness of activities
        @return: for each employee, a list of theoretical activity.
        @rtype: list of list or list
        """
        raise NotImplementedError()

    def all_identical(self, activities, name):
        """Checks that all values for the name key are identical in the activities list."""
        l = map(operator.itemgetter(name), activities)
        return all(e==l[0] for e in l[1:])

    def register_event(self, cr, uid, leave_request_id, context=None):
        """Transfers the leave request to Presta."""
        raise NotImplementedError()

class ServiceStub(Service):
    """Simulates proxy service for decoupled testing.
    We do not simulate an xmlrpc server so we do not test for xmlrpc exception.
    We do not simulate activity context create_or_update so we don't call hrself.manager."""

    def _get_seed(self, employee_id, datetime_from, datetime_to):
        # postgresql seed must be in range [-1, 1], sinus of the sum seems a good candidate
        import math, time
        t1 = time.mktime(datetime_from.timetuple())
        t2 = time.mktime(datetime_to.timetuple())
        value = employee_id + t1 + t2
        seed = math.sin(value)
        return seed

    def _get_random_activity_context_ids(self, cr, uid, employee_id, date_from, date_to, n=1):
        """Fetches at most n different random rows:
        if the table contains r rows, r < n, then only r rows are returned, in random order.
        """
        # initializes random seed with f(employee_id, date_from, date_to) to make tests reproducible
        seed = self._get_seed(employee_id, date_from, date_to)
        cr.execute("SELECT setseed(%s)", (seed,))
        cr.execute("SELECT id FROM hrself_activity_context ORDER BY RANDOM() LIMIT %s", str(n))
        rows = cr.fetchall()
        return map(operator.itemgetter(0), rows)

    def _absence_only_in_weekend(self, date):
        if date.isoweekday() >= 6:
            presence = "A"
            duration = 0
        else:
            presence = "P"
            duration = 8 * 60 # in minutes
        return presence, duration
        
    def _get_fake_theoretical_activities_for_employee(self, cr, uid, employee_id, date_from, date_to, n=1):
        """@param n: specifies how many different activity contexts (at most) the specified period may have
        """
        activities = []
        datetime_from = hrself_datetime.to_datetime(date_from)
        datetime_to = hrself_datetime.to_datetime(date_to)
        random_activity_context_ids = self._get_random_activity_context_ids(cr, uid, employee_id, datetime_from, datetime_to, n)
        if random_activity_context_ids:
            # initializes random seed with f(employee_id, date_from, date_to) to make tests reproducible
            seed = self._get_seed(employee_id, datetime_from, datetime_to)
            random.seed(seed)
            while datetime_from <= datetime_to:
                presence_absence, duration = self._absence_only_in_weekend(datetime_from)
                activity = {
                    "date": xmlrpclib.DateTime(datetime_from),
                    "am_pm": "A", # random.choice(["A", "P"]),
                    "duration": duration, # in minutes
                    "presence_absence": presence_absence,
                    "activity_context": random.choice(random_activity_context_ids), 
                }
                activities.append(activity)
                datetime_from += 1 * hrself_datetime.DAY
        return activities

    def get_theoretical_activities(self, cr, uid, employee_id, date_from, date_to, enforce_uniqueness=True, context=None):
        """
        @param enforce_uniqueness: needs be True in the context of a leave request
        """
        activities = self._get_fake_theoretical_activities_for_employee(cr, uid, employee_id, date_from, date_to)
        if enforce_uniqueness:
            if not self.all_identical(activities, 'activity_context'):
                raise osv.except_osv(_('Error !'), "Multiple activity contexts")
        return activities

    # TODO: is it register (for callback) or record/notification (for archival/notification)
    def register_event(self, cr, uid, leave_request_id, context=None):
        """Transfers the leave request to Presta."""
        leave_request = self.pool.get('hrself.holidays.request.leave').browse(cr, uid, leave_request_id)
        precision = self.pool.get('hrself.holidays.type.precision').browse(cr, uid, leave_request.type_id.precision_id_for_context(leave_request.activity_context_id.id))
        if precision.prestation_code_id:
            employee = self.pool.get('hr.employee').browse(cr, uid, leave_request.employee_id.id)
            event = [
                ('universe_id', employee.universe_id.groupes_id),
                ('person_id', employee.groupes_id),
                ('mnemonic', precision.prestation_code_id.mnemonic),
                ('datetime_from', hrself_datetime.to_datetime(leave_request.date_from, hrself_datetime.DATE_FORMAT)),
                ('datetime_to', hrself_datetime.to_datetime(leave_request.date_to, hrself_datetime.DATE_FORMAT)),
                ('duration', leave_request.duration),
            ]
            self.logger.debug("if existe_personne then record event: %s", event)

class hrself_holidays_theoretical_activity_error(osv.osv_memory):
    """Error returned by the service get_theoretical_activities."""

    _name = 'hrself.holidays.theoretical.activity.error'

    _columns = {
        'employee_id': fields.many2one('hr.employee', 'Employee', required=True),
        'message': fields.selection(THEORETICAL_ACTIVITY_ERRORS.items(), 'Error message', required=True),
    }

hrself_holidays_theoretical_activity_error()

class ProxyService(Service):
    """Service implemented through a proxy."""

    def __init__(self, url, pool):
        """Constructor."""
        super(ProxyService, self).__init__(url, pool)
        self.url = url
        self.pool = pool

    def get_theoretical_activities(self, cr, uid, ids, date_from, date_to, enforce_uniqueness=True, context=None):
        """Returns the theoretical activities for a list of employee ids and a period.
        :param employee_ids: employee ids
        :type employee_ids: list of int or int
        :param date_from: start date
        :param date_to: end date
        :return: for each employee, a list of theoretical activities
        :rtype: dict of list where keys are employee ids and 
        values list or list
        """
        result = {}
        datetime_from = hrself_datetime.to_datetime(date_from)
        datetime_to = hrself_datetime.to_datetime(date_to)
        person_ids = {}
        if isinstance(ids, int):
            employee_ids = [ids]
        else:
            employee_ids = ids
        for employee in self.pool.get('hr.employee').browse(cr, uid, employee_ids):
            universe_id = employee.universe_id.groupes_id
            if not employee.external:
                person_ids[employee.groupes_id] = employee.id
        with hrself_services.HRSelfContextManager(cr, self.url, 'hrself', context=context) as context_manager:
            if context_manager.is_connected():
                persons_activities = context_manager.proxy.hrself.activites_theoriques_personnes(universe_id, person_ids.keys(), datetime_from, datetime_to)
        for (person_id, activities) in persons_activities.items():
            employee_id = person_ids[int(person_id)]
            #int activities means there was an error and activities is the error code
            if isinstance(activities, int):
                self.pool.get('res.log').create(cr, uid, {
                    'name': THEORETICAL_ACTIVITY_ERRORS[activities],
                    'res_model': 'hr.employee',
                    'res_id': employee_id,
                }, context=context)
                if len(employee_ids) == 1:
                    raise osv.except_osv(_('Error !'), _( """Unable to complete this operation.
                                                         Please contact your functional administrator."""))
                else:
                    result[employee_id] = activities
            else:
                if enforce_uniqueness:
                    if not self.all_identical(activities, 'activity_context'):
                        raise osv.except_osv(_('Error !'), "Multiple activity contexts")
                result[employee_id] = self._theoretical_activities(cr, uid, activities, employee_id, enforce_uniqueness)
        if isinstance(ids, int) and len(result) == 1:
            result = result[result.keys()[0]]
        hrself_services.log('theoretical_activities', result)
        return result

    def _theoretical_activities(self, cr, uid, activities, employee_id, enforce_uniqueness):
        """Returns HRSelf theoretical activities corresponding to Groupe S theoretical activities.
        @param activities: Groupe S theoretical activities
        @type activities: list
        @return: HRSelf theoretical activities
        @rtype: list
        """
        activity_context_id = 0
        for activity in activities:
            activity_context = activity['activity_context']
            if (not activity_context_id and enforce_uniqueness) or not enforce_uniqueness:
                activity_context_id = self.pool.get('hrself.manager').create_hrself_activity_context(cr, uid, activity_context)
            activity['activity_context'] = activity_context_id
            activity['employee_id'] = employee_id
        return activities

    def get_counter_balances(self, cr, uid, employee_ids, counter_type_ids, validity_date, context=None):
        """Returns for every employee the balances of each counter type at a given date.
        @param employee_ids: employees identifiers
        @type employee_ids: list of int
        @param counter_type_ids: counter types identifiers
        @type counter_type_ids: list of int
        @param validity_date: validity date
        @return: for every employee, the balances of each counter type.
        @rtype: list of dict(employee_id, activity_context_id, counter_type_id, value)
        """
        result = []
        counter_type_object = self.pool.get('hrself.counter.type')
        date = hrself_datetime.to_datetime(validity_date, hrself_datetime.DATE_FORMAT)
        person_ids = {}
        counter_types = []
        for employee in self.pool.get('hr.employee').browse(cr, uid, employee_ids):
            universe_id = employee.universe_id.groupes_id
            person_ids[employee.groupes_id] = employee.id
        for counter_type in counter_type_object.browse(cr, uid, counter_type_ids):
            counter_types.append({
                'mnemonic': counter_type.mnemonic,
                'employer_id': counter_type.employer_id.groupes_id
            })
        with hrself_services.HRSelfContextManager(cr, self.url, 'hrself', context=context) as context_manager:
            if context_manager.is_connected():
                balances = context_manager.proxy.hrself.soldes_compteurs_personnes(universe_id, person_ids.keys(), counter_types, date)
        for balance in balances:
            employer_id = self.pool.get('hrself.employer').search(cr, uid, [
                ('groupes_id', '=', balance['counter_type_id']['employer_id']),
            ])[0]
            activity_context_id = self.pool.get('hrself.manager').create_hrself_activity_context(cr, uid, balance['activity_context_id'])
            result.append(dict(balance, **{
                'employee_id': person_ids[int(balance['employee_id'])],
                'counter_type_id': counter_type_object.search(cr, uid, [
                    ('mnemonic', '=', balance['counter_type_id']['mnemonic']),
                    ('employer_id', '=', employer_id),
                ])[0],
                'activity_context_id': activity_context_id,
                'date_from': validity_date,
            }
            ))
        tools.debug(result)
        return result

    def get_counter_movements(self, cr, uid, employee_ids, counter_type_ids, date_from, date_to, context=None):
        """Returns for every employee the movements of each counter type between two dates.
        @param employee_ids: employees identifiers
        @type employee_ids: list of int
        @param counter_type_ids: counter types identifiers
        @type counter_type_ids: list of int
        @param date_from: movements start date
        @param date_to: movements end date
        @return: for every employee, the movements of each counter type.
        @rtype: list of dict
        """
        result = []
        counter_type_object = self.pool.get('hrself.counter.type')
        datetime_from = hrself_datetime.to_datetime(date_from, hrself_datetime.DATE_FORMAT)
        datetime_to = hrself_datetime.to_datetime(date_to, hrself_datetime.DATE_FORMAT)
        person_ids = {}
        counter_types = []
        for employee in self.pool.get('hr.employee').browse(cr, uid, employee_ids):
            universe_id = employee.universe_id.groupes_id
            person_ids[employee.groupes_id] = employee.id
        for counter_type in counter_type_object.browse(cr, uid, counter_type_ids):
            counter_types.append({
                'mnemonic': counter_type.mnemonic,
                'employer_id': counter_type.employer_id.groupes_id
            })
        with hrself_services.HRSelfContextManager(cr, self.url, 'hrself', context=context) as context_manager:
            if context_manager.is_connected():
                movements = context_manager.proxy.hrself.mouvements_compteurs_personnes(universe_id, person_ids.keys(), counter_types, datetime_from, datetime_to)
        for movement in movements:
            employer_id = self.pool.get('hrself.employer').search(cr, uid, [
                ('groupes_id', '=', movement['counter_type_id']['employer_id']),
            ])[0]
            result.append(dict(movement, 
                               date_from = hrself_datetime.to_date(movement['date_from'], hrself_datetime.XML_RPC_DATETIME_FORMAT),
                               date_to = hrself_datetime.to_date(movement['date_to'], hrself_datetime.XML_RPC_DATETIME_FORMAT),
                               employee_id = person_ids[int(movement['employee_id'])],
                               counter_type_id = counter_type_object.search(cr, uid, [
                                   ('mnemonic', '=', movement['counter_type_id']['mnemonic']),
                                   ('employer_id', '=', employer_id),
                               ])[0],
                              ))
        tools.debug(result)
        return result

    def register_event(self, cr, uid, leave_request_id, context=None):
        """Transfers the leave request to Presta."""
        leave_request = self.pool.get('hrself.holidays.request.leave').browse(cr, uid, leave_request_id)
        precision = self.pool.get('hrself.holidays.type.precision').browse(cr, uid, leave_request.type_id.precision_id_for_context(leave_request.activity_context_id.id))
        if precision.prestation_code_id:
            employee = self.pool.get('hr.employee').browse(cr, uid, leave_request.employee_id.id)
            universe_id = employee.universe_id.groupes_id
            person_id = employee.groupes_id
            with hrself_services.HRSelfContextManager(cr, self.url, 'hrself', context=context) as context_manager:
                if context_manager.is_connected():
                    if context_manager.proxy.hrself.existe_personne(universe_id, person_id):
                        datetime_from = hrself_datetime.to_datetime(leave_request.date_from)
                        datetime_to = hrself_datetime.to_datetime(leave_request.date_to)
                        if leave_request.date_from == leave_request.date_to and leave_request.unit == hrself.TIME_UNIT_HOUR:
                            duration = leave_request.duration * 60
                        else:
                            duration = 0.0
                        context_manager.proxy.hrself.enregistrer_evenement(universe_id, person_id, precision.prestation_code_id.mnemonic, datetime_from, datetime_to, duration)
                    else:
                        raise osv.except_osv(_('Error !'), _("""The person identified by (%d, %d) doesn't exist!""") % (universe_id, person_id))
