# -*- coding: utf-8 -*-
##############################################################################
#
#    OpenERP, Open Source Management Solution
#    Copyright (C) 2004-2010 Tiny SPRL (<http://tiny.be>).
#
#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU Affero General Public License as
#    published by the Free Software Foundation, either version 3 of the
#    License, or (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU Affero General Public License for more details.
#
#    You should have received a copy of the GNU Affero General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
##############################################################################

import logging
import datetime
import exceptions

import netsvc
from osv import fields, osv
from tools.translate import _
import workflow

from hrself import hrself_datetime, hrself

STATE_DRAFT = 'draft'
STATE_CONFIRMED = 'confirmed'
STATE_VALIDATED = 'validated'
STATE_REFUSED = 'refused'
STATE_CANCELLED = 'cancelled'

# states of a leave request
STATES = [
    (STATE_DRAFT, 'Draft'),
    (STATE_CONFIRMED, 'Confirmed'),
    (STATE_VALIDATED, 'Validated'),
    (STATE_REFUSED, 'Refused'),
    (STATE_CANCELLED, 'Cancelled'),
]

PROCESS_TYPE_STANDARD = 'standard'
PROCESS_TYPE_LEAVE_APPROVEMENT = 'leave approvement'

# process type defines for which responsibility an employee is on leave / absent
PROCESS_TYPE = [
    (PROCESS_TYPE_LEAVE_APPROVEMENT, 'Leave Approvement'),
]

class NoDefaultWorkflow(exceptions.LookupError):
    pass

class hrself_holidays_reason(osv.osv):
    """Reason of holidays"""

    _name = "hrself.holidays.reason"
    _description = "Reason of holidays"

    _columns = {
        'name': fields.char('Name', size=64, required=True, translate=True),
    }

hrself_holidays_reason()    

class hrself_holidays_type(osv.osv):
    """Types of holidays (normal holidays, sick, working time reduction,...)."""

    _name = "hrself.holidays.type"
    _description = "Types of holidays"

    def search(self, cr, uid, args, offset=0, limit=None, order=None, context=None, count=False):
        """Redefined to search only types applicables for an employee and a period.
            If context contains activity_context_id (or employee_id and date_from to calculates it), returns only compatible type.
            Else returns super.search result
        """

        if not context:
            context = {}
        ids = super(hrself_holidays_type, self).search(cr, uid, args, offset, limit, order, context, count)
        result = ids
        activity_context_id = context.get('activity_context_id')
        if not activity_context_id:
            employee_id = context.get('employee_id')
            date_from = context.get('date_from')
            date_to = context.get('date_to')
            if not date_to:
                date_to = date_from
            if employee_id and date_from:
                services = self.pool.get('hrself.services').hrself_holidays_service(cr, uid, context=context)
                activities = services.get_theoretical_activities(cr, uid, employee_id, date_from, date_to, context=context)
                if activities:
                    activity_context_id = activities[0]['activity_context']
        if activity_context_id:
            result = []
            for leave_type in self.browse(cr, uid, ids):
                for precision in sorted(leave_type.precision_ids, key=lambda precision: precision.order, reverse=True):
                    if precision.activity_context_id.is_compatible(activity_context_id):
                        result.append(leave_type.id)
                        break
        return result

    def precision_id_for_context(self, cr, uid, ids, activity_context_id):
        """Precision identifier for an activity context."""
        for type in self.browse(cr, uid, ids):
            for precision in sorted(type.precision_ids, key=lambda precision: precision.order, reverse=True):
                if precision.activity_context_id.is_compatible(activity_context_id):
                    return precision.id
        return None

    def _user_employee(self, cr, uid, context=None):
        """Employee id, if it exists, corresponding to the user id.
        @return: employee id
        @rtype: int"""
        employee_ids = self.pool.get('hr.employee').search(cr, uid, [('user_id','=', uid)])
        if employee_ids:
            return employee_ids[0]
        return False

    def get_initialization_duration_for_employee(self, cr, uid, ids, context):
        """Returns the initialization duration of a leave type for the employee. Only for the current year and activity context.
        Duration is in hours or days (defined by leave type).
        Requires employee_id, date_from and activity_context of the leave in context:
        {'employee_id':employee_id, 'date_from':date_from, 'activity_context':activity_context}
        @return: dictionary {leave_type: duration}
        """
        max_leaves = dict.fromkeys(ids, 0)
        employee_id = context.get('employee_id')
        # calculation of initializations at the date of leave request
        # if leave request has no date_from yet, using today
        self._update_context(cr, uid, context=context)
        date_from = context.get('date_from')
        activity_context_id = context.get('activity_context')
        current_year = hrself_datetime.to_date(date_from).year
        start_of_year = hrself_datetime.to_string(datetime.date(current_year, 1, 1))
        for leave_type in self.browse(cr, uid, ids, context): 
            initialization_object = self.pool.get('hrself.holidays.request.initialization')
            initialization_ids = initialization_object.search(cr, uid, [
                    ('type_id', '=', leave_type.id), ('employee_id', '=', employee_id),
                    ('date_from', '>=', start_of_year), ('date_from', '<=', date_from)],
                    order='date_from desc')
            duration = 0
            if initialization_ids:
                for initialization in initialization_object.browse(cr, uid, initialization_ids, context=context):
                    if initialization.activity_context_id.is_compatible(activity_context_id):
                        duration = initialization.duration
                        break
            max_leaves[leave_type.id] = duration
        return max_leaves

    def get_remaining_leaves_for_employee(self, cr, uid, ids, context=None, remaining_date_to=None):
        """Returns the remaining number of leaves for the employee by leave type. Only for the current year and activity context.
           Context must contain employee_id, date_from and activity_context of the leave:
           {'employee_id':employee_id, 'date_from':date_from, 'activity_context':activity_context}
           If not, employee_id based on res.user, date_from = today, activity_context calculated
           Uses date_from from context to specify current year.
           remaining_date_to is end of year by default, can be used to specify an end date period for calculation of remaining leaves.
           @return: dictionary {leave_type: duration}
        """
        result = {}
        values = self.compute_values(cr, uid, ids, context=context, remaining_date_to=remaining_date_to)
        for key, value in values.iteritems():
            result[key] = value['remaining_leaves']
        return result

    def _update_context(self, cr, uid, context=None):
        """Update context passed as argument."""
        employee_id = context.get('employee_id')
        if not employee_id:
            employee_id = self._user_employee(cr, uid, context)
            context['employee_id'] = employee_id
        date_from = context.get('date_from')
        if not date_from:
            date_from = datetime.date.today()
            context['date_from']=date_from
        if not context.get('activity_context'):
            services = self.pool.get('hrself.services').hrself_holidays_service(cr, uid, context=context)
            activities = services.get_theoretical_activities(cr, uid, employee_id, date_from, date_from, context=context)
            if activities:
                context['activity_context'] = activities[0]['activity_context']

    def compute_values(self, cr, uid, ids, context, remaining_date_to=None):
        """Compute values of remaining_leaves and unit functional fields."""
        return self._compute(cr, uid, ids, None, None, context=context, remaining_date_to=remaining_date_to)

    def _compute(self, cr, uid, ids, field_name, arg, context, remaining_date_to=None):
        """Method signature conformed to functional field."""
        result = dict.fromkeys(ids, {'unit': None, 'remaining_leaves': 0})
        self._update_context(cr, uid, context=context)
        remaining_leaves = self.get_initialization_duration_for_employee(cr, uid, ids, context)
        date_from = context.get('date_from')
        current_year = hrself_datetime.to_date(date_from).year
        start_of_year = hrself_datetime.to_string(datetime.date(current_year, 1, 1))
        if remaining_date_to:
            date_to = remaining_date_to
        else:
            date_to = hrself_datetime.to_string(datetime.date(current_year, 12, 31))
        activity_context_id = context.get('activity_context')
        leave_object = self.pool.get('hrself.holidays.request.leave')
        for leave_type in self.browse(cr, uid, ids, context): 
            leave_ids = leave_object.search(cr, uid, [
                    ('type_id', '=', leave_type.id), ('employee_id', '=', context.get('employee_id')),
                    ('date_from', '>=', start_of_year), ('date_to', '<=', date_to), 
                    ('state', 'in', [STATE_CONFIRMED, STATE_VALIDATED] ), ('activity_context_id','=', activity_context_id) ])
            remaining_leaves[leave_type.id] -= sum(leave.duration for leave in leave_object.browse(cr, uid, leave_ids))
            precision_id = leave_type.precision_id_for_context(activity_context_id)
            if precision_id:
                result[leave_type.id] = {
                    'remaining_leaves': remaining_leaves[leave_type.id],
                    'unit': self.pool.get('hrself.holidays.type.precision').browse(cr, uid, precision_id).unit,
                }
        return result

    _columns = {
        'name': fields.char('Name', size=64, required=True, translate=True),
        'active': fields.boolean('Active', help="If the active field is set to false, it will allow you to hide the leave type without removing it."),
        'remaining_leaves': fields.function(_compute, method=True, string='Remaining Leaves', help='Remaining leaves', multi='function'),
        'unit': fields.function(_compute, method=True, type='selection', selection=hrself.TIME_UNITS, string='Time unit', multi='function'),
        'reason_mandatory': fields.boolean('Reason mandatory'),
        'reason_ids': fields.one2many('hrself.holidays.reason', 'type_id'),
        'color_name': fields.selection([('red', 'Red'), ('lightgreen', 'Light Green'), ('lightblue','Light Blue'), ('lightyellow', 'Light Yellow'), \
                     ('magenta', 'Magenta'), ('lightcyan', 'Light Cyan'), ('black', 'Black'), ('lightpink', 'Light Pink'), ('brown', 'Brown'), \
                     ('violet', 'Violet'), ('lightcoral', 'Light Coral'), ('lightsalmon', 'Light Salmon'), ('lavender', 'Lavender'), \
                     ('wheat', 'Wheat'), ('ivory', 'Ivory')], 'Color in Report', required=True, \
                     help='This color will be used in the leaves summary located in Reporting\Leaves by Departement'),
    }

    _defaults = {
        'active': True,
    }

hrself_holidays_type()

class hrself_holidays_reason_extended(osv.osv):
    """Reason of holidays"""

    _inherit = "hrself.holidays.reason"

    _columns = {
        'type_id': fields.many2one('hrself.holidays.type', 'Leave type'),
    }

hrself_holidays_reason_extended()

class hrself_holidays_type_precision(osv.osv):
    """Defines the precision of holiday types."""

    _name = "hrself.holidays.type.precision"
    _inherit = "hrself.common.type"
    _description = "Holiday type precisions"

    _columns = {
        'type_id': fields.many2one('hrself.holidays.type', 'Leave type', required=True, select=True),
        'is_unlimited' : fields.boolean('Allow to Override Limit', help='If you thick this checkbox, the system will allow, for this section, the employees to take more leaves than the available ones.'),
        'priority_type_precision_id': fields.many2one('hrself.holidays.type.precision', 'Priority type precision', help='Leave type precision to deplete in priority'),
        'limit_date': fields.date('Limit date', help='Cannot take a leave after this limit date.'),
        'prestation_code_id': fields.many2one('hrself.prestation.code', 'Prestation code'),
        'activity_context_id': fields.many2one('hrself.activity.context', 'Activity context', required=True, select=True),
        'counter_type_id': fields.many2one('hrself.counter.type', 'Counter type'),
        'init_prestation_code_id': fields.many2one('hrself.prestation.code', 'Initialization prestation code'),
        'order': fields.related('activity_context_id', 'order', type='integer'),
    }
    
    def name_get(self, cr, uid, ids, context=None):
        """ Redefined as there is no name field."""
        res = []
        for precision in self.browse(cr, uid, ids, context):
            res.append((precision.id, 
                        precision.type_id.name_get(context=context)[0][1] + ' / ' 
                        + precision.activity_context_id.name_get(context=context)[0][1]))
        return res

    _defaults = {
        'unit': hrself.TIME_UNIT_HOUR,
    }

    def _check_counter_type(self, cr, uid, ids, context=None):
        """Check if there is already another leave type with the same counter type."""
        for precision in self.browse(cr, uid, ids, context=context):
            if precision.counter_type_id: 
                precision_ids = self.search(cr, uid, [
                    ('type_id', '!=', precision.type_id.id),
                    ('counter_type_id', '=', precision.counter_type_id.id),
                ], context=context)
                if precision_ids:
                    return False
        return True
    
    def _check_priority(self, cr, uid, ids, context=None):
        """Check that the priority precision is not the same as the current precision."""
        for precision in self.browse(cr, uid, ids, context=context):
            if precision.priority_type_precision_id and precision.priority_type_precision_id.id == precision.id: 
                return False
        return True
    
    _constraints = [
        (_check_counter_type, _('There is already another leave type with the same counter type'), ['counter_type_id']),
        (_check_priority, _('The priority precision cannot be the same as the current precision'), ['priority_type_precision_id']),
    ]

    def onchange_counter_type_id(self, cr, uid, ids, counter_type_id, context):
        """Action to execute when the user changes counter_type_id."""
        result = {}
        if counter_type_id:
            counter_type = self.pool.get('hrself.counter.type').browse(cr, uid, counter_type_id, context)
            if counter_type.has_initialization_value:
                result['value'] = {
                    'has_initialization_value': counter_type.has_initialization_value,
                    'init_value': counter_type.init_value,
                    'init_unit': counter_type.init_unit,
                    'init_periodicity': counter_type.init_periodicity,
                    'init_date': counter_type.init_date,
                }
        else:
            result['value'] = {
                'has_initialization_value': None,
                'init_value': None,
                'init_unit': None,
                'init_periodicity': None,
                'init_date': None,
            }
        return result

hrself_holidays_type_precision()

class hrself_holidays_type_extended(osv.osv):

    _inherit = "hrself.holidays.type"

    _columns = {
        'precision_ids': fields.one2many('hrself.holidays.type.precision', 'type_id'),
    }

hrself_holidays_type_extended()

class hrself_holidays_request(osv.osv):
    """Abstract class, there should be no record in the related table."""

    _name = "hrself.holidays.request"

    def _default_employee(self, cr, uid, context=None):
        """Employee id, if it exists, corresponding to the user id.
        @return: employee id
        @rtype: int"""

        employee_ids = self.pool.get('hr.employee').search(cr, uid, [('user_id', '=', uid)])
        if employee_ids:
            return employee_ids[0]
        return False

    def _get_label_for_calendar(self, cr, uid, ids, field_name, arg, context=None):
        """Labels for display in calendar view
        @return: A calendar label for each request.
        """
        labels = dict.fromkeys(ids, "NOLABEL")

        for request in self.browse(cr, uid, ids, context=context): 
            if request.employee_id:
                emp_name = request.employee_id.name
                labels[request.id] = "%s : %s" % (request.state, emp_name)
        return labels

    _columns = {
        'name': fields.char('Description', readonly=True, size=64, states={STATE_DRAFT:[('readonly',False)]}),
        'date_from': fields.date('Start Date', required=True, readonly=True, states={STATE_DRAFT:[('readonly',False)]}),
        'date_to': fields.date('End Date', readonly=True, states={STATE_DRAFT:[('readonly',False)]}),
        'state': fields.selection(STATES, 'State', readonly=True),
        'notes': fields.text('Notes',readonly=True, states={STATE_DRAFT:[('readonly',False)]}),
        'calendar_label': fields.function(_get_label_for_calendar, method=True, type='string', string='Calendar label', help='Label for calendar view'),

        'employee_id': fields.many2one('hr.employee', "Employee", required=True, select=True, invisible=False, readonly=True, states={STATE_DRAFT:[('readonly',False)]}),

        # user that encoded the request (either for himself or for someone else)
        'user_id': fields.many2one('res.users', 'User', states={STATE_DRAFT:[('readonly',False)]}, select=True, readonly=True),
    }

    _defaults = {
        # state is not used in all osv of this type hierarchy as we only need a workflow on hrself.leave.request
        'state': STATE_DRAFT,
        'employee_id': _default_employee,
    }

    def _check_date_interval(self, cr, uid, ids, context=None):
        """Check to see if date_from is smaller or equal to date_to.
        """
        for request in self.browse(cr,uid,ids,context=context):
            if request.date_to and (request.date_from > request.date_to): 
                return False
        return True

    def _check_year(self, cr, uid, ids, context=None):
        """Ensures date_from and date_to are for the same year.
        @rtype: boolean
        """
        for request in self.browse(cr,uid,ids,context=context):
            if request.date_to:
                date_to_year = hrself_datetime.to_date(request.date_to).year
                date_from_year = hrself_datetime.to_date(request.date_from).year
                if date_to_year != date_from_year:
                    return False
        return True
                                                                                        
    _constraints = [
        (_check_date_interval, _('Date from must be smaller or equal to date to!'), ['date_from', 'date_to']),
        (_check_year, _('Date from must have the same year than date to!'), ['date_from', 'date_to']),
    ]

hrself_holidays_request()

class hrself_holidays_request_absence(osv.osv):
    """Absence is not subjected to a workflow so it does not use the 'state' field of its superclass."""

    _name = 'hrself.holidays.request.absence'
    _inherit = 'hrself.holidays.request'

    def _get_label_for_calendar(self, cr, uid, ids, field_name, arg, context=None):
        """Labels for display in calendar view
        @return: A calendar label for each request.
        """
        labels = dict.fromkeys(ids, "NOLABEL")

        for request in self.browse(cr, uid, ids, context=context): 
            if request.employee_id:
                emp_name = request.employee_id.name
                description = request.name or "NONAME"
                labels[request.id] = "%s : %s" % (emp_name, description)
        return labels

    _columns = {
        # ex: if absent for process_type=="leave approvement" then the employee is not on leave
        #     but he cannot endorse the role of leave approver for the specified period
        'process_type': fields.selection(PROCESS_TYPE, 'Process Type', help="Process for which the absence is requested"),
        'calendar_label': fields.function(_get_label_for_calendar, method=True, type='string', string='Calendar label', help='Label for calendar view'),
    }

    _defaults = {
        # TODO set default value in the view instead
        #'process_type': PROCESS_TYPE_LEAVE_APPROVEMENT,
    }

    def _is_absent_for_leave_approvement(self, cr, uid, employee_ids, date_from, date_to=None, context=None):
        """Returns True if the employee is declared as absent for the leave approvement processus within the specified
        period, i.e. he is not on leave but he cannot approve leave at least during a part of this period.
        Returns false otherwise.
        @rtype: boolean."""

        date_from = hrself_datetime.to_string(date_from)
        if date_to is None:
            date_to = date_from
        else:
            date_to = hrself_datetime.to_string(date_to)
        for employee in self.pool.get('hr.employee').browse(cr, uid, employee_ids, context):
            domain = ['&', ('employee_id', '=', employee.id), \
                      '|', '&', ('date_from', '<=', date_to), ('date_to', '>=', date_from), \
                      '&', ('date_from', '>=', date_from), ('date_to', '<=', date_to),
                     ]
            return self.search(cr, uid, domain, count=True) != 0

hrself_holidays_request_absence()

class hrself_holidays_request_initialization(osv.osv):
    """Initialization request."""

    _name = 'hrself.holidays.request.initialization'
    _inherit = 'hrself.holidays.request'

    _columns = {
        'duration': fields.float('Duration', required=True, readonly=True, states={STATE_DRAFT:[('readonly',False)]}, help="Duration value of holiday, the unit is given by the type of holiday (day, hour)"),
        'type_id': fields.many2one("hrself.holidays.type", "Leave Type", required=True, readonly=True, states={STATE_DRAFT:[('readonly',False)]}),
        'activity_context_id': fields.many2one('hrself.activity.context', 'Activity context', readonly=True, \
                                               help='Filled when someone set an initialization for an employee'),
        'generate_date': fields.datetime('Generate date'),
        'generate_source': fields.selection([
            ('p', 'Leave type precision'), 
            ('a', 'Annual initialization'),
        ], 'Generate source'),
    }

    def write(self, cr, uid, ids, vals, context=None):
        """Redefining write to add activity context """
        for initialization in self.browse(cr, uid, ids, context=context):
            date_from = vals.get('date_from', initialization.date_from)
            employee_id = initialization.employee_id.id
            services = self.pool.get('hrself.services').hrself_holidays_service(cr, uid, context=context)
            activities = services.get_theoretical_activities(cr, uid, employee_id, date_from, date_from, context=context)
            if activities:
                activity_context_id = activities[0]['activity_context']
                type_id = self.browse(cr, uid, ids[0], context=context).type_id.id
                precision_id = self.pool.get('hrself.holidays.type').precision_id_for_context(cr, uid, [type_id], activity_context_id)
                vals['activity_context_id'] = self.pool.get('hrself.holidays.type.precision').browse(cr, uid, precision_id, context=context).activity_context_id.id
                # test if new duration > number of days/hours already taken
                context = {'employee_id':employee_id, 'date_from':date_from, 'activity_context':activity_context_id}
                old_duration_dic = initialization.type_id.get_initialization_duration_for_employee(context)
                remaining_dic = initialization.type_id.get_remaining_leaves_for_employee(context)
                remaining = remaining_dic[initialization.type_id.id]
                old_duration = old_duration_dic[initialization.type_id.id]
                days_taken = old_duration - remaining
                new_duration = vals.get('duration', initialization.duration)
                if new_duration < days_taken:
                    raise osv.except_osv(_('Error !'), _( 'Duration must be greater than days/hours already taken.\n Days/Hours taken: %s') % (str(days_taken))) 
        return super(hrself_holidays_request_initialization, self).write(cr, uid, ids, vals, context=context)
    
    def create(self, cr, uid, vals, context=None):
        """Redefining create to add activity context """
        if not 'activity_context_id' in vals:
            date_from = vals['date_from']
            employee_id = int(vals['employee_id'])
            services = self.pool.get('hrself.services').hrself_holidays_service(cr, uid, context=context)
            activities = services.get_theoretical_activities(cr, uid, employee_id, date_from, date_from, context=context)
            if activities:
                activity_context_id = activities[0]['activity_context']
                precision_id = self.pool.get('hrself.holidays.type').precision_id_for_context(cr, uid, [vals['type_id']], activity_context_id)
                vals['activity_context_id'] = self.pool.get('hrself.holidays.type.precision').browse(cr, uid, precision_id, context=context).activity_context_id.id
        return super(hrself_holidays_request_initialization, self).create(cr, uid, vals, context=context)
    
    _defaults = {
        'employee_id': False,
        'date_from': hrself_datetime.to_string(datetime.date.today()),
    }

    def _check_type_activity_context(self, cr, uid, ids):
        """Check if type is compatible with activity context. 

        @rtype:boolean"""
        for initialization in self.browse(cr, uid, ids):
            if initialization.activity_context_id:
                context = {'activity_context_id': initialization.activity_context_id.id}
            else:
                context = {
                    'employee_id': initialization.employee_id.id,
                    'date_from': initialization.date_from,
                    'date_to': initialization.date_from ,
                }
            domain =  []
            type_ids = self.pool.get('hrself.holidays.type').search(cr, uid, domain, context=context)
            if initialization.type_id.id in type_ids:
                return True
            return False

    def _check_duration(self, cr, uid, ids, context=None):
        """Check if duration is > 0.
        """
        for request in self.browse(cr,uid,ids,context=context):
            if request.duration < 1 :
                return False
        return True

    _constraints = [
        (_check_duration, _('Duration must be greater than 0 !'), ['duration']),
        (_check_type_activity_context, _('This type is not compatible with activity context of employee'), ['type']),
    ]

hrself_holidays_request_initialization()

class hrself_holidays_request_leave(osv.osv):
    """Represents holidays / leave requests.
    Holidays require a type."""

    _name = "hrself.holidays.request.leave"
    _description = 'Leave request'
    _inherit = 'hrself.holidays.request'
    _order = '(CASE WHEN write_date is null THEN create_date ELSE write_date END) DESC'

    _columns = {
        'duration': fields.float('Duration', required=True, readonly=True, states={STATE_DRAFT:[('readonly',False)]}, help="Duration value of holiday, the unit is given by the type of holiday (day, hour)"),
        'unit': fields.selection(hrself.TIME_UNITS, 'Time unit', readonly=True, states={STATE_DRAFT:[('readonly',False)]}),
        'leave_approver_id': fields.many2one('hr.employee', 'Next Leave Approver', invisible=False, readonly=True, \
                help='Filled when the employee confirms this leave request'),
        'type_id': fields.many2one("hrself.holidays.type", "Leave Type", required=True, readonly=True, states={STATE_DRAFT:[('readonly',False)]}),
        'reason_id': fields.many2one('hrself.holidays.reason' ,'Reason'),
        'activity_context_id': fields.many2one('hrself.activity.context', 'Activity context', readonly=True, \
                help='Filled when the employee confirms this leave request'),
        'generate_date': fields.datetime('Generate date'),
    }

    def _check_mandatory_reason(self, cr, uid, ids, context=None):
        """Check mandatory reason"""
        for holiday in self.browse(cr,uid,ids,context=context):
            if holiday.type_id.reason_mandatory and not holiday.reason_id: 
                return False
        return True

    def _get_limit_date(self, cr, uid, ids):
        """Returns limit date or None if no limit date set.
        @rtype:date"""
        for leave_request in self.browse(cr, uid, ids):
            if leave_request.activity_context_id:
                activity_context_id = leave_request.activity_context_id.id
            else:
                date_to = leave_request.date_to
                date_from = leave_request.date_from
                employee_id = leave_request.employee_id.id
                services = self.pool.get('hrself.services').hrself_holidays_service(cr, uid)
                activities = services.get_theoretical_activities(cr, uid, employee_id, date_from, date_to)
                if activities:
                    activity_context_id = activities[0]['activity_context']
            if activity_context_id:
                precision_id = leave_request.type_id.precision_id_for_context(activity_context_id)
                type_precision = self.pool.get('hrself.holidays.type.precision').browse(cr, uid, precision_id)
                if type_precision.limit_date:
                    limit_date = hrself_datetime.to_date(type_precision.limit_date)
                    return limit_date
            return None

    def _check_limit_date_message(self, cr, uid, ids):
        """Returns error message for check_limit_date constraint.
           @rtype:string"""
        for leave_request in self.browse(cr, uid, ids):
            limit_date = self._get_limit_date(cr, uid, ids)
            return _('You may not make this request after %s'),  _(limit_date)

    def _check_limit_date(self, cr, uid, ids):
        """User cannot take a leave after a limit date if set.
        Returns False if date_to > limit_date, True otherwise.
        @rtype:boolean"""
        for leave_request in self.browse(cr, uid, ids):
            limit_date = self._get_limit_date(cr, uid, ids)
            date_to = hrself_datetime.to_date(leave_request.date_to)
            if limit_date and date_to > limit_date :
                return False
            return True

    def _check_type_activity_context(self, cr, uid, ids):
        """Check if type is compatible with activity context. 

        @rtype:boolean"""
        for leave_request in self.browse(cr, uid, ids):
            if leave_request.activity_context_id:
                context = {'activity_context_id': leave_request.activity_context_id.id}
            else:
                context = {'employee_id': leave_request.employee_id.id, 'date_from': leave_request.date_from, 'date_to': leave_request.date_to}
            domain = []
            type_ids = self.pool.get('hrself.holidays.type').search(cr, uid, domain, context=context)
            if leave_request.type_id.id in type_ids:
                return True
            return False

    def _check_duration(self, cr, uid, ids, context=None):
        """Check if duration is > 0.
        """
        for request in self.browse(cr,uid,ids,context=context):
            if request.duration < 1 :
                return False
        return True

    _constraints = [
        (_check_duration, _('Duration must be greater than 0 !'), ['duration']),
        (_check_mandatory_reason, _('Type of holiday requires a reason'), ['reason']),
        (_check_limit_date, _check_limit_date_message, ['date_to']),
        (_check_type_activity_context, _('This type is not compatible with activity context of employee'), ['type']),
    ]

    def write(self, cr, uid, ids, values, context=None):
        """Redefined write to check constraints."""
        logger = logging.getLogger('hrself.holidays.request.leave')
        write_ids = []

        def _to_date(value, default):
            """Converts value into a date or returns default if value is None."""
            if value:
                try:
                    result = hrself_datetime.to_date(value)
                except ValueError: #FIXME: In the calendar view, value is a str datetime, not a str date :(
                    result = hrself_datetime.to_date(value, hrself_datetime.DATETIME_FORMAT)
            else:
                result = default
            return result

        for leave in self.browse(cr, uid, ids, context=context):
            old_date_from = hrself_datetime.to_date(leave.date_from)
            new_date_from = _to_date(values.get('date_from'), old_date_from)
            old_date_to = hrself_datetime.to_date(leave.date_to)
            new_date_to = _to_date(values.get('date_to'), old_date_to)
            if (new_date_from != old_date_from) or (new_date_to != old_date_to):
                on_change = self._onchange_date_type(cr, uid, [leave.id], leave.employee_id.id, new_date_from, new_date_to, leave.type_id.id, context)
                warning = on_change.get('warning')
                if warning:
                    logger.warning(warning['message'])
                    raise osv.except_osv(warning['title'], warning['message']) #FIXME Grrrrr... does not work :(
                else:
                    write_ids.append(leave.id)
                    values['duration'] = on_change['value']['duration']
            else:
                write_ids.append(leave.id)
        return super(hrself_holidays_request_leave, self).write(cr, uid, write_ids, values, context=context)
    
    def onchange_employee_id(self, cr, uid, ids, employee_id, date_from, date_to, type_id, context):
        """Action to execute when the user changes employee_id."""
        return self._onchange_date_type(cr, uid, ids, employee_id, date_from, date_to, type_id, context)

    def onchange_date_from(self, cr, uid, ids, employee_id, date_from, date_to, type_id, context):
        """Action to execute when the user changes date_from."""
        return self._onchange_date_type(cr, uid, ids, employee_id, date_from, date_to, type_id, context)

    def onchange_date_to(self, cr, uid, ids, employee_id, date_to, date_from, type_id, context):
        """Action to execute when the user changes date_to."""
        return self._onchange_date_type(cr, uid, ids, employee_id, date_from, date_to, type_id, context)

    def onchange_type(self, cr, uid, ids, employee_id, date_from, date_to, type_id, context):
        """Action to execute when the user changes the type."""
        return self._onchange_date_type(cr, uid, ids, employee_id, date_from, date_to, type_id, context)

    def _disable_checks(self, cr, uid, type_id, activity_context_id):
        """Disable checks?"""
        result = False
        if type_id and activity_context_id:
            precision_id = self.pool.get('hrself.holidays.type').browse(cr, uid, type_id).precision_id_for_context(activity_context_id)
            if precision_id:
                prestation_code = self.pool.get('hrself.holidays.type.precision').browse(cr, uid, precision_id).prestation_code_id
                if prestation_code and prestation_code.gross_leave:
                    result = True
        return result

    def onchange_duration(self, cr, uid, ids, employee_id, date_from, date_to, type_id, duration, unit, context=None):
        """Action to execute when the user changes the duration."""
        result = {}
        if employee_id and date_from and date_to:
            services = self.pool.get('hrself.services').hrself_holidays_service(cr, uid, context=context)
            activities = services.get_theoretical_activities(cr, uid, employee_id, date_from, date_to, context=context)
            if not self._disable_checks(cr, uid, type_id, activities and activities[0] and activities[0]['activity_context']):
                allowed_duration = self._duration(cr, uid, type_id, activities)['value']['duration']
                if unit == hrself.TIME_UNIT_HOUR and date_from == date_to:
                    if duration > allowed_duration:
                        result['warning'] = {
                            'title': _('Warning'), 
                            'message': _('The leave request duration is greater than allowed. \n You have %d %s(s) left.') % 
                            (allowed_duration, hrself.TIME_UNITS_DICT[unit])
                        }
                        result['value'] = {'duration': allowed_duration}
                elif duration <> allowed_duration:
                    result['warning'] = {'title': _('Warning'), 'message': _('You cannot change the duration.')}
                    result['value'] = {'duration': allowed_duration}
        return result

    def onchange_unit(self, cr, uid, ids, employee_id, date_from, date_to, type_id, unit, context=None):
        """Action to execute when the user changes the unit."""
        result = {}
        services = self.pool.get('hrself.services').hrself_holidays_service(cr, uid, context=context)
        activities = services.get_theoretical_activities(cr, uid, employee_id, date_from, date_to, context=context)
        allowed_unit = self._duration(cr, uid, type_id, activities)['value']['unit']
        if unit <> allowed_unit:
            result['warning'] = {'title': _('Warning'), 'message': _('You cannot change the unit.')}
            result['value'] = {'unit': allowed_unit}
        return result

    def _onchange_date_type(self, cr, uid, ids, employee_id, date_from, date_to, type_id, context):
        """Action to execute when the user changes a date (from or to) or the type."""
        result = {}
        activities = None
        if employee_id and date_from and date_to:
            if date_from > date_to:
                message = _('The start date (%s) must be smaller or equal to the end date (%s)!') % (date_from, date_to)
                result['warning'] = {'title': _('Invalid dates'), 'message': message}
            else:
                services = self.pool.get('hrself.services').hrself_holidays_service(cr, uid, context=context)
                activities = services.get_theoretical_activities(cr, uid, employee_id, date_from, date_to, context=context)
                if not self._disable_checks(cr, uid, type_id, activities and activities[0] and activities[0]['activity_context']):
                    presence_days = []
                    absence_days = []
                    for activity in activities:
                        date = hrself_datetime.to_string(hrself_datetime.to_date(activity['date']))
                        if activity['presence_absence'] == 'A' and date not in absence_days:
                            absence_days.append(date)
                        elif activity['presence_absence'] == 'P' and date not in presence_days:
                            presence_days.append(date)
                    if not presence_days:
                        message = _('You cannot take leave on absence days: ') + ', '.join(absence_days) 
                        result['warning'] = {'title': _('Invalid dates'), 'message': message}
        result.update(self._duration(cr, uid, type_id, activities))
        return result

    def _duration(self, cr, uid, type_id, activities):
        """Duration based on type and activities"""
        result = {}
        duration = 0
        unit = False
        if type_id and activities:
            days = []
            activity_context_id = activities[0]['activity_context']
            precision_id = self.pool.get('hrself.holidays.type').browse(cr, uid, type_id).precision_id_for_context(activity_context_id)
            if precision_id:
                unit = self.pool.get('hrself.holidays.type.precision').browse(cr, uid, precision_id).unit
                for activity in activities:
                    if activity['presence_absence'] == 'P':
                        date = hrself_datetime.to_date(activity['date'])
                        if (unit == hrself.TIME_UNIT_DAY and not date in days):
                            duration += 1
                            days.append(date)
                        elif unit == hrself.TIME_UNIT_HOUR:
                            duration += activity['duration'] / 60
            else:
                type = self.pool.get('hrself.holidays.type').browse (cr, uid, type_id)
                message = _('No precision defined for the holidays type named %s identified by %d') % (type.name , type.id)
                result['warning'] = {'title': _('No precision'), 'message': message}
        result['value'] = {'duration': duration, 'unit': unit}
        return result
    
    def _get_workflow_id(self, cr, uid, ids):
        """Get the workflow identifier."""

        workflow_association_object = self.pool.get('hrself.holidays.workflow')
        for leave_request in self.browse(cr, uid, ids):
            workflow_association_id = workflow_association_object.get_most_specific(cr, uid, leave_request.type_id.id, leave_request.activity_context_id.id)
            return workflow_association_object.browse(cr, uid, workflow_association_id).workflow_id.id

    def freeze(self, cr, uid, ids, context=None):
        """Freezes the leave request on confirmation.
        @overrides: L{hrself_holidays_request_leave}.
        """
        for leave_request in self.browse(cr, uid, ids):
            # TODO factorize this and use it also in _onchange_date_type
            services = self.pool.get('hrself.services').hrself_holidays_service(cr, uid, context=context)
            activities = services.get_theoretical_activities(cr, uid, leave_request.employee_id.id, leave_request.date_from, leave_request.date_to, context=context)
            if activities:
                leave_request.write({'activity_context_id': activities[0]['activity_context']})            

    def _check_balance(self, cr, uid, ids):
        """Check balance limit if override is not allowed.
           Returns the number of remaining days or hours.
           @rtype: int
        """

        for leave_request in self.browse(cr, uid, ids):
            if leave_request.activity_context_id:
                activity_context_id = leave_request.activity_context_id.id
                precision_id = leave_request.type_id.precision_id_for_context(activity_context_id)
                type_precision = self.pool.get('hrself.holidays.type.precision').browse(cr, uid, precision_id)
                if not type_precision.is_unlimited :
                    context = {
                        'employee_id': leave_request.employee_id.id, 
                        'date_from': leave_request.date_from, 
                        'activity_context': activity_context_id,
                    }
                    remaining_dic = leave_request.type_id.get_remaining_leaves_for_employee(context)
                    remaining = remaining_dic[leave_request.type_id.id]
                    if leave_request.duration > remaining:
                        return (remaining, hrself.TIME_UNITS_DICT[leave_request.unit])
            return None 

    def _check_priority(self, cr, uid, ids, context=None):
        """Check balance of priority leave type (if existing)"""
        type_precision_object = self.pool.get('hrself.holidays.type.precision')
        for leave_request in self.browse(cr, uid, ids, context=context):
            if leave_request.activity_context_id:
                activity_context_id = leave_request.activity_context_id.id
                precision_id = leave_request.type_id.precision_id_for_context(activity_context_id)
                type_precision = type_precision_object.browse(cr, uid, precision_id, context=context)
                type_id_object = self.pool.get('hrself.holidays.type')
                if type_precision.priority_type_precision_id:
                    priority_type = type_id_object.browse(cr, uid, type_precision.priority_type_precision_id.type_id.id, context=context)
                    employee = leave_request.employee_id.id
                    date_from = leave_request.date_from
                    remaining_dic = priority_type.get_remaining_leaves_for_employee({
                        'employee_id':employee, 
                        'date_from':date_from, 
                        'activity_context':activity_context_id
                    })
                    remaining = remaining_dic[priority_type.id]
                    if remaining > 0:
                        return priority_type.name
        return False

    def _overlap(self, cr, uid, ids):
        """Returns true if these requests overlap an existing validated or confirmed one for a specific employee.
        @rtype: boolean."""

        for leave in self.browse(cr, uid, ids):
            if not self._disable_checks(cr, uid, leave.type_id.id, leave.activity_context_id and leave.activity_context_id.id):
                if leave.unit == hrself.TIME_UNIT_DAY:
                    date_from = leave.date_from
                    date_to = leave.date_to
                    domain = ['&', '&', '&', ('employee_id', '=', leave.employee_id.id), \
                              ('unit', '=', hrself.TIME_UNIT_DAY), \
                              '|', ('state', '=', STATE_VALIDATED), ('state', '=', STATE_CONFIRMED), \
                              '|', '&', ('date_to', '>=', date_from), ('date_to', '<=', date_to), \
                              '&', ('date_from', '>=', date_from), ('date_from', '<=', date_to)]
                    count_overlap_ids = self.search(cr, uid, domain, count=True)
                    if count_overlap_ids > 0:
                        return True
        return False

    def _is_on_leave(self, cr, uid, employee_ids, date_from, date_to=None, context=None):
        """Returns True if the employee is on leave within the specified period.
        Returns false otherwise.
        @rtype: boolean."""

        date_from = hrself_datetime.to_string(date_from)
        if date_to is None:
            date_to = date_from
        else:
            date_to = hrself_datetime.to_string(date_to)
        for employee in self.pool.get('hr.employee').browse(cr, uid, employee_ids, context):
            domain = ['&', '&', ('employee_id', '=', employee.id), \
                      '|', ('state', '=', STATE_VALIDATED), ('state', '=', STATE_CONFIRMED), \
                      '|', '&', ('date_from', '<=', date_to), ('date_to', '>=', date_from), \
                      '&', ('date_from', '>=', date_from), ('date_to', '<=', date_to),
                     ] 
            #Use 1 instead of uid to call the method with the admin privileges (à la sudo)
            return self.search(cr, 1, domain, count=True) != 0

    def _create_workflow(self, cr, uid, oid, wkf_id):
        """Creates one instance of wkf_id for object oid."""
        
        ident = (uid, self._name, oid)
        workflow.instance.create(cr, ident, wkf_id)

    def _delete_workflow(self, cr, uid, oid):
        """Deletes all workflow instances linked to object oid."""
        
        ident = (uid, self._name, oid)
        workflow.instance.delete(cr, ident)

    def confirm(self, cr, uid, ids, context=None):
        """Action executed when a leave request is confirmed."""

        self.freeze(cr, uid, ids, context)
        
        if self._overlap(cr, uid, ids):
            raise osv.except_osv(_('Warning!'), _('One of the leave request you want to confirm is overlapping an existing one.'))

        remaining = self._check_balance(cr, uid, ids)
        if remaining :
            raise osv.except_osv(_('Warning!'), _('The leave request duration is greater than balance. \n You have %d %s(s) left.') % remaining)
        
        for leave_request in self.browse(cr, uid, ids, context=context):
            type_name = self._check_priority(cr, uid, ids, context)
            if type_name:
                raise osv.except_osv(_('Warning!'), _('Leave type "%s" has higher priority and a positive balance. \n "%s" must be depleted before "%s" ') % (type_name,type_name,leave_request.type_id.name ))

            # spawn appropriate validation workflow
            workflow_id = leave_request._get_workflow_id()
            self._create_workflow(cr, uid, leave_request.id, workflow_id)
            self.write(cr, uid, [leave_request.id], {'state': STATE_CONFIRMED})
        return True

    def _transfer_to_approver(self, cr, uid, ids):
        """doc (backup approver)."""

        today = datetime.date.today()
        for leave_request in self.browse(cr, uid, ids):
            if leave_request.activity_context_id:
                context = {'activity_context_id': leave_request.activity_context_id.id}
            else:
                context = {}
            leave_approver_id = leave_request.employee_id.get_leave_approver(today, context)[leave_request.employee_id.id]
            self.write(cr, uid, [leave_request.id], {'leave_approver_id': leave_approver_id})

    def _transfer_to_direct_approver(self, cr, uid, ids):
        """doc (backup approver)."""

        today = datetime.date.today()
        for leave_request in self.browse(cr, uid, ids):
            leave_approver_id = leave_request.employee_id._get_assigned_approver(today)[leave_request.employee_id.id]
            self.write(cr, uid, [leave_request.id], {'leave_approver_id': leave_approver_id})

    def _transfer_to_department_approver(self, cr, uid, ids):
        """doc (backup approver)."""

        today = datetime.date.today()
        for leave_request in self.browse(cr, uid, ids):
            leave_approver_id = leave_request.employee_id._get_department_approver(today)[leave_request.employee_id.id]
            self.write(cr, uid, [leave_request.id], {'leave_approver_id': leave_approver_id})

    def _transfer_to_functional_administrator(self, cr, uid, ids):
        """doc (backup approver)."""

        for leave_request in self.browse(cr, uid, ids):
            leave_approver = leave_request.employee_id.universe_id.functional_administrator_id
            self.write(cr, uid, [leave_request.id], {'leave_approver_id': leave_approver.id})

    def validate(self, cr, uid, ids, context=None, state=STATE_VALIDATED):
        """Action executed when a leave request is validated"""

        leave_approver_ids = self.pool.get("hr.employee").search(cr, uid, [('user_id', '=', uid)])
        if not leave_approver_ids:
            raise osv.except_osv(_('Error!'), _('You do not have an associated employee. Please contact the administrator'))
        for leave_request in self.browse(cr, uid, ids):
            self.write(cr, uid, [leave_request.id], {'state': state})
            services = self.pool.get('hrself.services').hrself_holidays_service(cr, uid, context=context)
            if state == STATE_VALIDATED:
                services.register_event(cr, uid, leave_request.id, context=context)
        return True

    def partial_validate(self, cr, uid, ids):
        """Action executed when a leave request is partially validated."""

        # partial validation actually should not change object status since there could be any number of steps involved
        self.validate(cr, uid, ids, state=STATE_CONFIRMED)

    def refuse(self, cr, uid, ids, context=None):
        """Action executed when a leave request is refused"""

        return self.validate(cr, uid, ids, state=STATE_REFUSED)

    def cancel(self, cr, uid, ids, context=None):
        """Action executed when a leave request is cancelled"""
        self.write(cr, uid, ids, {'state': STATE_CANCELLED})
        return True

    def set_to_draft(self, cr, uid, ids, context=None):
        """Action executed when a leave request is reset to draft"""
        self.write(cr, uid, ids, {
            'state': STATE_DRAFT,
            'leave_approver_id': False,
        })
        wf_service = netsvc.LocalService("workflow")
        for oid in ids:
            self._delete_workflow(cr, uid, oid) # deletes all workflow instances for this object
            wf_service.trg_create(uid, self._name, oid, cr)
        return True

    def search_planned_leaves_by_employee_in_date_range(self, cr, uid, employee_id, date_from, date_to):
        sop = hrself_datetime.to_date(date_from)
        eop = hrself_datetime.to_date(date_to)
        domain = ['&', ('employee_id', '=', employee_id), \
                  '&', '|', ('state', '=', STATE_CONFIRMED), ('state', '=', STATE_VALIDATED), \
                  '|', '&', ('date_from', '<', eop), ('date_to', '>=', sop), \
                  '&', ('date_from', '>=', sop), ('date_to', '<', eop)]
        leave_ids = self.search(cr, uid, domain)
        return leave_ids

hrself_holidays_request_leave()

class hrself_holidays_theoretical_activity(osv.osv):
    """Proxy for theoretical activiy returned by service."""
    
    _name = 'hrself.holidays.theoretical.activity'
    _description = 'Theoretical activiy'
    # TODO: Theoretical activity shall not inherit from hrself.holidays.request: they might be close, but are not from same hierarchy
    _inherit = 'hrself.holidays.request'
    
    def _get_label_for_calendar(self, cr, uid, ids, field_name, arg, context=None):
        """Labels for display in calendar view
        @return: A calendar label for each theoretical activity.
        """
        labels = dict.fromkeys(ids, 'NOLABEL')
        for activity in self.browse(cr, uid, ids, context=context): 
            duration = activity.duration <> 0.0 and '(%sH)' % activity.duration or ''
            labels[activity.id] = '%s %s: %s' % (activity.name, duration, activity.employee_id.name)
        return labels

    _columns = {
        'calendar_label': fields.function(_get_label_for_calendar, method=True, type='string', string='Calendar label', help='Label for calendar view'),
        'duration': fields.float('Duration', help='Duration, in hour'),
    }

    def search(self, cr, uid, domain, offset=0, limit=None, order=None, context=None, count=False):
        """Searches for proxied objects and creates entries for new ones."""

        # invalidates expired entries
        oids = super(hrself_holidays_theoretical_activity, self).search(cr, uid, domain)
        if oids:
            super(hrself_holidays_theoretical_activity, self).unlink(cr, uid, oids, context)

        results = []
        employee_id = self.pool.get('hr.employee').search(cr, uid, [('user_id', '=', uid)])[0]
        date_from = date_to = hrself_datetime.to_string(datetime.date.today()) # default date bounds
        for element in domain:
            if element[0] == 'date_from' and element[1] in ('>', '>=', '='):
                date_from = element[2]
                if element[1] == '>': # services does not support strict inequality
                    date_from = hrself_datetime.to_string(hrself_datetime.to_date(date_from) + datetime.timedelta(1))
            if element[0] == 'date_to' and element[1] in ('<', '<=', '='):
                date_to = element[2]
                if element[1] == '<': # services does not support strict inequality
                    date_to = hrself_datetime.to_string(hrself_datetime.to_date(date_to) - datetime.timedelta(1))
        services = self.pool.get('hrself.services').hrself_holidays_service(cr, uid, context=context)
        records = services.get_theoretical_activities(cr, uid, employee_id, date_from, date_to, False, context=context)
        for record in records:
            values = {
                'name': record['presence_absence'],
                'date_from': hrself_datetime.to_date(record['date']),
                'date_to': hrself_datetime.to_date(record['date']),
                'employee_id': employee_id,
                'duration': record['duration'] / 60,
            }
            if values['name'] <> 'P' or (values['name'] == 'P' and values['duration'] <> 0.0):
                oid = super(hrself_holidays_theoretical_activity, self).create(cr, uid, values, context)
            results.append(oid)

        return results

    def search_planned_activities_by_employee_in_date_range(self, cr, uid, employee_id, date_from, date_to):
        sop = hrself_datetime.to_date(date_from)
        eop = hrself_datetime.to_date(date_to)
        domain = ['&', ('employee_id', '=', employee_id), \
                  '&', ('date_from', '>=', sop), ('date_to', '<', eop)]
        activity_ids = self.search(cr, uid, domain)
        return activity_ids

hrself_holidays_theoretical_activity()

class hrself_holidays_workflow(osv.osv):
    
    _name = 'hrself.holidays.workflow'
    
    UNIQUE_FIELDS_NAME = ['activity_context_id', 'leave_type_id', 'start_date']

    _columns = {
        'activity_context_id': fields.many2one('hrself.activity.context', 'Activity Context'),
        'leave_type_id': fields.many2one('hrself.holidays.type', 'Leave Type', select=True),
        'workflow_id': fields.many2one('workflow', 'Request Leave Approvement Workflow', required=True),
        'start_date': fields.date('Start date', required=True, select=True),
        'end_date': fields.date('End date'),
    }
    
    _sql_constraints = [
        ('uniqueness', 'unique(%s)' % ', '.join(UNIQUE_FIELDS_NAME), 'Workflow definition must be unique.')
    ]
    
    def _is_null(self, record, field):
        """Test if a record field is null in the database.
        This should definitely be implemented in fields.py or orm.py .
        Might not be completely accurate (does not check function, *2many,...)."""
        
        if self._columns[field]._type == 'many2one':
            return isinstance(record[field], osv.orm.browse_null)
        elif self._columns[field]._type == 'boolean':
            return record[field] is None
        else:
            return record[field] == False
        
    def _build_domain(self, record, fields):
        domain = []
        for field in fields:
            if self._columns[field]._type == 'many2one':
                value = record[field].id
            else:
                value = record[field]
            domain.append((field, '=', value))
        return domain
    
    def _check_unique_without_null(self, cr, uid, ids):
        """This method is used to complement the SQL UNIQUE constraint:
        according to the SQL standard, in the presence of a unique constraint it
        is possible to store duplicate rows that contain a null value in at
        least one of the constrained columns."""

        unique_fields = set(self.UNIQUE_FIELDS_NAME)
        for record in self.browse(cr, uid, ids):
            null_fields = set([field for field in unique_fields if self._is_null(record, field)])
            if len(null_fields) > 0:
                domain = self._build_domain(record, unique_fields)
                if self.search(cr, uid, domain, count=True) > 1:
                    return False
        return True

    # TODO: check non overlap start_date end_date

    def _check_associated_precision(self, cr, uid, ids, context=None):
        """Checks that there is a precision linked to this leave_type_id and to 
        an activity context compatible with this activity_context_id."""
        
        for wkf_assoc in self.browse(cr, uid, ids, context=context):
            leave_type_id = wkf_assoc.leave_type_id
            activity_context_id = wkf_assoc.activity_context_id
            if leave_type_id and activity_context_id:
                domain = [('type_id', '=', leave_type_id.id)]
                precision_ids = self.pool.get('hrself.holidays.type.precision').search(cr, uid, domain)
                for precision in self.pool.get('hrself.holidays.type.precision').browse(cr, uid, precision_ids):
                    if precision.activity_context_id.is_compatible(wkf_assoc.activity_context_id.id):
                        break
                else: # else of for loop, reached if the for loop didn't break
                    return False
        return True
    
    _constraints = [
        (_check_unique_without_null, _('Workflow definition must be unique.'), UNIQUE_FIELDS_NAME),
        (_check_associated_precision, _('There is no precision associated to both the specified leave type and to an activity context compatible with the specified one.'), ['leave_type_id', 'activity_context_id']),
    ]
    
    def _get_most_specific(self, cr, uid, ids, activity_context_id):
        """Get the most specific association between ids."""
        best_association = None
        for association in self.browse(cr, uid, ids):
            if not best_association:
                best_association = association
            else:
                if association.leave_type_id and not best_association.leave_type_id:
                    best_association = association
                elif best_association.activity_context_id and (association.activity_context_id and association.activity_context_id.is_compatible(activity_context_id) and association.activity_context_id.order > best_association.activity_context_id.order):
                    best_association = association
        if best_association:
            return best_association.id
    
    def get_most_specific(self, cr, uid, leave_type_id, activity_context_id):
        """Returns the most specific matching workflow association; most specific is worked out in this order:
        on the strict date overlap, equality leave_type value and maximum activity_context order."""

        today = hrself_datetime.to_string(datetime.datetime.now().date())
        date_domain = [('start_date', '<=', today), '|', ('end_date', '>=', today), ('end_date', '=', None),]
        type_domain = [('leave_type_id', '=', leave_type_id),]

        ids = self.search(cr, uid, date_domain)
        if ids:
            ids = ids[:1]
        ids += self.search(cr, uid, type_domain + date_domain)
        id = self._get_most_specific(cr, uid, ids, activity_context_id)
        if not id:
            raise NoDefaultWorkflow(_("There is no global validation workflow defined."))
        else:
            return id

hrself_holidays_workflow()

class res_users(osv.osv):

    def __init__(self, pool, cr):
        """Redefined to mark 'hrself_view_type' as a writable field."""
        res = super(res_users, self).__init__(pool, cr)
        self.SELF_WRITEABLE_FIELDS.append('hrself_view_type')
        self._uid_cache.clear()
        return res

    _inherit = "res.users"

    def _get_parent_of_department_if_approver(self, cr, uid, ids, field_name, args, context=None):
        """Returns the parent department if the user is the approver of the department."""
        res = dict.fromkeys(ids, False)
        for user in self.browse(cr, uid, ids, context):
            employee = user.employee_id
            if employee:
                department = employee.department_id
                if department:
                    department_approver = department.leave_approver_id
                    if department_approver and department_approver.id == employee.id:
                        parent_department = department.parent_id
                        if parent_department:
                            res[user.id] = parent_department.id
        return res
    
    def _set_profile(self, cr, uid, ids, profile, context=None):
        """Redefined to change default home action of approvers and admin """

        default_admin_or_approver_view = 'open_holidays_request_leave_tree'
        view_module = 'hrself_holidays'

        if not context:
            context = {}
        ret= super(res_users, self)._set_profile(cr, uid, ids, profile, context)
        
        for user in self.browse(cr, uid, ids, context=context):
            if profile == self.PROFILE_APPROVER or profile == self.PROFILE_ADMIN:
                model_data_object = self.pool.get('ir.model.data')
                model_data_id = model_data_object._get_id(cr, uid, view_module, default_admin_or_approver_view)
                action_id = model_data_object.browse(cr, uid, model_data_id).res_id
                self.pool.get('res.users').write(cr, uid, user.id, {'action_id': action_id})

    def _get_available_view_types(self, cr, uid, context=None):
        """Define view types the user can have"""

        result = [('user','User')]
        user_object = self.pool.get('res.users')
        user = user_object.browse(cr, uid, uid, context=context)
        model_data_object = self.pool.get('ir.model.data')
        model_data_ids = model_data_object.search(cr, uid, [('module', '=', 'hrself'), ('name', '=', 'group_hr_approver')])
        group_ids = [group_id.id for group_id in user.groups_id]
        # If employee is approver
        if model_data_object.browse(cr, uid, model_data_ids[0]).res_id in group_ids:
                 result.append(('approver','Approver'))
        model_data_ids = model_data_object.search(cr, uid, [('module', '=', 'base'), ('name', '=', 'group_hr_manager')])
        # If employee is manager
        if model_data_object.browse(cr, uid, model_data_ids[0]).res_id in group_ids:
                 result.append(('administrator','Administrator'))
        return result


    _columns = {
        # This field is only used for record rule.
        'approver_department_parent_id': fields.function(_get_parent_of_department_if_approver, method=True, type='many2one', relation='hr.department', store=True),
        # User may want to have different views depending on his role/function
        'hrself_view_type': fields.selection(_get_available_view_types, 'View type for reports', required=True, readonly=0 )
    }
        
    _defaults = {
                'hrself_view_type': 'user',
    }

res_users()

class hrself_holidays_balance(osv.osv):
    """Represents holidays time balance for a given type and employee (similar to account balance)."""

    _name = 'hrself.holidays.balance'
    _description = 'Holidays time balance'

    _columns = {
        'employee_id': fields.many2one('hr.employee', 'Employee', required=True),
        'type_id': fields.many2one('hrself.holidays.type', 'Leave Type', required=True),
        'unit': fields.selection(hrself.TIME_UNITS, 'Time unit', required=True),
        'initial_value': fields.float('Initial value', required=True),
        'today_balance': fields.float('Balance (Today)', required=True),
        'end_year_balance': fields.float('Balance (End of the year)', required=True),
    }

hrself_holidays_balance()
