# -*- coding: utf-8 -*-
##############################################################################
#
#    OpenERP, Open Source Management Solution
#    Copyright (C) 2004-2010 Tiny SPRL (<http://tiny.be>).
#
#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU Affero General Public License as
#    published by the Free Software Foundation, either version 3 of the
#    License, or (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU Affero General Public License for more details.
#
#    You should have received a copy of the GNU Affero General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
##############################################################################

import datetime
import urllib
from hrself import hrself_datetime
from osv import fields, osv
import tools

class hr_employee(osv.osv):
    """Represents an employee in a human resource context."""

    _inherit = 'hr.employee'

    def _get_current_leave_approver(self, cr, uid, ids, name, args, context=None):
        """Returns the current leave approver (the one that is defined for today) for an employee.
        @return: The current leave approver for each employee.
        @rtype: A dictionary where keys are employee ids and values are approver ids
        """
        res = dict.fromkeys(ids, False)
        for employee in self.browse(cr, uid, ids, context=context):
            approvement_object = self.pool.get('hrself.employee.approvement')
            approvement_ids = approvement_object.search(cr, uid, [('employee_id', '=', employee.id)])
            for approvement in approvement_object.browse(cr, uid, approvement_ids):
                if datetime.date.today() in hrself_datetime.period(approvement.start_date, approvement.end_date):
                    res[employee.id] = approvement.approver_id.id
        return res

    def _get_approver_from_definition(self, cr, uid, ids, context=None):
        """Get the leave approver from the approvement definition."""
        res = []
        for approvement in self.pool.get('hrself.employee.approvement').browse(cr, uid, ids, context=context):
            res.append(approvement.employee_id.id)
        return res
    
    def _get_backup_from_definition(self, cr, uid, ids, context=None):
        """Get the leave backup from the approvement definition."""
        res = []
        for approvement in self.pool.get('hrself.backup.approvement').browse(cr, uid, ids, context=context):
            res.append(approvement.approver_id.id)
        return res
    
    def _get_current_leave_backup(self, cr, uid, ids, name, args, context=None):
        """Returns the current leave backup (the one that is defined for today) for an approver.
        @return: The current leave backup for each approver.
        @rtype: A dictionary of tuple where the tuple is 
        (approver id, approver backup id)
        (L{hrself_employee}, L{hrself_employee})
        """
        res = dict.fromkeys(ids, False)
        for approver in self.browse(cr, uid, ids, context=context):
            approvement_object = self.pool.get('hrself.backup.approvement')
            approvement_ids = approvement_object.search(cr, uid, [('approver_id', '=', approver.id)])
            for approvement in approvement_object.browse(cr, uid, approvement_ids):
                if datetime.date.today() in hrself_datetime.period(approvement.start_date, approvement.end_date):
                    res[approver.id] = approvement.backup_approver_id.id
        return res
    
    def _get_actual_leave_approver(self, cr, uid, ids, name, args, context=None):
        """Returns the actual leave approver, i.e. (s)he is not absent or on leave and (s)he can approve right now."""

        today = datetime.date.today()
        return self.get_leave_approver(cr, uid, ids, today, context)

    def _get_openerp_server_url(self, cr, uid, ids, field_name, arg, context=None):
        """Returns url of openerp server.
        """
        result = dict.fromkeys(ids)
        if 'openerp_server_url' in tools.config.options:
            base_url = tools.config['openerp_server_url']
            for employee in self.browse(cr, uid, ids):
                login = employee.user_id.login
                universe = employee.universe_id.name
                url = base_url + '?universe=' + urllib.quote(universe.encode('utf8')) + '&user=' + urllib.quote(login.encode('utf8'))
                result[employee.id] = url
        return result

    _columns = {
        'leave_approver_id': fields.function(
                                _get_current_leave_approver,
                                method=True,
                                type='many2one',
                                relation='hr.employee',
                                string='Current leave approver',
                                store={ 'hrself.employee.approvement': (_get_approver_from_definition, ['employee_id', 'approver_id', 'start_date', 'end_date'], 10) },
                            ),
        # backup is defined for an employee that is himself an approver
        'leave_backup_id': fields.function(
                                _get_current_leave_backup,
                                method=True,
                                type='many2one',
                                relation='hr.employee',
                                string='Current leave backup',
                                store={ 'hrself.backup.approvement': (_get_backup_from_definition, ['backup_approver_id', 'approver_id', 'start_date', 'end_date'], 10) },
                            ),
        # actual leave approver provided is not absent or on leave
        'actual_leave_approver_id': fields.function(_get_actual_leave_approver, method=True, type='many2one', relation='hr.employee', string='Actual leave approver'),
        'workflow_id': fields.many2one('workflow', 'Request Leave Approvement Workflow'),
        'openerp_server_url': fields.function(_get_openerp_server_url, method=True, type='str'),
    }

    def _is_absent_for_leave_approvement(self, cr, uid, ids, date_from, date_to=None, context=None):
        """Returns True if the employee is set as absent for leave approvement or on leave during the specified period, False otherwise.
        @rtype: boolean."""

        is_absent = self.pool.get('hrself.holidays.request.absence')._is_absent_for_leave_approvement(cr, uid, ids, date_from, date_to, context) 
        is_on_leave = self.pool.get('hrself.holidays.request.leave')._is_on_leave(cr, uid, ids, date_from, date_to, context)
        return is_absent or is_on_leave

    def _get_assigned_approver(self, cr, uid, ids, date, context=None):
        """Returns the direct assigned leave approver for a specific day or None if none found."""
        result = dict.fromkeys(ids)
        for employee in self.browse(cr, uid, ids, context):
            if employee.leave_approver_id:
                if self._is_absent_for_leave_approvement(cr, uid, [employee.leave_approver_id.id], date, date, context):
                    if employee.leave_approver_id.leave_backup_id:
                        # if approver is absent, approver backup cannot be absent
                        result[employee.id] = employee.leave_approver_id.leave_backup_id.id
                    else:
                        result[employee.id] = employee.universe_id.functional_administrator_id.id
                else:
                    result[employee.id] = employee.leave_approver_id.id
        return result

    def _get_direct_department_approver(self, cr, uid, ids, date, context=None):
        """Returns the direct department leave approver for a specific day or None if none found."""
        result = dict.fromkeys(ids)
        for employee in self.browse(cr, uid, ids, context=context):
            department = employee.department_id
            if department:
                result[employee.id] = department.get_leave_approver(date, context=context)[department.id]
        return result

    def _get_department_approver(self, cr, uid, ids, date, context=None):
        """Returns the department leave approver for a specific day. If no direct department approver is found,
        then returns the approver of the department implicitly referenced by the level of the theoretical activity,
        or None if not found."""
        if context is None:
            context = {}
        direct_approvers = self._get_direct_department_approver(cr, uid, ids, date, context)
        approvers = dict(direct_approvers)
        employee_ids = [employee_id for (employee_id, approver_id) in direct_approvers.items() if approver_id is None]
        if len(employee_ids) > 0:
            level_ids = []
            universe_id = self.pool.get('hr.employee').browse(cr, uid, employee_ids[0]).universe_id.id
            department_osv = self.pool.get('hr.department')
            department_ids = department_osv.search(cr, uid, [('universe_id', '=', universe_id), ('level_id', '!=', None)])
            approver_ids = department_osv.get_leave_approver(cr, uid, department_ids, date, context=context)
            for department in department_osv.browse(cr, uid, department_ids, context=context):
                if approver_ids[department.id]:
                    level_ids.append(department.level_id.id)
            levels_members = self.pool.get('hrself.level').get_members(cr, uid, level_ids, context=context)
            for (level_id, member_ids) in levels_members.items():
                department_ids = department_osv.search(cr, uid, [('universe_id', '=', universe_id), ('level_id', '=', level_id)])
                for employee_id in employee_ids:
                    if employee_id in member_ids:
                        approvers[employee_id] = approver_ids[department_ids[0]]

        result = dict(approvers)
        for (employee_id, approver_id) in approvers.items():
            if approver_id == employee_id:
                result[employee_id] = None
        return result

    def get_leave_approver(self, cr, uid, ids, date, context=None):
        """Returns the direct leave approver for a specific day or None if none found."""
        result = {}
        if len(ids) > 0:
            assigned_approvers = self._get_assigned_approver(cr, uid, ids, date, context=context)
            employee_ids = [employee_id for (employee_id, approver_id) in assigned_approvers.items() if approver_id is None]
            department_approvers = self._get_department_approver(cr, uid, employee_ids, date, context=context)
            result = dict(assigned_approvers, **department_approvers)
            employee_ids = [employee_id for (employee_id, approver_id) in department_approvers.items() if approver_id is None]
            if len(employee_ids) > 0:
                functional_administrator_id = self.browse(cr, uid, ids[0], context=context).universe_id.functional_administrator_id.id
                admin_approvers = dict.fromkeys(employee_ids, functional_administrator_id)
                result.update(admin_approvers)
        return result

hr_employee()

