# -*- coding: utf-8 -*-
##############################################################################
#
#    OpenERP, Open Source Management Solution
#    Copyright (C) 2004-2010 Tiny SPRL (<http://tiny.be>).
#
#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU Affero General Public License as
#    published by the Free Software Foundation, either version 3 of the
#    License, or (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU Affero General Public License for more details.
#
#    You should have received a copy of the GNU Affero General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
##############################################################################

import datetime
from hrself import hrself_datetime
from osv import fields,osv

class hrself_department(osv.osv):
    """Represents a department in a human resource context.
    """

    _inherit = 'hr.department'
    
    def _get_approver_from_definition(self, cr, uid, ids, context=None):
        """Get the leave approver from the approvement definition."""
        res = {}
        for approvement in self.pool.get('hrself.department.approvement').browse(cr, uid, ids, context=context):
            res[approvement.department_id.id] = True
        return res.keys()
    
    def _get_current_leave_approver(self, cr, uid, ids, name, args, context=None):
        """Returns the current leave approver (the one that is defined for today) for a department.
        @return: The current leave approver for each department.
        @rtype: A dictionary {department.id: approver.id}.
        (L{hrself_department}, L{hrself_employee})
        """
        res = dict.fromkeys(ids, False)
        for department in self.browse(cr, uid, ids, context):
            approvement_object = self.pool.get('hrself.department.approvement')
            approvement_ids = approvement_object.search(cr, uid, [('department_id', '=', department.id)])
            for approvement in approvement_object.browse(cr, uid, approvement_ids):
                if datetime.date.today() in hrself_datetime.period(approvement.start_date, approvement.end_date):
                    res[department.id] = approvement.approver_id.id
        return res
    
    _columns = {
        'leave_approver_id': fields.function(_get_current_leave_approver, method=True, type='many2one', relation='hr.employee', string='Current leave approver', store = {'hrself.department.approvement': (_get_approver_from_definition, ['department_id', 'approver_id', 'start_date', 'end_date'], 10)}),
        'department_approvement_ids': fields.one2many('hrself.department.approvement', 'department_id', string='Department approvement'),
    }    
    
    def get_leave_approver(self, cr, uid, ids, date, context=None):
        """Returns the direct department leave approver for a specific day or None if none found."""
        result = dict.fromkeys(ids)
        for department in self.browse(cr, uid, ids, context=context):
            if department.leave_approver_id: # FIXME: this doen't take the 'date' param into account
                if self.pool.get('hr.employee')._is_absent_for_leave_approvement(cr, uid, [department.leave_approver_id.id], date, date, context):
                    if department.leave_approver_id.leave_backup_id:
                        # if approver is absent, approver backup cannot be absent
                        result[department.id] = department.leave_approver_id.leave_backup_id.id
                    else:
                        result[department.id] = department.universe_id.functional_administrator_id.id
                else:
                    result[department.id] = department.leave_approver_id.id
        return result

hrself_department()

# vim:expandtab:smartindent:tabstop=4:softtabstop=4:shiftwidth=4:
