import logging
import types

from osv import fields, osv
from tools.safe_eval import safe_eval

class CalendarProxy(osv.osv):
    """Delegates calendar related operation to/from objects.
    Allows to displays and interact with instances of several object types in the calendar.
    Those object type shall comply to this proxy's underlying interface.
    
    Note: CalendarProxy could be an osv_memory but osv_memory currently fails to implement domain parsing correctly.
    """
    _name = "hrself.holidays.calendar.proxy"
    _description = "Allows multiple object types in the calendar."
   
    logger = logging.getLogger(_name)

    # also, shouldn't res_model be many2one to ir_model (see why it is not in other object)
    _columns = {
        'name': fields.char('Description', size=64, required=True, select=1),
        'date_start': fields.datetime('Start Date', required=True, select=1),
        'date_stop': fields.datetime('End Date', required=True, select=1),
        'res_model': fields.char('Target model', size=64, required=True, select=1),
        'res_id': fields.integer('Target instance id', required=True, select=1),
        'version': fields.datetime('Write date of source record', select=1),
    }
    
    # As the object is not manipulated by user, the following constraints should always be met.
    _sql_constraints = [
        ('uniqueness', 'unique(res_model, res_id)', 'CalendarProxy objects must be unique, please contact the Administrator.'),
    ]

    # label is not a field of CalendarProxy, it is used to map an optional calendar_label function field in target objects
    _target_fields = ['label', 'name', 'date_start', 'date_stop']

    def _adapt_domain(self, domain, config):
        adapted_domain = domain[:]
        for i, element in enumerate(domain):
            if isinstance(element, types.TupleType):
                fieldname = element[0]
                if fieldname in config:
                    adapted_domain[i] = (config[fieldname], element[1], element[2])
        return adapted_domain

    def _adapt_fields(self, fields, fieldname_map):
        adapted_fields = [fieldname_map.get(field, field) for field in fields]
        return adapted_fields
    
    def search(self, cr, uid, domain, offset=0, limit=None, order=None, context=None, count=False):
        """Searches for proxied objects, invalidates expired ones and creates entries for new ones."""

        # to allow search on any field -> remove fields specific to target object from domain when calling super().search

        if any([element[0] in ('id', 'res_id', 'res_model') for element in domain if isinstance(element, types.TupleType)]):
            self.logger.warn("The domain spans at least one of ('id', 'res_id', 'res_model'), shunting to super(%s, self).search ." % self.__class__.__name__)
            return super(CalendarProxy, self).search(cr, uid, domain, offset, limit, order, context, count)
        
        results = []
        proxy_config = self.pool.get('hrself.holidays.calendar.proxy.config')
        config_ids = proxy_config.search(cr, uid, [], context=context)
        for config in proxy_config.browse(cr, uid, config_ids, context=context):
            model_name = config.res_model.model
            model = self.pool.get(model_name)
            mapping = {
                'date_start': config.date_start and config.date_start.name or 'date_start',
                'date_stop': config.date_stop and config.date_stop.name or 'date_stop',
                'label': config.label and config.label.name or 'label',
            }
            adapted_domain = self._adapt_domain(domain, mapping)
            if config.extended_domain:
                adapted_domain.extend(safe_eval(config.extended_domain))

            # invalidate expired target records
            res_ids = model.search(cr, uid, adapted_domain, offset, limit, order, context, count=False)
            expired_domain = domain + [('res_model', '=', model_name), '!', ('res_id', 'in', res_ids)]
            expired_ids = super(CalendarProxy, self).search(cr, uid, expired_domain, context=context)
            super(CalendarProxy, self).unlink(cr, uid, expired_ids, context)

            # create proxy records
            adapted_fields = self._adapt_fields(self._target_fields, mapping)
            records = model.read(cr, uid, res_ids, adapted_fields, context)
            perms = model.perm_read(cr, uid, res_ids, context, False)
            exist_domain = [('res_model', '=', model_name), ('res_id', 'in', [record['id'] for record in records])]
            oids = super(CalendarProxy, self).search(cr, uid, exist_domain)
            if oids:
                super(CalendarProxy, self).unlink(cr, uid, oids, context)
            for i, record in enumerate(records):
                values = {
                    'name': record.get(mapping['label'], record.get('name')) or 'NONAME',
                    'date_start': record[mapping['date_start']],
                    'date_stop': record[mapping['date_stop']],
                    'res_model': model_name,
                    'res_id': record['id'],
                    'version': perms[i]['write_date'],
                }
                oid = super(CalendarProxy, self).create(cr, uid, values, context)
                results.append(oid)

        if count:
            return len(results)
        return results

    def create(self, cr, uid, values, context=None):
        self.logger.warn("You may not directly create proxy instances.")

    def unlink(self, cr, uid, ids, context=None):
        self.logger.warn("You may not directly unlink proxy instances.")

    def write(self, cr, uid, ids, values, context=None):
        self.logger.warn("You may not directly write to proxy instances.")

CalendarProxy()

class CalendarProxyConfig(osv.osv):
    """Calendar proxy configuration."""

    _name = "hrself.holidays.calendar.proxy.config"
    _description = "Calendar proxy configuration."
   
    _columns = {
        'res_model': fields.many2one('ir.model', 'Target model', required=True),
        'extended_domain': fields.char('Extended domain', size=64, help='Domain (list of tuples) that will be appended to search the target model'),
        'date_start': fields.many2one('ir.model.fields', 'Date start', domain="[('model_id', '=', res_model)]", help='Field in target model corresponding to date_start'),
        'date_stop': fields.many2one('ir.model.fields', 'Date stop', domain="[('model_id', '=', res_model)]", help='Field in target model corresponding to date_stop'),
        'label': fields.many2one('ir.model.fields', 'Label', domain="[('model_id', '=', res_model)]", help='Field in target model corresponding to label'),
    }

CalendarProxyConfig()
